using GLib;
using Sqlite;
using Gtk;
using Hildon;

#if FREMANTLE
public class Tear.OptionsManager : Hildon.StackableWindow {
#else
public class Tear.OptionsManager : Hildon.Window {
#endif

    public HashTable<string,Tear.Option> cache;
	
    private ToolButton edit_button;
    private ToolButton add_button;
    private ToolButton remove_button;

#if FREMANTLE
	private PannableArea scrolled_window;
#else
	private ScrolledWindow scrolled_window;
#endif
	private TreeView tree_view;
	private ListStore model;
	
	private bool buffered_write;

	construct {
	    int nrows, ncols, result;
	    string[] table;
	    
	    this.buffered_write = false;
	    
		this.set_icon_from_file (Config.PIXMAPS_DIR + "/" + Config.PACKAGE + ".png"); 
        program.add_window (this);
        var toolbar = new Toolbar ();

		var prefix = Config.APP_DIR.concat("/", Config.APPNAME);
		this.add_button = new ToolButton( new Gtk.Image.from_file( prefix + "_add.png" ), "New Option" );
		this.edit_button = new ToolButton( new Gtk.Image.from_file( prefix + "_edit.png" ), "Edit Option" );
		this.remove_button = new ToolButton( new Gtk.Image.from_file( prefix + "_remove.png" ), "Remove Option" );
        
        toolbar.add (this.add_button);
        toolbar.add (this.edit_button);
        toolbar.add (new SeparatorToolItem ());
        toolbar.add (this.remove_button);
        
        this.add_button.sensitive = false;
        
        this.add_toolbar (toolbar);

		this.model = new ListStore( 5, typeof(string), typeof(string), typeof(string), typeof(string), typeof(string) );
		
#if FREMANTLE
		this.tree_view = (Gtk.TreeView) Hildon.gtk_tree_view_new_with_model ( UIMode.EDIT, (TreeModel) this.model );
		
//		Gtk.rc_parse_string("""
//		style "tear-treeview"
//		{
//			GtkTreeView::even-row-color = darker(@bg_color)
//		}
//		widget "*TreeView*" style "tear-treeview"
//        """);
//
//        this.tree_view.set_name("TreeView");
#else
		this.tree_view = new Gtk.TreeView.with_model( (TreeModel) this.model );
#endif
        this.tree_view.headers_visible = true;
        this.tree_view.headers_clickable = true;
        this.tree_view.rubber_banding = true;
        this.tree_view.reorderable = false;
        this.tree_view.row_activated += (s, p, c) => {
    		edit_option();
        };

		Gdk.Color color;
		Gtk.TreeViewColumn column;
		string[] headers = new string[5];
		headers[0] = "name";
		headers[1] = "Option";
		headers[2] = "Type";
		headers[3] = "Value";
		headers[4] = "enum";
		
	    for ( var col = 0; col <= 4; col++ ) {
            var cell_renderer = new Gtk.CellRendererText();
			cell_renderer.font = "Sans 18";
			cell_renderer.height = 60;

            column = new Gtk.TreeViewColumn.with_attributes( headers[col], cell_renderer, "text", col);
	        column.set_cell_data_func( cell_renderer, format_col );
			column.expand = true;
			column.resizable = true;
			if (col == 0 || col == 4)
				column.set_visible(false);
			
			if ( col == 3 )
				column.set_max_width( 300 );
	
	        var label = new Gtk.Label( headers[col] );
	        label.modify_font(Pango.FontDescription.from_string("Sans Bold 20"));
	        label.justify = Gtk.Justification.CENTER;

	        column.set_widget(label);
	        label.show();

	        this.tree_view.append_column( column );
	    }

        this.model.sort_column_id = 1;
		this.delete_event += hide_on_delete;

	    if (DEBUG) {
		    var selection = this.tree_view.get_selection ();
	    
		    selection.changed += (s) => {
				Gtk.TreeModel model;
				Gtk.TreeIter iter;
	
			    s.get_selected ( out model, out iter );
			    
			    GLib.Value option;
			    model.get_value( iter, 0, out option );
			    warning( "selected "+option.get_string()+"\n" );
		    };
		}

#if FREMANTLE
        scrolled_window = new PannableArea.full ( PannableAreaMode.ACCEL, true, 0, 200, 0.8, 20 );
        scrolled_window.set( "mov_mode", MovementMode.VERT );
        scrolled_window.initial_hint = true;
        scrolled_window.hscrollbar_policy = PolicyType.NEVER;
        scrolled_window.low_friction_mode = false;
        scrolled_window.add_with_viewport (this.tree_view);
#else
        scrolled_window = new ScrolledWindow (null, null);
        scrolled_window.set_policy (PolicyType.AUTOMATIC, PolicyType.AUTOMATIC);
        scrolled_window.add (this.tree_view);
#endif        
	        
        var vbox = new VBox (false, 0);
        vbox.pack_start (scrolled_window, true, true, 0);
        this.add (vbox);

        this.cache = new HashTable<string,Tear.Option>(str_hash, str_equal);
        
        table = dbqueue.pull ("SELECT `option`, `value`, `public`, `default`, `title`, `enum`, `type` FROM `options`;", out nrows, out ncols, out result );

        if (result == Sqlite.OK) { 
	        if (table != null) {
		        if (nrows > 0) {
					for (int i = 1; i <= nrows; i++) {
						var new_option = new Tear.Option();
						
						new_option.name = table[i*7];
						new_option.value = table[i*7+1];
						new_option.is_public = preg_test( table[i*7+2], "true" );
						new_option.default_value = table[i*7+3];
						new_option.title = table[i*7+4];
						new_option.enum_values = table[i*7+5];
						new_option.type = table[i*7+6];
						
						if (new_option.is_public) {
							Gtk.TreeIter iter;
							this.model.insert_with_values( out iter, 999999, 0, new_option.name, 1, new_option.title, 2, new_option.type, 3, new_option.value, 4, new_option.enum_values );
							new_option.iterator = iter;
						}
	
						this.cache.insert( new_option.name, new_option );
				    }
				    if (DEBUG)
						warning ("populated options cache");
				}
			}
		}

        this.add_button.clicked += this.add_option;
        this.edit_button.clicked += this.edit_option;
        this.remove_button.clicked += this.delete_option;

        this.set_title( "Options Manager" );
	}
	
	public static void format_col ( Gtk.CellLayout cell_layout, Gtk.CellRenderer cell, Gtk.TreeModel treemodel, Gtk.TreeIter iter) {
		var model = (ListStore) treemodel;
		
	    Gtk.TreePath path = model.get_path( iter );
	    var row_num = path.get_indices()[0];
	    
	    var row_color = ( row_num % 2 == 1 ) ? "@bg_color" : "darker(@bg_color)";
	    cell.cell_background = row_color;
	}
	
	public void add_option() {
		
	}
	
#if FREMANTLE
	public TouchSelector create_selector( string[] items, int active, TouchSelectorSelectionMode selection_mode ) {
		var selector = new TouchSelector.text();
		selector.set_column_selection_mode ( selection_mode );
		int index = 0;
		
		foreach ( string item in items ) {
			selector.append_text( item );
			index++;
		}
		
		if (active >= 0)
			selector.set_active( 0, active );
		else {
			TreeIter iter;
			var model = selector.get_model(0);
			if (model.get_iter_first(out iter)) {
				do {
					selector.select_iter( 0, iter, false );
				} while (model.iter_next( ref iter ));
			}
		}

		return selector;
	}
	
	public TouchSelectorEntry create_selector_entry( string[] items, string text ) {
		var selector = new TouchSelectorEntry.text();
		
		foreach ( string item in items ) {
			selector.append_text( item );
		}
		
		selector.get_entry().set_text( text );
		
		return selector;
	}
#endif

	public void edit_option() {
		Gtk.TreeModel list;
		Gtk.TreeIter iter;
		GLib.Value option;
		GLib.Value type;
		GLib.Value val;
		GLib.Value title;
		GLib.Value enum_values;
		
		var selected = tree_view.get_selection().get_selected(out list, out iter);
		
		if (selected) {
			list.get_value( iter, 0, out option );
			list.get_value( iter, 1, out title );
			list.get_value( iter, 2, out type );
			list.get_value( iter, 3, out val );
			list.get_value( iter, 4, out enum_values );

			var is_true = preg_test( val.get_string(), "true" );
			var is_false = preg_test( val.get_string(), "false" );
			
#if FREMANTLE
			var message = new PickerDialog ( this );
			message.set_done_label ("OK");
			message.set_title ( title.get_string() );

			string[] choices = enum_values.get_string().split( "|" );
			var bool_selector = create_selector( { "true", "false" }, is_true ? 0 : 1, TouchSelectorSelectionMode.SINGLE );
			var entry_selector = create_selector_entry( choices, val.get_string() );
			entry_selector.center_on_selected();
			
			if ( is_true || is_false )
				message.set_selector( bool_selector );
			else
				message.set_selector( entry_selector );
#else
			var message = new MessageDialog.with_markup(this, DialogFlags.MODAL, MessageType.QUESTION, ButtonsType.OK_CANCEL, 
												   "Enter value for option <b>"+option.get_string()+"</b> of type <b>"+type.get_string()+"</b>:");
												   
			var true_button = new Gtk.RadioButton.with_label( null, "true" );
			var false_button = new Gtk.RadioButton.with_label_from_widget( true_button, "false" );
			
			var entry = new Gtk.Entry();

			if ( is_true || is_false ) {
				message.vbox.add(true_button);
				message.vbox.add(false_button);
				
				true_button.set_mode( false );
				false_button.set_mode( false );
				
				if (is_true)
					true_button.active = true;
				else
					false_button.active = true;
					
				true_button.show();
				false_button.show();
			} else {
				entry.text = val.get_string();
				entry.show();
				message.vbox.add(entry);
			}
#endif
			
			var result = message.run();
			message.hide();
			
			if (result == Gtk.ResponseType.OK && (is_true || is_false)) {
#if FREMANTLE
				this.update_option( option.get_string(), bool_selector.get_current_text() );
#else
				this.update_option( option.get_string(), true_button.active.to_string() );
#endif
				refresh_all_settings();
			} else 
				if  (result == Gtk.ResponseType.OK) {
#if FREMANTLE
					this.update_option( option.get_string(), entry_selector.get_entry().text );
#else
					this.update_option( option.get_string(), entry.text );
#endif
					refresh_all_settings();
				}
		}
	}

	public void delete_option() {
		Gtk.TreeModel list;
		Gtk.TreeIter iter;
		GLib.Value option;
		
		var selected = tree_view.get_selection().get_selected(out list, out iter);
		
		if (selected) {
			list.get_value( iter, 0, out option );
			
			var message = new MessageDialog.with_markup(this, DialogFlags.MODAL, MessageType.QUESTION, ButtonsType.OK_CANCEL, 
												   "Do you want to remove the <b>"+option.get_string()+"</b> option?");
			var result = message.run();
			message.hide();
			
			if (result == Gtk.ResponseType.OK)
				this.remove_option( option.get_string() );
		}
	}
	
	public void set_buffered () {
		this.buffered_write = true;
	}
	
	public void flush (bool force_refresh) {
		TimeVal time = TimeVal();
		
		if (DEBUG) {
			time.get_current_time();
		}
		
		dbqueue.push ("BEGIN TRANSACTION");
		this.cache.for_each ((k, v) => {
			int result;
			unowned Tear.Option val = (Tear.Option) v;
			if (force_refresh) {
        		result = dbqueue.db.exec ("UPDATE OR IGNORE `options` SET `title` = '"+ val.title +"', `enum` = '"+ val.enum_values +"', `type` = '"+ val.type +"' WHERE `option` = '"+ val.name +"';" );
		        if (result != Sqlite.OK) { 
		            return;
		    	}
			} else {
				if (val.is_modified) {
	        		result = dbqueue.db.exec ("UPDATE OR IGNORE `options` SET `value` = '"+ val.value +"', `public` = '"+ val.is_public.to_string() +"' WHERE `option` = '"+ val.name +"';" );
			        if (result != Sqlite.OK) { 
			            return;
			    	}
			    	val.is_modified = false;
		   		}
				if (val.is_new) {
			        result = dbqueue.db.exec ("INSERT OR IGNORE INTO `options` (`option`, `value`, `public`, `default`, `title`, `enum`, `type`) VALUES ('"+ val.name +"', '"+ val.value +"', '"+ val.is_public.to_string() +"', '"+ val.default_value +"', '"+ val.title +"', '"+ val.enum_values +"', '"+ val.type +"');" );
			        if (result != Sqlite.OK) { 
			            return;
			    	}
			    	val.is_new = false;
		    	}
	    	}
		});
		dbqueue.push ("COMMIT TRANSACTION");

		if (DEBUG) {
			var newtime = TimeVal();
			newtime.get_current_time();
			warning ( "Options flushed for " + (newtime.tv_sec - time.tv_sec).to_string() + " seconds." );
		}
		
		this.buffered_write = false;
	}
	
    public string? get_option (string? option) {
		if (option == null)
			return null;

        var opt = this.cache.lookup(option);

        if (opt == null) { 
            return null;
        }
        
        return opt.value;
    }

    public bool? get_bool (string? option) {
		if (option == null)
			return null;

        var opt = this.cache.lookup(option);

        if (opt == null) { 
            return null;
        }
        
        return preg_test( opt.value, "true" );
    }
    
    public string set_or_get ( string option, string value, bool pub = true, string title = "", string type = "string", string enum_values = "" ) {
      	string? ali = this.get_option ( option );
		if (ali == null) {
			this.update_option ( option, value, pub, title, type, enum_values );
			return value;
		} else
			return ali;
    }

    public bool set_or_get_bool ( string option, bool value, bool pub = true, string title = "" ) {
      	string? option_string = this.get_option ( option );
		if (option_string == null) {
			this.update_option ( option, value.to_string(), pub, title, "bool" );
			return value;
		} else
			return preg_test( option_string, "true" );
    }

    public bool update_option (string? option, string val, bool pub = true, string title = "", string type = "string", string enum_values = "" ) {
	    int result;
	    
		if (option == null)
			return false;

        unowned Tear.Option opt = this.cache.lookup(option);

        if (opt != null) { 
	        if (!this.buffered_write) {
	        	result = dbqueue.db.exec ("UPDATE OR IGNORE `options` SET `value` = '"+ val +"', `public` = '"+ pub.to_string() +"' WHERE `option` = '"+ option +"';" );
		        if (result != Sqlite.OK) { 
		            return false;
		        }
	    	}
	        
	        opt.value = val;
	        opt.is_public = pub;
			opt.title = title;
			opt.type = type;
			opt.enum_values = enum_values;
	        if (this.buffered_write)
				opt.is_modified = true;
	        
			this.model.set( opt.iterator, 3, val );
			
    	} else {
	        if (!this.buffered_write) {
		        result = dbqueue.db.exec ("INSERT OR IGNORE INTO `options` (`option`, `value`, `public`, `default`, `title`, `enum`, `type`) VALUES ('"+ option +"', '"+ val +"', '"+ pub.to_string() +"', '"+ val +"', '"+ title +"', '"+ enum_values +"', '"+ type +"');" );
		        if (result != Sqlite.OK) { 
		            return false;
		        }
	    	}
	        
			var new_option = new Tear.Option();
			
			new_option.name = option;
			new_option.value = val;
			new_option.is_public = pub;
			new_option.default_value = val;
			new_option.title = title;
			new_option.type = type;
			new_option.enum_values = enum_values;
	        if (this.buffered_write)
				new_option.is_new = true;
			
			if (new_option.is_public) {
				Gtk.TreeIter iter;
				this.model.insert_with_values( out iter, 999999, 0, new_option.name, 1, new_option.title, 2, new_option.type, 3, new_option.value, 4, new_option.enum_values );
				new_option.iterator = iter;
			}

			this.cache.insert( new_option.name, new_option );
    	}
        
        return true;
    }

    public bool remove_option (string option) {
	    int result;

		if (DEBUG)
			warning ( "remove option " + option );
	
		if (option == null)
			return false;

    	result = dbqueue.db.exec ("delete from `options` WHERE `option` = '"+ option +"';" );
        if (result != Sqlite.OK) { 
            return false;
        }

		Gtk.TreeIter iter;
	    iter = this.cache.lookup( option ).iterator;
	    
		this.model.remove( iter );
		this.cache.remove( option );
		
        return true;
    }
	
}
