using GLib;
using Gtk;
using Hildon;

#if FREMANTLE
public class Tear.DownloadManager : Hildon.StackableWindow {
#else
public class Tear.DownloadManager : Hildon.Window {
#endif

    private ToolButton open_button;
    private ToolButton add_button;
    private ToolButton remove_button;
    private HashTable<string,Tear.DownloadItem> downloads;

	private TreeView tree_view;
	private ListStore model;

	construct {
		this.set_icon_from_file (Config.PIXMAPS_DIR + "/" + Config.PACKAGE + ".png"); 
        program.add_window (this);
        var toolbar = new Toolbar ();

		var prefix = Config.APP_DIR.concat("/", Config.APPNAME);
		this.add_button = new ToolButton( new Gtk.Image.from_file( prefix + "_add.png" ), "Add Download" );
		this.open_button = new ToolButton( new Gtk.Image.from_file( prefix + "_go.png" ), "Open File" );
		this.remove_button = new ToolButton( new Gtk.Image.from_file( prefix + "_remove.png" ), "Remove Download" );
        
        toolbar.add (this.add_button);
        toolbar.add (this.remove_button);
        toolbar.add (new SeparatorToolItem ());
        toolbar.add (this.open_button);
        
        this.add_toolbar (toolbar);

		this.model = new ListStore( 3, typeof(string), typeof(int), typeof(string) );
#if FREMANTLE
		this.tree_view = (Gtk.TreeView) Hildon.gtk_tree_view_new_with_model ( UIMode.EDIT, (TreeModel) this.model );
#else
		this.tree_view = new Gtk.TreeView.with_model( (TreeModel) this.model );
#endif
        this.tree_view.headers_visible = true;
        this.tree_view.headers_clickable = true;
        this.tree_view.rubber_banding = true;
        this.tree_view.reorderable = true;

		Gdk.Color color;
		Gtk.TreeViewColumn column;
		string[] headers = new string[3];
		headers[0] = "Filename";
		headers[1] = "Progress";
		headers[2] = "Size/Total Size";
		
	    for ( var col = 0; col <= 2; col++ ) {
	        if ( col == 1 ) {
	            var cell_renderer = new Gtk.CellRendererProgress();
#if !FREMANTLE
				cell_renderer.height = 40;
#else
				cell_renderer.height = 60;
#endif
	            column = new Gtk.TreeViewColumn.with_attributes( headers[col], cell_renderer, "value", col);
		        column.set_cell_data_func( cell_renderer, format_col );
				column.expand = true;
				column.resizable = true;
	        } else {
	            var cell_renderer = new Gtk.CellRendererText();
	            column = new Gtk.TreeViewColumn.with_attributes( headers[col], cell_renderer, "text", col);
		        column.set_cell_data_func( cell_renderer, format_col );
				cell_renderer.font = "Sans 18";

				column.expand = true;
				column.resizable = true;
		        this.model.sort_column_id = col;
	        }
	
	        var label = new Gtk.Label( headers[col] );
	        label.modify_font(Pango.FontDescription.from_string("Sans Bold 20"));
	        label.justify = Gtk.Justification.CENTER;

	        column.set_widget(label);
	        label.show();

	        this.tree_view.append_column( column );
	        
	    }

		this.delete_event += hide_on_delete;

	    if (DEBUG) {
		    var selection = this.tree_view.get_selection ();
	    
		    selection.changed += (s) => {
				Gtk.TreeModel model;
				Gtk.TreeIter iter;
	
			    s.get_selected ( out model, out iter );
			    
			    GLib.Value filename;
			    model.get_value( iter, 0, out filename );
			    warning( "selected "+filename.get_string()+"\n" );
		    };
		}

        var vbox = new VBox (false, 0);
        vbox.pack_start (tree_view, true, true, 0);
        this.add (vbox);

        this.downloads = new HashTable<string,Tear.DownloadItem>(str_hash, str_equal);

        this.add_button.clicked += add_download_uri;
        this.open_button.clicked += open_download;
        this.remove_button.clicked += remove_download;

        this.set_title( "Download Manager" );
	}
	
	public void add_download_uri () {
		
	}

	public void remove_download () {
		Gtk.TreeModel list;
		Gtk.TreeIter iter;
		GLib.Value filename;
		
		var selected = tree_view.get_selection().get_selected(out list, out iter);
		
		if (selected) {
			list.get_value( iter, 0, out filename );
			
			this.downloads.remove( Path.get_basename( filename.get_string() ) );
			((ListStore) list).remove( iter );
		}
	}

	public void open_download () {
		Gtk.TreeModel list;
		Gtk.TreeIter iter;
		GLib.Value filename;
		
		var selected = tree_view.get_selection().get_selected(out list, out iter);
		
		if (selected) {
			list.get_value( iter, 0, out filename );
			var name = Path.get_basename( filename.get_string() );
			var fname = this.downloads.lookup(name).filename;
			
			if (fname.has_suffix(".install")) 
				Hildon.mime_open_file_with_mime_type ( dbus_service.connection, this.downloads.lookup(name).filename, "application/x-install-instructions" );
			else
				Hildon.mime_open_file ( dbus_service.connection, fname );
		}
	}

	public void add_download_item ( string uri, string filename, string action, double total_size, string mime = "" ) {
		var name = Path.get_basename( filename );

		if (this.downloads.lookup(name) != null)
			Banner.show_information_with_markup( get_topmost_window(), null, "Download <b>" + name + "</b> exists, <b color='red'>aborting</b>..." );
			
		Gtk.TreeIter iter;
		this.model.insert_with_values( out iter, 999999, 0, name, 1, 0, 2, "0/"+total_size.to_string()+" Bytes" );

		var item = new Tear.DownloadItem();
		item.filename = filename;
		item.iterator = iter;
		item.action = action;
		item.uri = uri;
		item.mime_type = mime;
		
		this.downloads.insert( name, item ); 
	}

	public void add_temporary_download_item ( string uri, string mime = "" ) {
		var name = Path.get_basename( uri );

		if (this.downloads.lookup(name) != null)
			Banner.show_information_with_markup( get_topmost_window(), null, "Download <b>" + name + "</b> exists, <b color='red'>aborting</b>..." );
			
		Gtk.TreeIter iter;
		this.model.insert_with_values( out iter, 999999, 0, name, 1, 0, 2, "" );

		var item = new Tear.DownloadItem();
		item.filename = "/var/tmp/" + name;
		item.iterator = iter;
		item.action = "open";
		item.uri = uri;
		item.mime_type = mime;
		
		this.downloads.insert( name, item ); 
	}

	public void remove_download_item ( string name ) {

		Gtk.TreeIter iter;
	    iter = this.downloads.lookup( name ).iterator;
	    
		this.model.remove( iter );
		this.downloads.remove( name );
	}

	public static void format_col ( Gtk.CellLayout cell_layout, Gtk.CellRenderer cell, Gtk.TreeModel treemodel, Gtk.TreeIter iter) {
//		var model = (ListStore) treemodel;
		
//	    Gtk.TreePath path = model.get_path( iter );
//	    var row_num = path.get_indices()[0];
	    
//	    var row_color = ( row_num % 2 == 1 ) ? "#e9e9e9" : "#fff";
//	    cell.cell_background = row_color;
	}

	public void progress_update_cb (WebKit.WebDownload download, uint current_bytes, uint total_bytes) {
		Gtk.TreeIter iter;
	    iter = this.downloads.lookup( Path.get_basename( download.get_local_uri() ) ).iterator;
	    
	    long percent = Math.lround(((double) current_bytes / (double) total_bytes) * 100);
	    string size = (current_bytes / 1024).to_string() + "/" + (total_bytes / 1024).to_string() + " KBytes";

   	    this.model.set(iter, 1, percent, 2, size);
	}

	public void started_cb (WebKit.WebDownload download) {
        if (DEBUG)
        	warning ("download started");
		var name = Path.get_basename( download.get_local_uri() );

		Banner.show_information_with_markup( get_topmost_window(), null, "Downloading <b>" + name + "</b>.");
	}

	public void finished_cb (WebKit.WebDownload download) {
        if (DEBUG)
        	warning ("download finished");
        	
		var name = Path.get_basename( download.get_local_uri() );
		var download_item = this.downloads.lookup( name );
		
		if (download_item.action == "open") {
			var fname = download.get_local_uri().substring(7);
			
			if (download_item.mime_type != "")
				Hildon.mime_open_file_with_mime_type ( dbus_service.connection, download_item.filename, download_item.mime_type );
			else {
				if (fname.has_suffix(".install")) 
					Hildon.mime_open_file_with_mime_type ( dbus_service.connection, download_item.filename, "application/x-install-instructions" );
				else
					Hildon.mime_open_file ( dbus_service.connection, fname );
			}
				
			if (download_item.filename.has_prefix("/var/tmp/")) {
				FileUtils.remove(download_item.filename);
				this.remove_download_item(name);
			}
		} else
			Banner.show_information_with_markup( get_topmost_window(), null, "Download of <b>" + name + "</b> finished.");
	}

	public void error_cb (WebKit.WebDownload download, int p1, WebKit.WebDownloadError error, string errormsg) {
        if (DEBUG)
        	warning ("download error");

		var name = Path.get_basename( download.get_local_uri() );

		Banner.show_information_with_markup( get_topmost_window(), null, "Download of <b>" + name + "</b> failed." );
	}

}
