//      stopish.c
//
//      Copyright 2009 Michael Cronenworth <mike@cchtml.com>
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.

#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include <libosso.h>

#include "stopish.h"

static GtkWidget *timerLabel = NULL;
static GtkWidget *timerHistoryLabel1 = NULL;
static GtkWidget *timerHistoryLabel2 = NULL;
static GtkWidget *timerHistoryLabel3 = NULL;
static GtkWidget *timerHistoryLabel4 = NULL;
static GSList *historyList = NULL;
static int stopishMode = STOPISH_MODE_START;
static int timerHandle = -1;

//Prototypes
gint dbus_callback( const gchar *interface, const gchar *method,
	                GArray *arguments, gpointer data, osso_rpc_t *retval );
static GtkWindow *stopish_new( void );
static void start_cb( GtkButton* button, gpointer data );
static void reset_cb( GtkButton* button, gpointer data );
static void close_cb( GtkButton* button, gpointer data );


int main( int argc, char *argv[] )
{
    osso_context_t *ctxt;
    osso_return_t ret;
    GtkWindow *window;

    //printf( "stopish: starting up\n" );

    ctxt = osso_initialize( "com.nokia.stopish", PACKAGE_VERSION, TRUE, NULL );
    if ( ctxt == NULL ) {
        fprintf( stderr, "osso_initialize failed.\n" );
        exit( 1 );
    }

    gtk_init( &argc, &argv );

    window = stopish_new(  );

    ret = osso_rpc_set_default_cb_f( ctxt, dbus_callback, window );
    if ( ret != OSSO_OK ) {
        fprintf( stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret );
        exit( 1 );
    }

    gtk_main(  );

    return 0;
}


gint dbus_callback( const gchar *interface, const gchar *method,
    	            GArray *arguments, gpointer data, osso_rpc_t *retval )
{
    //printf( "stopish dbus: %s, %s\n", interface, method );

    if ( !strcmp( method, "top_application" ) )
        gtk_window_present( GTK_WINDOW( data ) );

    retval->type = DBUS_TYPE_INVALID;

    return OSSO_OK;
}


int stopish_get_mode( void )
{
    return stopishMode;
}


static GtkWindow *stopish_new( void )
{
    GtkWidget *window, *hBox, *label, *button, *button0;
    GtkWidget *vBox, *vBox0, *vBox1;

    window = gtk_window_new( GTK_WINDOW_TOPLEVEL );

    gtk_container_set_border_width( GTK_CONTAINER( window ), 20 );

    gtk_window_set_title( GTK_WINDOW( window ), "Stopish" );

    g_signal_connect( G_OBJECT( window ), "destroy",
                      G_CALLBACK( close_cb ), window );

    vBox = gtk_vbox_new( FALSE, 20 );

    label = gtk_label_new( "Stopish - The Stopwatch" );
    gtk_box_pack_start( GTK_BOX( vBox ), label, FALSE, FALSE, 0 );

    hBox = gtk_hbox_new( FALSE, 10 );

    // stop watch area
    vBox0 = gtk_vbox_new( FALSE, 5 );
    gtk_widget_set_size_request( vBox0, 250, -1 );

    // main timer
    timerLabel = gtk_label_new( NULL );
    gtk_widget_set_size_request( timerLabel, -1, 150 );
    gtk_label_set_markup( GTK_LABEL( timerLabel ),
                          "<span font_family=\"monospace\" size=\"xx-large\">00:00:00.0</span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerLabel, FALSE, FALSE, 0 );

    // history area
    timerHistoryLabel1 = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel1, FALSE, FALSE, 0 );
    timerHistoryLabel2 = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel2, FALSE, FALSE, 0 );
    timerHistoryLabel3 = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel3, FALSE, FALSE, 0 );
    timerHistoryLabel4 = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel4, FALSE, FALSE, 0 );
    label = gtk_label_new( NULL );
    gtk_container_add( GTK_CONTAINER( vBox0 ), label );

    gtk_container_add( GTK_CONTAINER( hBox ), vBox0 );

    // button area
    vBox1 = gtk_vbox_new( FALSE, 15 );
    gtk_widget_set_size_request( vBox1, 200, -1 );

    // start/pause stopwatch button
    button = gtk_button_new_with_label( "Start" );
    button0 = gtk_button_new_with_label( "Reset" );
    gtk_widget_set_size_request( button, -1, 60 );
    g_signal_connect( G_OBJECT( button ), "clicked",
                      G_CALLBACK( start_cb ), button0 );
    gtk_box_pack_start( GTK_BOX( vBox1 ), button, FALSE, FALSE, 0 );

    // reset button
    gtk_widget_set_sensitive( button0, FALSE );
    gtk_widget_set_size_request( button0, -1, 60 );
    g_signal_connect( G_OBJECT( button0 ), "clicked",
                      G_CALLBACK( reset_cb ), button );
    gtk_box_pack_start( GTK_BOX( vBox1 ), button0, FALSE, FALSE, 0 );

    gtk_container_add( GTK_CONTAINER( hBox ), vBox1 );

    gtk_container_add( GTK_CONTAINER( vBox ), hBox );

    gtk_container_add( GTK_CONTAINER( window ), vBox );

    gtk_widget_show_all( window );

    return GTK_WINDOW( window );
}


static void start_cb( GtkButton* button, gpointer data )
{
    if ( stopishMode == STOPISH_MODE_START ) {
        // set label text and add timer handle
        gtk_button_set_label( button, "Pause" );
        stopishMode = STOPISH_MODE_PAUSE;
        stopish_set_time_start( stopish_current_time(  ) );
        timerHandle = g_timeout_add( 100, stopish_timeout_cb, timerLabel );
    }
    else if ( stopishMode == STOPISH_MODE_RESUME ) {
        // resume timer
        gtk_button_set_label( button, "Pause" );
        stopishMode = STOPISH_MODE_PAUSE;
        stopish_timer_resume(  );
        timerHandle = g_timeout_add( 100, stopish_timeout_cb, timerLabel );
    }
    else {
        // pause timer, remove timeout
        gtk_button_set_label( button, "Resume" );
        stopishMode = STOPISH_MODE_RESUME;
        g_source_remove( timerHandle );
        stopish_timer_save(  );
    }

    // allow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( data ), TRUE );
}


static void reset_cb( GtkButton* button, gpointer data )
{
    GSList *tempList;
    char *tempString;
    char formatString[64];

    if ( stopishMode == STOPISH_MODE_RESUME )
        stopish_timer_resume(  );

    // set label text and remove timer handle
    gtk_button_set_label( GTK_BUTTON( data ), "Start" );
    stopishMode = STOPISH_MODE_START;
    gtk_label_set_markup( GTK_LABEL( timerLabel ),
                          "<span font_family=\"monospace\" size=\"xx-large\">00:00:00.0</span>" );
    g_source_remove( timerHandle );

    // add current time to history
    historyList = g_slist_prepend( historyList,
                                   ( gpointer ) stopish_get_time_string(  ) );
    gtk_label_set_text( GTK_LABEL( timerHistoryLabel1 ),
                        ( char * ) historyList->data );
    tempList = historyList;
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel2 ),
                              formatString );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"x-small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel3 ),
                              formatString );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"xx-small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel4 ),
                              formatString );
    }

    // remove the history time after the 4th
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        tempString = tempList->data;
        historyList = g_slist_remove( historyList, tempList->data );
        free( tempString );
    }

    // reset start time
    stopish_set_time_start( 0 );

    // disallow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( button ), FALSE );
}


static void close_cb( GtkButton* button, gpointer data )
{
    // destroy main window and exit gtk main loop
    gtk_widget_destroy( GTK_WIDGET( data ) );
    gtk_main_quit(  );
}
