/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <QProgressBar>
#include <QGraphicsAnchorLayout>
#include <QGraphicsProxyWidget>
#include <QMenuBar>
#include <QGraphicsGridLayout>
#include <QSignalTransition>
#include <QPropertyAnimation>
#include <QStateMachine>
#include <QGraphicsSceneMouseEvent>
#include <QParallelAnimationGroup>
#include <QLayout>
#include <QStackedWidget>
#include <QTimer>

#ifdef Q_WS_MAEMO_5
 #include <QtMaemo5>
 #include <QDBusConnection>
 #include <mce/dbus-names.h>
#endif


#include "StelTranslator.hpp"
#include "StelCore.hpp"
#include "StelNavigator.hpp"
#include "ConstellationMgr.hpp"
#include "StelModuleMgr.hpp"
#include "GridLinesMgr.hpp"
#include "StarMgr.hpp"
#include "StelLocaleMgr.hpp"
#include "StelMovementMgr.hpp"
#include "StelObjectMgr.hpp"

#include "StelApp.hpp"
#include "StelSkyDrawer.hpp"
#include "StelAppGraphicsWidget.hpp"
#include "NebulaMgr.hpp"
#include "SolarSystem.hpp"
#include "LandscapeMgr.hpp"
#include "StelMainWindow.hpp"
#include "StelMainGraphicsView.hpp"

#include "MobileGui.hpp"
#include "TimeWidget.hpp"
#include "HelpMessage.hpp"
#include "dialogs/LocationDialog.hpp"
#include "dialogs/DateTimeDialog.hpp"
#include "dialogs/ProjectionDialog.hpp"
#include "dialogs/LandscapeDialog.hpp"
#include "dialogs/StarloreDialog.hpp"
#include "dialogs/SearchDialog.hpp"
#include "dialogs/SkyDialog.hpp"
#include "dialogs/MarkingDialog.hpp"
#include "dialogs/AboutDialog.hpp"

#include "InfoPanel.hpp"
#include "StatusBar.hpp"
#include "ImageButton.hpp"
#include "QuickBar.hpp"
#include "TimeBar.hpp"
#include "GraphicsTextLabel.hpp"
#include "MobileLocationMgr.hpp"

static QDBusConnection dbusConnection = QDBusConnection::systemBus();

QProgressBar* MobileGui::addProgressBar()
{
	return new QProgressBar();
}


void MobileGui::init(QGraphicsWidget* atopLevelGraphicsWidget, StelAppGraphicsWidget* astelAppGraphicsWidget)
{
	qDebug() << "Initialising GUI";

	QCoreApplication::setApplicationName("Stellarium Mobile");
	QCoreApplication::setApplicationVersion(MOBILE_GUI_VERSION);
	QCoreApplication::setOrganizationDomain("stellarium-mobile.org");
	QCoreApplication::setOrganizationName("Stellarium Mobile");

	StelMainWindow::getInstance().setWindowTitle(QString("Stellarium Mobile ")+MOBILE_GUI_VERSION);

	// Allow to load the resources when used as a static plugin
	Q_INIT_RESOURCE(mobile);

	//grabZoomKeys(true); 	// now done in StelMainWindow.cpp to avoid strange escaping events

	dbusConnection.connect(MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,	MCE_TKLOCK_MODE_SIG,
			this, SLOT(cb_func(const QDBusMessage&)));


	StelGuiBase::init(atopLevelGraphicsWidget, astelAppGraphicsWidget);

	// Creating all GUI actions
	createActions();

	// Creating GUI elements
	baseItem = new GuiBaseItem(this, astelAppGraphicsWidget);

	QGraphicsGridLayout* l = new QGraphicsGridLayout();
	l->setContentsMargins(0,0,0,0);
	l->setSpacing(0);
	astelAppGraphicsWidget->setLayout(l);
	l->addItem(baseItem, 0, 0);

	StelApp::getInstance().getStelObjectMgr().setObjectSearchRadius(60);
	StelApp::getInstance().getStelObjectMgr().setDistanceWeight(0.2f);
	GETSTELMODULE(StelMovementMgr)->setDragTriggerDistance(30.f);

	GETSTELMODULE(StelMovementMgr)->setInitFov(130.);
	GETSTELMODULE(StelMovementMgr)->zoomTo(130., 0.f);
	Vec3d v(0.,-1,1.2);
	v.normalize();
	GETSTELMODULE(StelMovementMgr)->setViewDirectionJ2000(StelApp::getInstance().getCore()->getNavigator()->altAzToJ2000(v));
	GETSTELMODULE(StelMovementMgr)->setInitViewDirectionToCurrent();
	GETSTELMODULE(StelMovementMgr)->setFlagAutoZoomOutResetsDirection(false);

	// If kept to 16, 15 or 14 px, there is a font display bug...
	GETSTELMODULE(ConstellationMgr)->setFontSize(16);
	GETSTELMODULE(SolarSystem)->setFontSize(14);

	StelApp::getInstance().getCore()->getSkyDrawer()->setBig3dModelHaloRadius(70.f);
	StelApp::getInstance().getCore()->getSkyDrawer()->setTwinkleAmount(0.3f);
}


void MobileGui::createActions()
{
	QAction* action;

	// Time
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	QString group = N_("Date and Time");
	action = addGuiActions("actionSet_Real_Time_Speed", N_("Set normal time rate"), "K", group, false, false);
	QObject::connect(action, SIGNAL(triggered()), nav, SLOT(setRealTimeSpeed()));

	action = addGuiActions("actionDecrease_Time_Speed", N_("Decrease time speed"), "J", group, false, false);
	QObject::connect(action, SIGNAL(triggered()), nav, SLOT(decreaseTimeSpeed()));

	action = addGuiActions("actionIncrease_Time_Speed", N_("Increase time speed"), "L", group, false, false);
	QObject::connect(action, SIGNAL(triggered()), nav, SLOT(increaseTimeSpeed()));

	action = addGuiActions("actionSet_Time_Rate_Zero", N_("Set time rate to zero"), "7", group, false, false);
	QObject::connect(action, SIGNAL(triggered()), nav, SLOT(setZeroTimeSpeed()));

	action = addGuiActions("actionReturn_To_Current_Time", N_("Set time to now"), "8", group, false, false);
	QObject::connect(action, SIGNAL(triggered()), nav, SLOT(setTimeNow()));

	// Grids
	GridLinesMgr* gmgr = GETSTELMODULE(GridLinesMgr);
	group = N_("Display Options");
	action = addGuiActions("actionShow_Azimuthal_Grid", N_("Azimuthal grid"), "Z", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagAzimuthalGrid(bool)));
	action->setChecked(gmgr->getFlagAzimuthalGrid());

	action = addGuiActions("actionShow_Equatorial_Grid", N_("Equatorial grid"), "E", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagEquatorGrid(bool)));
	action->setChecked(gmgr->getFlagEquatorGrid());

	action = addGuiActions("actionShow_Equatorial_J2000_Grid", N_("Equatorial J2000 grid"), "", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagEquatorJ2000Grid(bool)));
	action->setChecked(gmgr->getFlagEquatorJ2000Grid());

	action = addGuiActions("actionShow_Galactic_Grid", N_("Galactic grid"), "", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagGalacticGrid(bool)));
	action->setChecked(gmgr->getFlagGalacticGrid());

	action = addGuiActions("actionShow_Ecliptic_Line", N_("Ecliptic line"), ",", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagEclipticLine(bool)));
	action->setChecked(gmgr->getFlagEclipticLine());

	action = addGuiActions("actionShow_Equator_Line", N_("Equator line"), ".", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagEquatorLine(bool)));
	action->setChecked(gmgr->getFlagEquatorLine());

	action = addGuiActions("actionShow_Meridian_Line", N_("Meridian line"), ";", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), gmgr, SLOT(setFlagMeridianLine(bool)));
	action->setChecked(gmgr->getFlagMeridianLine());

	// Ground
	LandscapeMgr* lmgr = GETSTELMODULE(LandscapeMgr);
	action = addGuiActions("actionShow_Landscape", N_("Ground & Atmosphere"), "G", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), lmgr, SLOT(setFlagLandscape(bool)));
	QObject::connect(action, SIGNAL(toggled(bool)), lmgr, SLOT(setFlagAtmosphere(bool)));
	QObject::connect(action, SIGNAL(toggled(bool)), lmgr, SLOT(setFlagFog(bool)));
	action->setChecked(lmgr->getFlagLandscape());

	addGuiActions("actionShow_Atmosphere", N_("Atmosphere"), "A", group, true, false);
	QObject::connect(getGuiActions("actionShow_Atmosphere"), SIGNAL(toggled(bool)), lmgr, SLOT(setFlagAtmosphere(bool)));
	getGuiActions("actionShow_Atmosphere")->setChecked(lmgr->getFlagAtmosphere());

	action = addGuiActions("actionShow_Cardinal_Points", N_("Cardinal points"), "Q", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), lmgr, SLOT(setFlagCardinalsPoints(bool)));
	action->setChecked(lmgr->getFlagCardinalsPoints());

	// Nebulas
	NebulaMgr* nmgr = GETSTELMODULE(NebulaMgr);
	action = addGuiActions("actionShow_Nebulas", N_("Nebulas labels"), "N", group, true, false);
	action->setChecked(nmgr->getFlagHints());
	QObject::connect(action, SIGNAL(toggled(bool)), nmgr, SLOT(setFlagHints(bool)));

	// Planets
	SolarSystem* ssmgr = GETSTELMODULE(SolarSystem);
	action = addGuiActions("actionShow_Planets_Labels", N_("Planets labels"), "P", group, true, false);
	action->setChecked(ssmgr->getFlagLabels());
	QObject::connect(action, SIGNAL(toggled(bool)), ssmgr, SLOT(setFlagLabels(bool)));

	// constellations
	ConstellationMgr* cmgr = GETSTELMODULE(ConstellationMgr);
	action = addGuiActions("actionShow_Constellation_Lines", N_("Constellation lines"), "C", group, true, false);
	action->setChecked(cmgr->getFlagLines());
	QObject::connect(action, SIGNAL(toggled(bool)), cmgr, SLOT(setFlagLines(bool)));

	action = addGuiActions("actionShow_Constellation_Art", N_("Constellation art"), "R", group, true, false);
	action->setChecked(cmgr->getFlagArt());
	QObject::connect(action, SIGNAL(toggled(bool)), cmgr, SLOT(setFlagArt(bool)));

	action = addGuiActions("actionShow_Constellation_Labels", N_("Constellation labels"), "V", group, true, false);
	action->setChecked(cmgr->getFlagLabels());
	QObject::connect(action, SIGNAL(toggled(bool)), cmgr, SLOT(setFlagLabels(bool)));

	action = addGuiActions("actionShow_Constellation_Boundaries", N_("Constellation boundaries"), "B", group, true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), cmgr, SLOT(setFlagBoundaries(bool)));
	action->setChecked(cmgr->getFlagBoundaries());

	StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
	addGuiActions("actionGoto_Selected_Object", N_("Center on selected object"), "Space", group, false, false);
	QObject::connect(getGuiActions("actionGoto_Selected_Object"), SIGNAL(triggered()), mmgr, SLOT(setFlagTracking()));

	addGuiActions("actionZoom_In_Auto", N_("Zoom in on selected object"), "/", group, false, false);
	QObject::connect(getGuiActions("actionZoom_In_Auto"), SIGNAL(triggered()), this, SLOT(zoomInAuto()));

	addGuiActions("actionZoom_Out_Auto", N_("Zoom out"), "\\", group, false, false);
	QObject::connect(getGuiActions("actionZoom_Out_Auto"), SIGNAL(triggered()), this, SLOT(zoomOutAuto()));
}

void MobileGui::cb_func(const QDBusMessage& rep)
{
	// Screen dimmed to save battery or was re-activated
	if (rep.type()==QDBusMessage::SignalMessage && rep.path()=="/com/nokia/mce/signal" && rep.member()=="tklock_mode_ind")
	{
		QString arg0 = rep.arguments().at(0).toString();
		if (arg0=="locked")
		{
			qWarning() << "Rendering paused to save battery";
			StelMainGraphicsView::getInstance().setDoPaint(false);
		}
		else if (arg0=="unlocked")
		{
			if (StelMainWindow::getInstance().isActiveWindow())
			{
				StelMainGraphicsView::getInstance().setDoPaint(true);
				qWarning() << "Rendering restarted";
			}
		}
	}
}

void MobileGui::zoomInAuto()
{
	StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
	if (!mmgr->getFlagTracking())
	{
		if (StelApp::getInstance().getStelObjectMgr().getWasSelected())
		{
			mmgr->moveToObject(StelApp::getInstance().getStelObjectMgr().getSelectedObject()[0]);
			mmgr->setFlagTracking(true);
		}
	}
	else
		mmgr->autoZoomIn();
}

void MobileGui::zoomOutAuto()
{
	GETSTELMODULE(StelMovementMgr)->autoZoomOut();
}

//void MobileGui::grabZoomKeys(bool grab)
//{
//	qDebug() << "Grab zoom keys";
//	if (!StelMainWindow::getInstance().winId()) {
//		qWarning("Can't grab keys unless we have a window id");
//		return;
//	}

//	unsigned long val = (grab) ? 1 : 0;
//	Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
//	if (!atom) {
//		qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
//				 "on a Maemo 5 device!");
//		return;
//	}

//	XChangeProperty (QX11Info::display(),
//			StelMainWindow::getInstance().winId(),
//			atom,
//			XA_INTEGER,
//			32,
//			PropModeReplace,
//			reinterpret_cast<unsigned char *>(&val),
//			1);
//}

void GuiBaseItem::paint(QPainter* p, const QStyleOptionGraphicsItem* s, QWidget* w)
{
	update();
	QGraphicsWidget::paint(p, s, w);
}

GuiBaseItem::GuiBaseItem(MobileGui* amobileGui, StelAppGraphicsWidget* parent):
		QGraphicsWidget(parent), mobileGui(amobileGui), fps(NULL), statusBar(NULL), accelerometerControl(false), lastWasDragging(false), firstUpdate(0), fastForwardToNight(false)
{
	setParent(parent);

	locMgr = new MobileLocationMgr(this);
	
	QGraphicsAnchorLayout* layout = new QGraphicsAnchorLayout(this);
	layout->setVerticalSpacing(0);
	layout->setContentsMargins(0,1,0,0); // why 0 doesn't work ?

	StelMainWindow& mainWindow = StelMainWindow::getInstance();
	createMenuBar(&mainWindow);

	bool flagShowFps = StelApp::getInstance().getSettings()->value("mobileGui/showFps", false).toBool();
	if (flagShowFps)
	{
		fps = new GraphicsTextLabel("43.2 FPS");
		fps->setContentsMargins(5,5,5,15);
		layout->addCornerAnchors(fps, Qt::TopRightCorner, layout, Qt::TopRightCorner);
	}
	
	helpMessage = new HelpMessage(this);
	layout->addAnchor(helpMessage, Qt::AnchorVerticalCenter, layout, Qt::AnchorVerticalCenter);
	layout->addAnchor(helpMessage, Qt::AnchorHorizontalCenter, layout, Qt::AnchorHorizontalCenter);

	QAction* action;
	SearchDialog* searchDialog = new SearchDialog(&mainWindow);
	action = mobileGui->addGuiActions("actionShow_SearchDialog", N_("Search"), "", "", false, false);
	connect(action, SIGNAL(triggered()), searchDialog, SLOT(show()));

	action = mobileGui->addGuiActions("actionAccelerometerControl", N_("Accelerometers Control"), "", "", true, false);
	connect(action, SIGNAL(toggled(bool)), this, SLOT(setAccelerometersControl(bool)));
	action->setChecked(getAccelerometersControl());

	action = mobileGui->addGuiActions("actionTimeControl", N_("Time Control"), "", "", true, false);
	connect(action, SIGNAL(toggled(bool)), this, SLOT(setDragTimeMode(bool)));
	action->setChecked(GETSTELMODULE(StelMovementMgr)->getDragTimeMode());

	action = mobileGui->addGuiActions("actionToggle_FullScreen", N_("Full-screen mode"), "", "", false, false);
	connect(action, SIGNAL(triggered()), &mainWindow, SLOT(toggleFullScreen()));

	// Add the Quick bar
	quickBar = new QuickBar(this, helpMessage);
	quickBar->setVisible(true);
	quickBar->setZValue(10);
	layout->addCornerAnchors(quickBar, Qt::BottomRightCorner, layout, Qt::BottomRightCorner);

	// Add the Time bar
	timeBar = new TimeBar(this, helpMessage);
	timeBar->setVisible(true);
	timeBar->setZValue(10);
	layout->addCornerAnchors(timeBar, Qt::BottomLeftCorner, layout, Qt::BottomLeftCorner);
	layout->addCornerAnchors(timeBar, Qt::BottomRightCorner, layout, Qt::BottomRightCorner);

	infoPanel = new InfoPanel(this, helpMessage);
	infoPanel->setVisible(false);
	infoPanel->setZValue(10);

	// Add the Quick bar button
	quickButton =  new ImageButton(":/mobile/openQuickBar.png", NULL, this, ":/mobile/openQuickBarHalo.png");
	quickButton->setZValue(10);
	layout->addCornerAnchors(quickButton, Qt::BottomRightCorner, layout, Qt::BottomRightCorner);

	// Add the Time bar button
	timeBarButton = new ImageButton(":/mobile/openTimeBar.png", NULL, this, ":/mobile/openTimeBarHalo.png");
	timeBarButton->setZValue(5);
	layout->addCornerAnchors(timeBarButton, Qt::BottomLeftCorner, layout, Qt::BottomLeftCorner);

	// Add the status bar
	statusBar = new StatusBar(this);
	layout->addCornerAnchors(statusBar, Qt::BottomLeftCorner, layout, Qt::BottomLeftCorner);
	layout->addCornerAnchors(statusBar, Qt::BottomRightCorner, quickButton, Qt::BottomLeftCorner);
	statusBar->setZValue(8);

	QStateMachine* stateMachine = new QStateMachine();
	QState* stateNormal = new QState();
	stateNormal->assignProperty(quickBar, "opacity", 0);
	stateNormal->assignProperty(quickButton, "opacity", 1);
	stateNormal->assignProperty(statusBar, "opacity", 1);
	stateNormal->assignProperty(timeBar, "opacity", 0);
	stateNormal->assignProperty(timeBarButton, "opacity", 1);

	QState* stateQuickBar = new QState();
	stateQuickBar->assignProperty(quickBar, "opacity", 1);
	stateQuickBar->assignProperty(quickButton, "opacity", 0);
	stateQuickBar->assignProperty(statusBar, "opacity", 0);
	stateQuickBar->assignProperty(timeBar, "opacity", 0);
	stateQuickBar->assignProperty(timeBarButton, "opacity", 0);

	QState* stateTimeBar = new QState();
	stateTimeBar->assignProperty(quickBar, "opacity", 0);
	stateTimeBar->assignProperty(quickButton, "opacity", 0);
	stateTimeBar->assignProperty(statusBar, "opacity", 0);
	stateTimeBar->assignProperty(timeBar, "opacity", 1);
	stateTimeBar->assignProperty(timeBarButton, "opacity", 0);
	QObject::connect(stateTimeBar, SIGNAL(entered()), this, SLOT(timeBarStateWasEntered()));
	QObject::connect(stateTimeBar, SIGNAL(exited()), this, SLOT(timeBarStateWasExited()));

	QPropertyAnimation* a1 = new QPropertyAnimation(quickBar, "opacity");
	a1->setDuration(200);
	QPropertyAnimation* a2 = new QPropertyAnimation(quickButton, "opacity");
	a2->setDuration(200);
	QPropertyAnimation* a3 = new QPropertyAnimation(statusBar, "opacity");
	a3->setDuration(200);
	QPropertyAnimation* a4 = new QPropertyAnimation(timeBar, "opacity");
	a4->setDuration(200);
	QPropertyAnimation* a5 = new QPropertyAnimation(timeBarButton, "opacity");
	a5->setDuration(200);
	QParallelAnimationGroup* modeSwitchAnimation = new QParallelAnimationGroup(this);
	modeSwitchAnimation->addAnimation(a1);
	modeSwitchAnimation->addAnimation(a2);
	modeSwitchAnimation->addAnimation(a3);
	modeSwitchAnimation->addAnimation(a4);
	modeSwitchAnimation->addAnimation(a5);

	QSignalTransition* tr1 = stateNormal->addTransition(quickButton, SIGNAL(triggered()), stateQuickBar);
	tr1->addAnimation(modeSwitchAnimation);

	QSignalTransition* tr2 = stateQuickBar->addTransition(this, SIGNAL(clicked()), stateNormal);
	tr2->addAnimation(modeSwitchAnimation);

	QSignalTransition* tr3 = stateNormal->addTransition(timeBarButton, SIGNAL(triggered()), stateTimeBar);
	tr3->addAnimation(modeSwitchAnimation);

	QSignalTransition* tr4 = stateTimeBar->addTransition(this, SIGNAL(clicked()), stateNormal);
	tr4->addAnimation(modeSwitchAnimation);

	stateMachine->addState(stateNormal);
	stateMachine->addState(stateQuickBar);
	stateMachine->addState(stateTimeBar);
	stateMachine->setInitialState(stateNormal);
	stateMachine->start();

	mainWindow.installEventFilter(this);
	StelMainGraphicsView::getInstance().getStelAppGraphicsWidget()->installSceneEventFilter(this);
}

void GuiBaseItem::mousePressEvent(QGraphicsSceneMouseEvent* event)
{
	QGraphicsWidget::mousePressEvent(event);
}

void GuiBaseItem::mouseReleaseEvent(QGraphicsSceneMouseEvent* event)
{
	QGraphicsWidget::mouseReleaseEvent(event);
}

bool GuiBaseItem::sceneEventFilter(QGraphicsItem* watched, QEvent* ev)
{
	//qDebug() << ev;
	if (watched==StelMainGraphicsView::getInstance().getStelAppGraphicsWidget())
	{
		if (ev->type() == QEvent::GraphicsSceneMousePress)
		{
			StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
			if (mmgr->getDragTimeMode() && !lastWasDragging)
			{
				lastWasDragging = true;
				GridLinesMgr* glmgr = GETSTELMODULE(GridLinesMgr);
				savedGridFlag = glmgr->getFlagEquatorGrid();
				glmgr->setFlagEquatorGrid(true);
				timeBar->setIsTimeDragging(true);
			}
		}

		if (ev->type() == QEvent::GraphicsSceneMouseRelease)
		{
			StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
			if (mmgr->getDragTimeMode() && lastWasDragging)
			{
				lastWasDragging = false;
				GridLinesMgr* glmgr = GETSTELMODULE(GridLinesMgr);
				glmgr->setFlagEquatorGrid(savedGridFlag);
				timeBar->setIsTimeDragging(false);
			}

			if (timeBar->isVisible() && timeBar->opacity()>0 && !StelApp::getInstance().getCore()->getMovementMgr()->getHasDragged())
			{
				emit clicked();
				return true;
			}

			if (quickBar->isVisible() && quickBar->opacity()>0 && !StelApp::getInstance().getCore()->getMovementMgr()->getHasDragged())
			{
				emit clicked();
				return true;
			}
		}
	}
	return false;
}

bool GuiBaseItem::eventFilter(QObject *obj, QEvent *ev)
{
	StelMainWindow* mainWindow = &StelMainWindow::getInstance();
	if (obj != mainWindow)
		return QGraphicsWidget::eventFilter(obj, ev);

	if (ev->type() == 6) // QEvent::KeyPress
	{
		QKeyEvent* event = static_cast<QKeyEvent*>(ev);
		switch (event->key())
		{
		case Qt::Key_F7:
			GETSTELMODULE(StelMovementMgr)->zoomIn(true);
			return true;

		case Qt::Key_F8:
			GETSTELMODULE(StelMovementMgr)->zoomOut(true);
			return true;
		}
	}

	if (ev->type() == 7) // QEvent::KeyRelease
	{
		QKeyEvent* event = static_cast<QKeyEvent*>(ev);
		switch (event->key())
		{
		case Qt::Key_F7:
			GETSTELMODULE(StelMovementMgr)->zoomIn(false);
			return true;

		case Qt::Key_F8:
			GETSTELMODULE(StelMovementMgr)->zoomOut(false);
			return true;
		}
	}

	if (ev->type() == 25) // QEvent::WindowDeactivate
	{
		StelMainGraphicsView::getInstance().setDoPaint(false);
	}
	if (ev->type() == 24) // QEvent::WindowActivate
	{
		StelMainGraphicsView::getInstance().setDoPaint(true);
	}
	return false;
}


StelGuiBase* MobileGuiPluginInterface::getStelGuiBase() const
{
	return new MobileGui();
}
Q_EXPORT_PLUGIN2(StelGui, MobileGuiPluginInterface)


void DialogCreator::show()
{
	StelMainWindow* mainWindow = &StelMainWindow::getInstance();
	QObject* object = metaClass->newInstance(Q_ARG(QWidget*, mainWindow));
	QDialog* dialog = qobject_cast<QDialog*>(object);
	Q_ASSERT(dialog);
	StelMainGraphicsView::getInstance().setDoPaint(false);
	dialog->exec();
	dialog->deleteLater();
	StelMainGraphicsView::getInstance().setDoPaint(true);
}

void GuiBaseItem::addDialogInMenuBar(const QString& name, const QMetaObject* metaClass)
{
	StelMainWindow& mainWindow = StelMainWindow::getInstance();
	DialogCreator* creator = new DialogCreator(metaClass);

	QString actionName = QString("actionShow_%1s").arg(name);
	QAction* action = mobileGui->addGuiActions(actionName, name, "", "", false, false);
	connect(action, SIGNAL(triggered()), creator, SLOT(show()));
	mainWindow.menuBar()->addAction(action);
}


QMenuBar* GuiBaseItem::createMenuBar(QMainWindow* mainWindow)
{
	QMenuBar* menuBar = mainWindow->menuBar();

	addDialogInMenuBar(N_("Time"), &DateTimeDialog::staticMetaObject);
	addDialogInMenuBar(N_("Location"), &LocationDialog::staticMetaObject);
	addDialogInMenuBar(N_("Sky"), &SkyDialog::staticMetaObject);
	addDialogInMenuBar(N_("Markings"), &MarkingDialog::staticMetaObject);
	addDialogInMenuBar(N_("Landscape"), &LandscapeDialog::staticMetaObject);
	addDialogInMenuBar(N_("Starlore"), &StarloreDialog::staticMetaObject);
	addDialogInMenuBar(N_("Projection"), &ProjectionDialog::staticMetaObject);
	addDialogInMenuBar(N_("About"), &AboutDialog::staticMetaObject);

	return menuBar;
}

static int ocnt=0;
static int oax=0;
static int oay=0;
static int oaz=0;
static const char *accel_filename = "/sys/class/i2c-adapter/i2c-3/3-001d/coord";
bool getAccelerometers(Vec3f& v, float averagingCoef)
{
	FILE *fd;
	fd = fopen(accel_filename, "r");
	if (fd==NULL)
	{
		qWarning() << "Cannot read accelerometer info";
		return false;
	}
	int ax,ay,az;
	int rs=fscanf(fd,"%i %i %i",&ax,&ay,&az);
	fclose(fd);
	if (rs!=3)
	{
		qWarning() << "Cannot parse accelerometer info";
		return false;
	}
	int bx=ax;
	int by=ay;
	int bz=az;
	if(ocnt>0)
	{
		ax=oax+(bx-oax)*averagingCoef;
		ay=oay+(by-oay)*averagingCoef;
		az=oaz+(bz-oaz)*averagingCoef;
	}
	oax=ax;
	oay=ay;
	oaz=az;
	++ocnt;
	v.set(ax, ay, az);
	v/=1000;
	return true;
}

void GuiBaseItem::setAccelerometersControl(bool b)
{
	accelerometerControl=b;
	if (!b)
	{
		StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
		static const Vec3d up(0,0,1);
		mmgr->setViewUpVectorJ2000(StelApp::getInstance().getCore()->getNavigator()->altAzToJ2000(up));
		StelApp::getInstance().getCore()->setDefautAngleForGravityText(0.f);
	}
}

void GuiBaseItem::setDragTimeMode(bool b)
{
	StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
	if (mmgr->getDragTimeMode()==b)
		return;
	mmgr->setDragTimeMode(b);
}

void GuiBaseItem::timeBarStateWasEntered()
{
	StelApp::getInstance().getGui()->getGuiActions("actionTimeControl")->setChecked(true);
	helpMessage->displayMessage(q_("Drag sky to move in time"));
}

void GuiBaseItem::timeBarStateWasExited()
{
	StelApp::getInstance().getGui()->getGuiActions("actionTimeControl")->setChecked(false);
}

void GuiBaseItem::update()
{
	statusBar->updateText();
	timeBar->updateText();

	if (fps!=NULL)
	{
		QString str;
		QTextStream wos2(&str);
		wos2 << qSetRealNumberPrecision(3) << StelApp::getInstance().getFps() << " FPS";
		if (fps->text()!=str)
		{
			fps->setText(str);
		}
	}

	if (accelerometerControl)
	{
		// Update viewing position and up vector according to accelerometers.
		Vec3f v;
		float fov=qMin(130.f, StelApp::getInstance().getCore()->getProjection(StelCore::FrameJ2000)->getFov());
		fov = qMax(fov, 5.f);
		if (getAccelerometers(v, 0.05f+(fov-5.f)*0.15f/125.f))
		{
			if (v[2]>=0.90f)
				v[2]=0.90f;
			if (v[2]<=-0.90f)
				v[2]=-0.90f;
			float roll = std::asin(v[2]);
			float pitch = -atan2(v[0], -v[1]);

			StelApp::getInstance().getCore()->setDefautAngleForGravityText(pitch*180.f/M_PI);
			
			StelMovementMgr* mmgr = GETSTELMODULE(StelMovementMgr);
			Vec3d viewDirection = StelApp::getInstance().getCore()->getNavigator()->j2000ToAltAz(mmgr->getViewDirectionJ2000());

			float lng, lat;
			StelUtils::rectToSphe(&lng, &lat, viewDirection);
			StelUtils::spheToRect(lng, roll, viewDirection);
			mmgr->setViewDirectionJ2000(StelApp::getInstance().getCore()->getNavigator()->altAzToJ2000(viewDirection));

			Vec3f viewHoriz;
			StelUtils::spheToRect(lng, 0, viewHoriz);
			Mat4f rot=Mat4f::rotation(viewHoriz, pitch);
			Vec3f up(0,0,1);
			up.transfo4d(rot);
			Vec3d tmp(up[0],up[1],up[2]);
			mmgr->setViewUpVectorJ2000(StelApp::getInstance().getCore()->getNavigator()->altAzToJ2000(tmp));
		}
	}

	// Wait for 5 updates for the display to settle down.
	if (firstUpdate>5)
	{
		firstUpdate=-1;	// Stop counting
		const Vec3d& sunPos = GETSTELMODULE(SolarSystem)->getSun()->getAltAzPos(StelApp::getInstance().getCore()->getNavigator());
		if (sunPos[2]>-0.15)
		{
			// We start during the day: move to the night!
			fastForwardToNight = true;
			previousSinSun.append(sunPos[2]);
			startFastForwardTime = StelApp::getInstance().getCore()->getNavigator()->getJDay();
			helpMessage->displayMessage("It is daytime, fast forward time until...", 3000);
			StelApp::getInstance().getCore()->getNavigator()->setTimeRate(JD_HOUR*2);
		}
	}
	else
	{
		if (firstUpdate>=0)
			++firstUpdate;
	}
	if (fastForwardToNight)
	{
		const Vec3d& sunPos = GETSTELMODULE(SolarSystem)->getSun()->getAltAzPos(StelApp::getInstance().getCore()->getNavigator());
		if (sunPos[2]<-0.15)
		{
			fastForwardToNight=false;
			QString d = StelApp::getInstance().getLocaleMgr().getPrintableTimeLocal(StelApp::getInstance().getCore()->getNavigator()->getJDay());
			d.chop(3);
			helpMessage->displayMessage("Tonight at " + d, 3000);
			StelApp::getInstance().getCore()->getNavigator()->setTimeRate(JD_SECOND);
		}
		previousSinSun.append(sunPos[2]);
		if (previousSinSun.size()>3)
			previousSinSun.pop_front();
		if (StelApp::getInstance().getCore()->getNavigator()->getJDay()-startFastForwardTime>1.f || 
				 (previousSinSun.size()==3 && previousSinSun.at(2)-previousSinSun.at(1)>0 && previousSinSun.at(1)-previousSinSun.at(0)<0))
		{
			fastForwardToNight=false;
			helpMessage->displayMessage("Ooops! It seems that you never\nenjoy night sky at your location...", 4000);
			StelApp::getInstance().getCore()->getNavigator()->setTimeRate(JD_SECOND);
			StelApp::getInstance().getCore()->getNavigator()->setTimeNow();
		}
	}
}
