/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "InfoPanel.hpp"

#include <QSettings>
#include <QFont>
#include <QAction>
#include <QTextDocument>
#include <QGraphicsAnchorLayout>
#include <QGraphicsSceneMouseEvent>
#include <QGraphicsDropShadowEffect>

#include "StelApp.hpp"
#include "StelModuleMgr.hpp"
#include "StelObjectMgr.hpp"
#include "StelMovementMgr.hpp"
#include "StelMainGraphicsView.hpp"
#include "StelAppGraphicsWidget.hpp"
#include "MobileGui.hpp"
#include "ButtonGroup.hpp"
#include "ImageButton.hpp"
#include "GraphicItemWrapper.hpp"
#include "HelpMessage.hpp"
#include "StelTranslator.hpp"

InfoPanelMgr::InfoPanelMgr()
{
	setObjectName("InfoPanelMgr");
	lastUpdateTime = StelApp::getInstance().getTotalRunTime();
	previousTracking = GETSTELMODULE(StelMovementMgr)->getFlagTracking();
	hasSelectedInfo = !GETSTELMODULE(StelObjectMgr)->getSelectedObject().isEmpty();
	if (hasSelectedInfo)
		update(0);
}

void InfoPanelMgr::init()
{
	infoTextFilters = StelObject::InfoStringGroup(StelObject::PlainText | StelObject::Name | StelObject::Size | StelObject::AltAzi | StelObject::Magnitude | StelObject::Distance | StelObject::RaDecOfDate);
}

void InfoPanelMgr::update(double)
{
	const bool newFlagTracking = GETSTELMODULE(StelMovementMgr)->getFlagTracking();
	if (previousTracking!=newFlagTracking)
	{
		emit trackingModeChanged();
	}
	previousTracking=newFlagTracking;
	
	const QList<StelObjectP>& selected = GETSTELMODULE(StelObjectMgr)->getSelectedObject();
	if (!hasSelectedInfo && !selected.isEmpty())
	{
		// There is a new selected object
		previousText = selected.at(0)->getInfoString(StelApp::getInstance().getCore(), infoTextFilters);
		hasSelectedInfo = true;
		emit toggled(true);
		emit htmlChanged(previousText);
		return;
	}
	if (hasSelectedInfo && selected.isEmpty())
	{
		hasSelectedInfo = false;
		emit toggled(false);
		return;
	}

	// Don't update more often than every 0.5 sec
	const float t = StelApp::getInstance().getTotalRunTime();
	if (t-lastUpdateTime<0.5)
		return;
	lastUpdateTime = t;
	if (hasSelectedInfo && !selected.isEmpty())
	{
		QString s = selected.at(0)->getInfoString(StelApp::getInstance().getCore(), infoTextFilters);
		if (s!=previousText)
		{
			previousText = s;
			emit htmlChanged(previousText);
		}
		return;
	}
}

//! Handle mouse click events.
void InfoPanel::mousePressEvent(QGraphicsSceneMouseEvent* event)
{
	// If the user click on the info panel, we unselect the selected object.
	GETSTELMODULE(StelObjectMgr)->unSelect();
	event->accept();
}

InfoPanel::InfoPanel(QGraphicsItem* parent, HelpMessage* msg) : QGraphicsWidget(parent)
{
	// Add the module that manages the info panel.
	InfoPanelMgr* infoPanelMgr = new InfoPanelMgr();
	infoPanelMgr->init();
	StelApp::getInstance().getModuleMgr().registerModule(infoPanelMgr);

	connect(infoPanelMgr, SIGNAL(toggled(bool)), this, SLOT(toggledChanged(bool)));
	connect(infoPanelMgr, SIGNAL(trackingModeChanged()), this, SLOT(refreshButtonGroup()));
	connect(infoPanelMgr, SIGNAL(htmlChanged(QString)), this, SLOT(htmlChanged(QString)));

	QGraphicsAnchorLayout* layout = new QGraphicsAnchorLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(5, 5, 5, 5);
	setContentsMargins(0, 0, 0, 0);

	setLayout(layout);
	
	titleTextItem = new QGraphicsSimpleTextItem(this);
	titleTextItem->setBrush(QColor(Qt::white));
	QFont font;
	font.setPixelSize(30);
	font.setBold(true);
	titleTextItem->setFont(font);
	titleTextItem->setText("Title");
	GraphicItemWrapper* wrap = new GraphicItemWrapper(this, titleTextItem);
	layout->addCornerAnchors(wrap, Qt::TopLeftCorner, layout, Qt::TopLeftCorner);
	
	StelGuiBase* gui = StelApp::getInstance().getGui();
	
	QAction* action = gui->addGuiActions("actionShowMoreObjectInfo", N_("Show more info"), "", "ObjectInfo", true, false);
	QObject::connect(action, SIGNAL(toggled(bool)), this, SLOT(setShowMoreInfo(bool)));
	action->setChecked(false);
	
	btgrp = new ButtonGroup(this);
	btMoreObjectInfo = new ImageButton(":/mobile/moreObjectInfo.png", gui->getGuiActions("actionShowMoreObjectInfo"), NULL, "", msg);
	btCenterOnObject = new ImageButton(":/mobile/centerOnObject.png", gui->getGuiActions("actionGoto_Selected_Object"), NULL, "", msg);
	btAutoZoomIn = new ImageButton(":/mobile/autoZoomIn.png", gui->getGuiActions("actionZoom_In_Auto"), NULL, "", msg);
	btAutoZoomOut = new ImageButton(":/mobile/autoZoomOut.png", gui->getGuiActions("actionZoom_Out_Auto"), NULL, "", msg);
	btgrp->add(btMoreObjectInfo);
	btgrp->add(btAutoZoomIn);
	btgrp->add(btAutoZoomOut);
	layout->addCornerAnchors(btgrp, Qt::TopLeftCorner, wrap, Qt::BottomLeftCorner);

	detailsTextItem = new QGraphicsSimpleTextItem(this);
	detailsTextItem->setBrush(QColor(Qt::white));
	QFont font2;
	font2.setPixelSize(20);
	detailsTextItem->setFont(font2);
	detailsTextItem->setText("tmp\ntmp\ntmp\ntmp");
	GraphicItemWrapper* wrap2 = new GraphicItemWrapper(this, detailsTextItem);
	layout->addCornerAnchors(wrap2, Qt::TopLeftCorner, btgrp, Qt::BottomLeftCorner);
	detailsTextItem->setVisible(false);

	QGraphicsDropShadowEffect* shadow = new QGraphicsDropShadowEffect(this);
	shadow->setOffset(1.);
	shadow->setColor(QColor(0, 0, 0, 255));
	wrap->setGraphicsEffect(shadow);
	shadow = new QGraphicsDropShadowEffect(this);
	shadow->setOffset(0.5);
	shadow->setColor(QColor(0, 0, 0, 255));
	wrap2->setGraphicsEffect(shadow);

	setGeometry(0,0,200,200);
	setCacheMode(QGraphicsItem::ItemCoordinateCache);
}

void InfoPanel::setShowMoreInfo(bool b)
{
	detailsTextItem->setVisible(b);
	detailsTextItem->update();
}

void InfoPanel::toggledChanged(bool b)
{
	if (b)
		refreshButtonGroup();
	setVisible(b);
}

void InfoPanel::refreshButtonGroup()
{
	btgrp->clear();
	btCenterOnObject->setVisible(false);
	btAutoZoomIn->setVisible(false);
	btAutoZoomOut->setVisible(false);
	
	btgrp->add(btMoreObjectInfo);
	if (GETSTELMODULE(StelMovementMgr)->getFlagTracking())
	{
		btgrp->add(btAutoZoomIn);
		btgrp->add(btAutoZoomOut);
		btAutoZoomIn->setVisible(true);
		btAutoZoomOut->setVisible(true);
	}
	else
	{
		btgrp->add(btCenterOnObject);
		btCenterOnObject->setVisible(true);
	}
}

void InfoPanel::htmlChanged(QString html)
{
	int idx= html.indexOf('\n');
	titleTextItem->setText(html.left(idx));
	detailsTextItem->setText(html.mid(idx+1));
}
