/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "Gps.hpp"
#include <QDebug>

Gps* Gps::gps = NULL;

Gps* Gps::getInstance()
{
	if (gps == NULL)
	{
		gps = new Gps();
	}
	return gps;
}

#ifdef Q_WS_MAEMO_5

static void on_error(LocationGPSDControl *, LocationGPSDControlError error, gpointer)
{
	qDebug() << "Location error: " << error;
}

static void on_changed(LocationGPSDevice *device, gpointer)
{
	if (!device)
		return;

	Gps::getInstance()->fixReceived();
}

static void on_stop(LocationGPSDControl *, gpointer)
{
	qDebug() << "GPS Stopped";
}

Gps::Gps() : available(true)
{
	control = location_gpsd_control_get_default();
	device = (LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

	if (!control || !device)
	{
		available = false;
		return;
	}

	g_object_set(G_OBJECT(control),
		"preferred-method", LOCATION_METHOD_USER_SELECTED,
		"preferred-interval", LOCATION_INTERVAL_DEFAULT,
		NULL);

	g_signal_connect(control, "error-verbose", G_CALLBACK(on_error), NULL);
	g_signal_connect(device, "changed", G_CALLBACK(on_changed), control);
	g_signal_connect(control, "gpsd-stopped", G_CALLBACK(on_stop), NULL);

	//g_idle_add(start_location, control);
}

Gps::~Gps()
{
	g_object_unref(device);
	g_object_unref(control);
}

void Gps::fixReceived()
{
	emit positionUpdate();
}

void Gps::start()
{
	//qDebug() << "Start GPS updates";
	location_gpsd_control_start(control);
}

void Gps::stop()
{
	//qDebug() << "Stop GPS updates";
	location_gpsd_control_stop(control);
}

bool Gps::lastKnownPosition(double* lon, double* lat, double* alt, double* precision)
{
	if (!device || !device->fix)
		return false;

	if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET)
	{
		//qDebug() << "GPS fix: " << device->fix->latitude << device->fix->longitude << device->fix->altitude;
		*lat = device->fix->latitude;
		*lon = device->fix->longitude;
		if (device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET)
		{
			*alt = device->fix->altitude;
		}
		else
		{
			*alt = -1000000;
		}
		*precision = 0.01f*device->fix->eph;
		return true;
	}
	return false;
}

#endif
