# performs a simple device inquiry, followed by a remote name request of each
# discovered device

import csv
import dbus
from dbus.mainloop.glib import DBusGMainLoop
import re
import sys
import system_functions as sf
import struct
import bluetooth._bluetooth as bluez
import bluetooth


def ID_from_MAC(MAC):
    """converts MAC addr to acceptable ID_LENGTH

    Examples:
    >>> ID_from_MAC('12:34:56:78:90:AB')
    '90AB'
    """
    matches = re.search("(([0-9A-F]{2}):?([0-9A-F]{2}):?){3}", MAC)
    ID = matches.group(2) + matches.group(3)
    return ID


def printpacket(pkt):
    for c in pkt:
        sys.stdout.write("%02x " % struct.unpack("B", c)[0])
    print


def combine_rssi_readings(readings):
    """
    Returns the avg rssi reading for each addr in readings

    Example:
    >>> combine_rssi_readings( (('dev1','10'),('dev1','20'),('dev2','30')))
    {'dev2': 30, 'dev1': 15}
    """
    counts = {}
    #initialize counters
    for addr, rssi in readings:
        counts[addr] = [0, 0]

    for addr, rssi in readings:
        counts[addr][0] += 1
        counts[addr][1] += int(rssi)

    for addr in counts:
        counts[addr] = counts[addr][1] / counts[addr][0]

    return counts


LOOKUP_TIMEOUT = 40


class Device():

    def read_inquiry_mode(self, sock):
        """returns the current mode, or -1 on failure"""
        # save current filter
        old_filter = sock.getsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, 14)

        # Setup socket filter to receive only events related to the
        # read_inquiry_mode command
        flt = bluez.hci_filter_new()
        opcode = bluez.cmd_opcode_pack(bluez.OGF_HOST_CTL,
                bluez.OCF_READ_INQUIRY_MODE)
        bluez.hci_filter_set_ptype(flt, bluez.HCI_EVENT_PKT)
        bluez.hci_filter_set_event(flt, bluez.EVT_CMD_COMPLETE)
        bluez.hci_filter_set_opcode(flt, opcode)
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, flt)

        # first read the current inquiry mode.
        bluez.hci_send_cmd(sock, bluez.OGF_HOST_CTL,
                bluez.OCF_READ_INQUIRY_MODE)

        pkt = sock.recv(255)

        status, mode = struct.unpack("xxxxxxBB", pkt)
        if status != 0:
            mode = -1

        # restore old filter
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, old_filter)
        return mode

    def write_inquiry_mode(self, sock, mode):
        """returns 0 on success, -1 on failure"""
        # save current filter
        old_filter = sock.getsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, 14)

        # Setup socket filter to receive only events related to the
        # write_inquiry_mode command
        flt = bluez.hci_filter_new()
        opcode = bluez.cmd_opcode_pack(bluez.OGF_HOST_CTL,
                bluez.OCF_WRITE_INQUIRY_MODE)
        bluez.hci_filter_set_ptype(flt, bluez.HCI_EVENT_PKT)
        bluez.hci_filter_set_event(flt, bluez.EVT_CMD_COMPLETE)
        bluez.hci_filter_set_opcode(flt, opcode)
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, flt)

        # send the command!
        bluez.hci_send_cmd(sock, bluez.OGF_HOST_CTL,
                bluez.OCF_WRITE_INQUIRY_MODE, struct.pack("B", mode))

        pkt = sock.recv(255)

        status = struct.unpack("xxxxxxB", pkt)[0]

        # restore old filter
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, old_filter)
        if status != 0:
            return -1
        return 0

    def device_inquiry_with_rssi(self):

        dev_id = 0
        try:
            sock = bluez.hci_open_dev(dev_id)
        except:
            print "error accessing bluetooth device..."
            sys.exit(1)

        try:
            mode = self.read_inquiry_mode(sock)
        except Exception, e:
            print "error reading inquiry mode.  "
            print "Are you sure this a bluetooth 1.2 device?"
            print e
            sys.exit(1)
        #print "current inquiry mode is %d" % mode

        if mode != 1:
            #print "writing inquiry mode..."
            try:
                result = self.write_inquiry_mode(sock, 1)
            except Exception, e:
                print "error writing inquiry mode.  Are you sure you're root?"
                print e
                sys.exit(1)
            if result != 0:
                print "error while setting inquiry mode"
           # print "result: %d" % result

        # save current filter
        old_filter = sock.getsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, 14)

        # perform a device inquiry on bluetooth device #0
        # The inquiry should last 8 * 1.28 = 10.24 seconds
        # before the inquiry is performed, bluez should flush its cache of
        # previously discovered devices
        flt = bluez.hci_filter_new()
        bluez.hci_filter_all_events(flt)
        bluez.hci_filter_set_ptype(flt, bluez.HCI_EVENT_PKT)
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, flt)

        duration = 8
        max_responses = 255
        cmd_pkt = struct.pack("BBBBB", 0x33, 0x8b, 0x9e, duration, max_responses)
        bluez.hci_send_cmd(sock, bluez.OGF_LINK_CTL, bluez.OCF_INQUIRY, cmd_pkt)

        results = []

        done = False
        while not done:
            pkt = sock.recv(255)
            ptype, event, plen = struct.unpack("BBB", pkt[:3])
            if event == bluez.EVT_INQUIRY_RESULT_WITH_RSSI:
                pkt = pkt[3:]
                nrsp = struct.unpack("B", pkt[0])[0]
                for i in range(nrsp):
                    addr = bluez.ba2str(pkt[1+6*i:1+6*i+6])
                    rssi = struct.unpack("b", pkt[1+13*nrsp+i])[0]
                    results.append((addr, rssi))
                    #print "[%s] RSSI: [%d]" % (addr, rssi)
            elif event == bluez.EVT_INQUIRY_COMPLETE:
                done = True
            elif event == bluez.EVT_CMD_STATUS:
                status, ncmd, opcode = struct.unpack("BBH", pkt[3:7])
                if status != 0:
                    print "bluez.EVT_CMD_STATUS error"
                    printpacket(pkt[3:7])
                    done = True
            else:
                print "unrecognized packet type 0x%02x" % ptype

        # restore old filter
        sock.setsockopt(bluez.SOL_HCI, bluez.HCI_FILTER, old_filter)

        return results

    def __init__(self):
        DBusGMainLoop(set_as_default=True)
        bus = dbus.SystemBus()
        obj = bus.get_object('org.bluez', '/')
        self._manager = dbus.Interface(obj, 'org.bluez.Manager')

        obj = bus.get_object('org.bluez', self._manager.DefaultAdapter())
        self._adapter = dbus.Interface(obj, 'org.bluez.Adapter')

        btaddr = self._adapter.GetProperties()['Address']
        self._btnamecache = NameCache('/var/lib/bluetooth/' + \
            btaddr + '/names')
        self._adapter.connect_to_signal("RemoteNameUpdated",\
            self._btnamecache.insert)

    def _lookup_name(self, addr):
        """
        returns device name based on MAC
        """

        name = bluetooth.lookup_name(addr, LOOKUP_TIMEOUT)
        if name is None:
            print "BT Name not found"
            if self._btnamecache.contains(addr):
                name = self._btnamecache.lookup(addr)
            else:
                name = ID_from_MAC(addr)
        else:
            pass#self._btnamecache.insert(addr, name[:4])

        return name

    def exists(self):
        """
        determines presence of device
        """
        try:
            sf.run("hciconfig hci0")
            return True
        except sf.RunProcessError:
            return False
        except OSError:
            print "bluez missing"
            return False

    def scan(self):
        """
        returns dict of name -> (addr, rssi)
        """
        scan_results = {}
        readings = self.device_inquiry_with_rssi()

        results = combine_rssi_readings(readings)

        for addr in results:
            name = self._lookup_name(addr)
            scan_results[name] = (addr, results[addr])

        return scan_results

    def setName(self, name):
        """
        set device broadcast name
        """
        self._adapter.SetProperty('Name', name)

    def setPowerOff(self):
        """
        set device power to off
        """
        sf.run("hciconfig hci0 down")

    def setPowerOn(self):
        """
        set device power to on
        """
        sf.run("hciconfig hci0 up")


class NameCache():

    def __init__(self, filename=""):
        self._cache = {}
        if filename == "":
            return
        else:
            csvReader = csv.reader(open(filename, 'rb'), delimiter=' ')
            for row in csvReader:
                self.insert(row[0], row[1])

    def contains(self, addr):
        if self.lookup(addr) != None:
            return True
        else:
            return False

    def insert(self, addr, name):
        #print "inserting to cache", addr, name
        self._cache[addr] = name

    def lookup(self, addr):
        if addr in self._cache:
            return self._cache[addr][:4]
        else:
            return None


if __name__ == "__main__":
    import doctest
    doctest.testmod(verbose=True)
