import math
import random


def average(L):
    if len(L):
        return float(sum(L)) / len(L)
    else:
        return 0


def avg_pts(points):
    """

    Examples:
    >>> avg_pts(((-1,-1), (1,1), (3,0)))
    (1, 0)
    >>> avg_pts(((5,5),))
    (5, 5)
    """
    sumx = 0
    sumy = 0

    if len(points) == 0:
        return (0, 0)
    if len(points) == 1:
        return points[0]

    for (x, y) in points:
        sumx = sumx + x
        sumy = sumy + y
    return (sumx/len(points), sumy/len(points))


def combinations(iterable, r):
    # combinations('ABCD', 2) --> AB AC AD BC BD CD
    # combinations(range(4), 3) --> 012 013 023 123
    pool = tuple(iterable)
    n = len(pool)
    if r > n:
        return
    indices = range(r)
    yield tuple(pool[i] for i in indices)
    while True:
        for i in reversed(range(r)):
            if indices[i] != i + n - r:
                break
        else:
            return
        indices[i] += 1
        for j in range(i+1, r):
            indices[j] = indices[j-1] + 1
        yield tuple(pool[i] for i in indices)


def sumOfDistances(pt, pts):
    """
    sum of distances
    >>> sumOfDistances((0,0),((1,0),(0,1),(-1,0),(0,-1)))
    4.0
    """
    dist = 0
    for other_pt in pts:
        dist = dist + math.hypot(other_pt[0]-pt[0], other_pt[1]-pt[1])
    return dist


def build_constraints(ng):
    """
    Input: ng['nodeid']['pairid'] = dist != dict['pairid']['nodeid']
    Output: dict['nodeid']['pairid'] = dist = dict['pairid']['nodeid']
    """
    constraints = {}
    for nodeid in ng:
        for pairid in ng[nodeid]:
            distances = [ng[nodeid][pairid]]
            if pairid in ng:
                if nodeid in ng[pairid]:
                    distances.append(ng[pairid][nodeid])

            if nodeid not in constraints:
                constraints[nodeid] = {}
            if pairid not in constraints:
                constraints[pairid] = {}

            avg_dist = average(distances)
            constraints[nodeid][pairid] = avg_dist
            constraints[pairid][nodeid] = avg_dist
    return constraints


def build_system(ng, originID):
    """
    Builds 2-d Coordinate system from neighbor graph
        -Uses initial placement to increase speed
        -Based on Vivaldi method
        
    Examples:
    >>> build_system({'a': {'b': 3, 'c': 4, 'd': 2}, 'c': {'a': 2, 'b': 4}, 'd': {'a': 3, 'b': 5, 'c': 2}}, 'a')
    {'a': (0.0, 0.0), 'c': (3.0, 0.0), 'b': (0.3333333333333333, 2.9814239699997196), 'd': (0.6023285344836573, -1.596614741936323)}
    
    """
    constraints = build_constraints(ng)
    guess_layout = fast_placement(constraints, originID)
    final_layout = vivaldi(constraints, guess_layout)
    #gen_image(final_layout)
    return final_layout


def fast_placement(constraints, originID):
    """
    Examples:
    >>> fast_placement({'a': {'b': 3}, 'b': {'a': 3}},'a')
    {'a': (0.0, 0.0), 'b': (3.0, 0.0)}
    """
    guess_layout = {}
    guess_layout[originID] = (0.0, 0.0)
    for nodeid in sorted(constraints, key=len, reverse=True):
        if nodeid == originID:
            continue
        else:
            intersect_coords = []

            #find nodes which are placed and constrain node's location
            node_constraints = set(guess_layout) & set(constraints[nodeid])
            
            if len(node_constraints) == 0:
                continue
            
            #node bound only be single radius
            if len(node_constraints) == 1:
                const = node_constraints.pop()
                newpt = (guess_layout[const][0] + constraints[nodeid][const], guess_layout[const][1])
                guess_layout[nodeid] = newpt
                continue


            #find the solutions(2) to every pair of constraints
            for (const1, const2) in combinations(node_constraints, 2):
                intersect_coords.append(find_intersects( \
                        guess_layout[const1], constraints[nodeid][const1],\
                        guess_layout[const2], constraints[nodeid][const2]))

            assert len(intersect_coords) > 0
            set_coords = []

            #for all intersect points choose the point closest to the set_coords
            random.shuffle(intersect_coords)
            for int_pts in intersect_coords:
                if len(int_pts) == 1:
                    set_coords.append(int_pts[0])
                    continue

                #choose intersect point as base for soln
                if len(set_coords) == 0:
                    set_coords.append(random.sample(int_pts, 1)[0])
                
                elif sumOfDistances(int_pts[0], set_coords) < \
                                sumOfDistances(int_pts[1], set_coords):
                    set_coords.append(int_pts[0])
                else:
                    set_coords.append(int_pts[1])

            #average the set_coords to produce the estimated location of the node
            assert len( set_coords[0]) == 2, set_coords
            guess_layout[nodeid] = avg_pts(set_coords)

    return guess_layout


def find_intersects(pt0, r0, pt1, r1):
    """
    Based on code from http://paulbourke.net/geometry/2circle/
    
    returns solution to circle-circle intersection problem or
    if no solution exists the closest point to both circles
    
    throws: NoSolution(str)

    Examples:
    >>> find_intersects((0,0), 2, (2,0), 2)
    ((1.0, 1.7320508075688772), (1.0, -1.7320508075688772))

    """

    x0, y0 = pt0
    x1, y1 = pt1
    # dx and dy are the vertical and horizontal distances between
    # the circle centers.

    dx = x1 - x0
    dy = y1 - y0

    # Determine the straight-line distance between the centers.
    d = math.hypot(dy, dx)

    # Check for solvability.
    if (d > (r0 + r1)):
        #no solution. circles do not intersect. */
        #FIXME
        return (((x1-x0)/2, (y1-y0)/2),)

    if (d < abs(r0 - r1)):
        # no solution. one circle is contained in the other
        #FIXME
        return (((x1-x0)/2, (y1-y0)/2),)
    
    if d == 0:
        #circles centered at same point
        #FIXME returns avg point to +x 
        return ((x0 + (r0 + r1) / 2, y0),)

    # 'point 2' is the point where the line through the circle
    # intersection points crosses the line between the circle
    # centers.

    # Determine the distance from point 0 to point 2.
    a = ((r0*r0) - (r1*r1) + (d*d)) / (2.0 * d)

    # Determine the coordinates of point 2. */
    x2 = x0 + (dx * a/d)
    y2 = y0 + (dy * a/d)

    # Determine the distance from point 2 to either of the
    # intersection points.

    h = math.sqrt((r0*r0) - (a*a))

    # Now determine the offsets of the intersection points from
    # point 2.

    rx = -dy * (h/d)
    ry = dx * (h/d)

    # Determine the absolute intersection points.

    return ((x2 + rx, y2 + ry), ((x2 - rx), (y2 - ry)))


class NoSolution(Exception):

    def __init__(self, string):
        self.value = string

    def __str__(self):
        return repr(self.value)


#import matplotlib.pyplot as plt
#def gen_image(graph):
#    x = []
#    y = []
#    for node in graph:
#        x.append(graph[node][0])
#        y.append(graph[node][1])
#
#    plt.scatter(x, y, s=20, c='b', marker='+')
#    plt.savefig('graph')


def vivaldi(ng, guess_layout):
    """
    """
    #TODO implement me
    return guess_layout

if __name__ == "__main__":
    import doctest
    doctest.testmod(verbose=True)
