#include "albumwindow.h"
#include "ui_albumwindow.h"

AlbumWindow::AlbumWindow(QWidget *parent, QString token) :
    QMainWindow(parent),
    ui(new Ui::AlbumWindow),
    qfacebook(new QFacebook(token, this)),
    accessToken(token),
    nam(new QNetworkAccessManager(this))
{
    ui->setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);
    connect(ui->actionUpload_photo, SIGNAL(triggered()), this, SLOT(uploadPhoto()));
    connect(ui->actionDownload_all_photos, SIGNAL(triggered()), this, SLOT(downloadAllPhotos()));
    ui->statusBar->hide();
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
}

AlbumWindow::~AlbumWindow()
{
    delete ui;
}

void AlbumWindow::browseAlbum(QString id, QString albumCount)
{
    this->albumId = id;
    this->albumCount = albumCount;
    this->updateAlbum();
}

void AlbumWindow::uploadPhoto()
{
    ImageUploadDialog *dialog = new ImageUploadDialog(this);
#ifdef Q_WS_S60
    dialog->showMaximized();
#else
    dialog->show();
#endif
}

void AlbumWindow::updateAlbum()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    this->reply = qfacebook->getObject(this->albumId + QString("/photos&limit=0"));
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyReceived()));
}

void AlbumWindow::onReplyReceived()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch album!";
        this->reply->deleteLater();
        this->reply = 0;
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
        return;
    }

    foreach (PhotoItem *item, photos) {
        ui->gridLayout->removeWidget(item);
        sourceLinks.clear();
        delete item;
    }

    column = 0;
    row = 0;

    QVariant jsonData = this->reply->data();
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();
    foreach(jsonData, listData) {
        // One image at a time...
        QVariantList imagesList = jsonData.toMap().value("images").toList();
        QString id = jsonData.toMap().value("id").toString();
        QString userId = jsonData.toMap().value("from").toMap().value("id").toString();
        //QString userName = jsonData.toMap().value("from").toMap().value("name").toString();
        QString normalUrl;
        QString smallUrl;
        QString thumbUrl;
        QString mediumUrl;
        QString sourceUrl = jsonData.toMap().value("source").toString();
        foreach (QVariant imageVariant, imagesList) {
            QString imageUrl = imageVariant.toMap().value("source").toString();
            if (imageUrl.endsWith("n.jpg"))
                normalUrl = imageUrl;
            else if (imageUrl.endsWith("s.jpg"))
                smallUrl = imageUrl;
            else if (imageUrl.endsWith("t.jpg"))
                thumbUrl = imageUrl;
            else if (imageUrl.endsWith("a.jpg"))
                mediumUrl = imageUrl;
        }
        sourceLinks.append(normalUrl);
        PhotoItem *item = new PhotoItem(this, accessToken, nam);
        item->setComments(jsonData.toMap().value("comments").toMap().value("data").toList());
        item->setTags(jsonData.toMap().value("tags").toMap().value("data").toList());
        connect(item, SIGNAL(clicked(PhotoItem*)), this, SLOT(onPhotoItemClicked(PhotoItem*)));
        item->setPixmapFromUrl(mediumUrl);
        item->setLargePhotoLink(normalUrl);
        item->setSmallPhotoLink(smallUrl);
        item->setPhotoId(id);
        item->setUserId(userId);
        item->setSourcePhotoLink(sourceUrl);
        if (column > MAX_COLUMNS) {
            column = 0;
            row++;
        }
        ui->gridLayout->addWidget(item, row, column);
        photos.append(item);
        column++;
    }

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif

    reply->deleteLater();
}

void AlbumWindow::onPhotoItemClicked(PhotoItem *item)
{
    PhotoWindow *photo = new PhotoWindow(this, accessToken);
    photo->setPixmapFromUrl(item->largeUrl()            );
    photo->setWindowTitle(tr("Photo"));
    photo->setComments(item->comments());
    photo->setTags(item->tags());
#ifdef Q_WS_S60
    photo->showMaximized();
#else
    photo->show();
#endif
}

void AlbumWindow::downloadAllPhotos()
{
    if (sourceLinks.isEmpty()) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr("Unable to download: album is empty"));
#else
        QMessageBox::warning(this, tr("Error"), tr("Unable to download: album is empty"));
#endif
        return;
    }

    setAttribute(Qt::WA_DeleteOnClose, false);
    m_finishedDownloads = 0;
    m_imagesCount = sourceLinks.count();
    m_downloadLocation = QFileDialog::getExistingDirectory(this, tr("Select folder to save images to"),
                                                           QDir::homePath());
    if (m_downloadLocation.isEmpty()) {
        setAttribute(Qt::WA_DeleteOnClose, true);
        return;
    }

    if (ui->statusBar->isHidden())
        ui->statusBar->show();
    ui->statusBar->showMessage(tr("%1/%2 photos downloaded").arg(m_finishedDownloads, m_imagesCount));

    foreach (QString url, sourceLinks) {
        QNetworkReply *reply = nam->get(QNetworkRequest(url));
        connect(reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
    }
}

void AlbumWindow::onDownloadFinished()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    QString url = reply->url().toString();
    int pos = url.lastIndexOf("/");
    QString baseName = url.remove(0, pos);
    baseName.remove("/");
    QString fileName = m_downloadLocation + "/" + baseName;
    QFile image(fileName, this);
    if (image.open(QFile::WriteOnly)) {
        image.write(reply->readAll());
        image.close();
    }
    m_finishedDownloads++;

    ui->statusBar->showMessage(tr("%1/%2 photos downloaded").arg(m_finishedDownloads, m_imagesCount));

    if (m_finishedDownloads == m_imagesCount) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr("Download complete"));
#else
        QMessageBox::information(this, tr("Success"), tr("Album downloaded"));
#endif
        setAttribute(Qt::WA_DeleteOnClose, true);
        ui->statusBar->hide();
        if (!isVisible())
            this->deleteLater();
        return;
    }
}
