#include "friendswindow.h"
#include "ui_friendswindow.h"

FriendsWindow::FriendsWindow(QWidget *parent, QString accessToken) :
    QMainWindow(parent),
    ui(new Ui::FriendsWindow),
    token(accessToken),
    qfacebook(new QFacebook(accessToken, this)),
    nam(new QNetworkAccessManager())
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
    ui->friendsList->setIconSize(QSize(64, 64));
#endif
    ui->searchWidget->hide();
#ifdef Q_WS_MAEMO_5
    ui->searchHideButton->setIcon(QIcon::fromTheme("general_close"));
#else
    ui->searchHideButton->setText("X");
#endif
    m_isDialog = false;
    m_shouldFetchAvatars = QSettings().value("friends/fetch-avatars", true).toBool();
    connect(nam, SIGNAL(finished(QNetworkReply*)), this, SLOT(onAvatarReplyFinished(QNetworkReply*)));
    connect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(ui->actionRefresh, SIGNAL(triggered()), this, SLOT(updateFriends()));
    connect(ui->searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    connect(ui->searchHideButton, SIGNAL(clicked()), ui->searchWidget, SLOT(hide()));
    connect(ui->searchHideButton, SIGNAL(clicked()), ui->searchEdit, SLOT(clear()));
}

FriendsWindow::~FriendsWindow()
{
    delete ui;
}

void FriendsWindow::updateFriends()
{
    this->browseId(this->idToBrowse);
}

void FriendsWindow::browseId(QString id)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    this->idToBrowse = id;

    if (id == "me" && QSettings().contains("friends/list")) {
        qDebug() << "FriendsWindow: Cache exists, loading from there...";
        m_friendsList = QSettings().value("friends/list");
        this->parseData(m_friendsList);
    }

    this->reply = qfacebook->getConnections(this->idToBrowse, "friends");
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void FriendsWindow::onReplyFinished()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch friends!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }

    QVariant jsonData = this->reply->data();
    if (idToBrowse == "me") {
        if (m_friendsList == jsonData) {
            qDebug() << "FriendsWindow: Friends list cache seems up to date...";
            return;
        } else {
            qDebug() << "FriendsWindow: Either there's no cache, or you gained or lost a friend, updating cache...";
            QSettings().setValue("friends/list", jsonData);
        }
    }

    this->parseData(jsonData);
}

void FriendsWindow::parseData(QVariant jsonData)
{
    ui->friendsList->clear();

    QThread *thread = new QThread;
    FriendParser *parser = new FriendParser(jsonData, m_isDialog, m_shouldFetchAvatars);
    parser->moveToThread(thread);

    connect(thread, SIGNAL(started()), parser, SLOT(start()));
    connect(thread, SIGNAL(finished()), parser, SLOT(deleteLater()));
    connect(thread, SIGNAL(finished()), thread, SLOT(deleteLater()));
    connect(parser, SIGNAL(avatarReceived(QString,QImage)), this, SLOT(onAvatarProcessed(QString,QImage)));
    connect(parser, SIGNAL(itemParsed(QListWidgetItem*)), this, SLOT(onItemParsed(QListWidgetItem*)));

    thread->start();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void FriendsWindow::onItemParsed(QListWidgetItem *item)
{
    ui->friendsList->addItem(item);
}

void FriendsWindow::onAvatarProcessed(QString id, QImage image)
{
    for (int i = 0; i < ui->friendsList->count(); i++) {
        if (ui->friendsList->item(i)->data(Qt::UserRole).toString() == id)
            ui->friendsList->item(i)->setIcon(QIcon(QPixmap::fromImage(image)));
    }
}

void FriendsWindow::onAvatarReplyFinished(QNetworkReply *reply)
{
    QString id = reply->objectName();
    for (int i = 0; i < ui->friendsList->count(); i++) {
        if (ui->friendsList->item(i)->data(Qt::UserRole).toString() == id)
            ui->friendsList->item(i)->setIcon(QIcon(QPixmap::fromImage(QImage::fromData(reply->readAll()))));
    }
}

void FriendsWindow::requestRecepientsList()
{
    ui->friendsList->setSelectionMode(QListWidget::MultiSelection);
    QPushButton *doneButton = new QPushButton(tr("Done"), this);
    disconnect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(doneButton, SIGNAL(clicked()), this, SLOT(onRecepientsSelected()));
    ui->verticalLayout->addWidget(doneButton);
    this->setWindowFlags(Qt::Dialog);
    m_isDialog = true;
    this->browseId("me");
#ifdef Q_WS_S60
    this->showMaximized();
#else
    this->show();
#endif
}

void FriendsWindow::requestFriend()
{
    disconnect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onRecepientsSelected()));
    this->setWindowFlags(Qt::Dialog);
    m_isDialog = true;
    this->browseId("me");
#ifdef Q_WS_S60
    this->showMaximized();
#else
    this->show();
#endif
}

void FriendsWindow::onRecepientsSelected()
{
    emit recepientsSelected(ui->friendsList->selectedItems());
    this->close();
}

void FriendsWindow::onFriendClicked(QListWidgetItem *item)
{
    ProfileWindow *window = new ProfileWindow(this, token);
    window->setAttribute(Qt::WA_DeleteOnClose);
    window->browseProfile(item->data(Qt::UserRole).toString());
    window->setWindowTitle(item->text());
#ifdef Q_WS_S60
    window->showMaximized();
#else
    window->show();
#endif
}

void FriendsWindow::keyReleaseEvent(QKeyEvent *e)
{
    if (e->key() == Qt::Key_Enter || e->key() == Qt::Key_Left || e->key() == Qt::Key_Right || e->key() == Qt::Key_Backspace)
        return;
    else if ((e->key() == Qt::Key_Up || e->key() == Qt::Key_Down) && !ui->searchWidget->isHidden())
        ui->friendsList->setFocus();
    else {
        ui->friendsList->clearSelection();
        if (ui->searchWidget->isHidden())
            ui->searchWidget->show();
        if (!ui->searchEdit->hasFocus())
            ui->searchEdit->setText(ui->searchEdit->text() + e->text());
        ui->searchEdit->setFocus();
    }
}

void FriendsWindow::onSearchTextChanged(QString text)
{
    for (int i=0; i < ui->friendsList->count(); i++) {
        if (ui->friendsList->item(i)->text().toLower().indexOf(text.toLower()) == -1)
            ui->friendsList->item(i)->setHidden(true);
        else
            ui->friendsList->item(i)->setHidden(false);
    }

    if (text.isEmpty())
        ui->searchWidget->hide();
}

Friend::Friend (QWidget *parent, QString id) :
    userId(id)
{
    this->setParent(parent);
}

Friend::~Friend() {}

void Friend::avatarReceived()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    QUrl redir = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        reply->deleteLater();
        QString url = redir.toString();
        int pos = url.lastIndexOf("/");
        QString baseName = url.remove(0, pos);
        baseName.remove("/");

        QFile cachedImage(QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName);
        if (cachedImage.exists()) {
            cachedImage.open(QIODevice::ReadOnly);
            QByteArray data = cachedImage.readAll();
            cachedImage.close();
            emit avatarProcessed(this->userId, QImage::fromData(data));
            return;
        }
        QNetworkAccessManager *nam = qobject_cast<QNetworkAccessManager*>(sender()->parent());
        QNetworkReply *reply = nam->get(QNetworkRequest(redir));

        connect(reply, SIGNAL(finished()), this, SLOT(avatarReceived()));
        return;
    }
    QByteArray data = reply->readAll();
    emit avatarProcessed(this->userId, QImage::fromData(data));

    QString url = reply->url().toString();
    int pos = url.lastIndexOf("/");
    QString baseName = url.remove(0, pos);
    baseName.remove("/");
    QDir tempDir(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));
    if (!tempDir.exists())
        tempDir.mkpath(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));

    QString filename = QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName;
    QFile cachedImage(filename);
    cachedImage.open(QIODevice::WriteOnly);
    cachedImage.write(data);
    cachedImage.close();

    reply->deleteLater();
    this->deleteLater();
}

FriendParser::FriendParser(QVariant jsonData, bool isDialog, bool shouldFetchAvatars)
{
    nam = new QNetworkAccessManager(this);
    m_isDialog = isDialog;
    m_jsonData = jsonData;
    m_shouldFetchAvatars = shouldFetchAvatars;
}

void FriendParser::start()
{
    QVariant jsonData = m_jsonData;
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();
    foreach(jsonData, listData) {
        QListWidgetItem *item = new QListWidgetItem();
        item->setText(jsonData.toMap().value("name").toString());
        item->setData(Qt::UserRole, jsonData.toMap().value("id").toString());
        if ((m_isDialog && m_shouldFetchAvatars) || !m_isDialog) {
            Friend *user = new Friend(0, jsonData.toMap().value("id").toString());
            connect(user, SIGNAL(avatarProcessed(QString,QImage)), this, SLOT(onAvatarProcessed(QString,QImage)));
            QString avatarUrl = QString("http://graph.facebook.com/%1/picture").arg(jsonData.toMap().value("id").toString());
            QNetworkReply *reply = nam->get(QNetworkRequest(avatarUrl));
            connect(reply, SIGNAL(finished()), user, SLOT(avatarReceived()));
        }
        emit itemParsed(item);
    }
}

void FriendParser::onAvatarProcessed(QString id, QImage avatar)
{
    emit avatarReceived(id, avatar);
}
