#!/usr/bin/env python
###########################################################################
# Name: 	         SleepAnalyser
# Description:	 Analyses your movement during your sleep
# Copyright:     George Ruinelli, george@ruinelli.ch
# Last change:  (See file "version")
# Licence:         GPL
###########################################################################


import sys
import random
import time
from time import strftime
import math
import array
import string

from PyQt4.QtCore import * 
from PyQt4.QtGui import * 

import ConfigParser
import os, os.path



import records #record data
import data #own data

#own functions
from functions_general import *
from functions_file import *
from functions_record import *
from functions_alarm import *


#GUIs
from Ui_StartWindow import *
from Ui_RecordWindow import *
from Ui_ViewWindow import *
from Ui_ConfigWindow import *
from Ui_DateWindow import *
from Ui_AlarmTimeWindow import *
from Ui_AskForKeepWindow import *




  
###########################################################################
# Main function
###########################################################################
def main():
    global app,  StartWindow,  RecordWindow,  ViewWindow ,  ConfigWindow,  DateWindow,  AlarmTimeWindow,  AskForKeepWindow
    global config
    global RecordWindowGraph
   
    try:
        if(sys.argv[1]=="debug"): 
            print "Enable Debug mode"
            data.debug=1
    except:
        data.debug=0

    #if not on the N900, a acceleration sensor has to be simulated
    if(os.path.exists("/sys/class/i2c-adapter/i2c-3/3-001d/coord")): data.demo=False	#runs on real N900 hardware
    else: data.demo=True	#runs on other hardware/OS


    if(data.demo==True):
        print "Runs not on a N900 => Use randomizer for acceleration sensor data"
        data.configfolder=data.devconfigfolder
        data.logfolder=data.devlogfolder
    else: data.logfolder=data.defaultlogfolder

    app = QApplication(sys.argv)
    
    config=LoadConfig()
    
    if(data.Ysize==120): #workaround for wrong defaults
        data.Ysize=121
        data.Yscale=1
        data.Trigger=30  
        config.set('main', 'Ysize', str(data.Ysize))      
        config.set('main', 'Yscale', str(data.Yscale))      
        print "changed default parameters for Yscale and Ysize"
    
    
    data.app_path=os.path.dirname(__file__)
    if(data.app_path==""): data.app_path="./"
    else: data.app_path=data.app_path+"/"
    
    StartWindow = frmStartWindow()    
    StartWindow.show()    
    StartWindow.UpdateScreenStart()
    
    # Create subwindows
    RecordWindow = frmRecordWindow(StartWindow)
    ConfigWindow = frmConfigWindow(StartWindow)
    ViewWindow = frmViewWindow(StartWindow)
    AlarmTimeWindow = frmAlarmTimeWindow(StartWindow)
    DateWindow = frmDateWindow(ViewWindow)
    AskForKeepWindow = frmAskForKeepWindow(RecordWindow)
    
    # Hide SubWindows at first
    RecordWindow.hide()
    ViewWindow.hide()
    ConfigWindow.hide()
    AlarmTimeWindow.hide()
    DateWindow.hide()    
    AskForKeepWindow.hide()    
    
    #get user name for in log files
    if(data.user==""):
        EditUserName()

    sys.exit(app.exec_())







def EditUserName():
    text = QInputDialog.getText(None, "SleepAnalyser", "Please enter your name.\n"+ \
     "It will be used in the records file, this makes it easier\n" + \
     "when you want to share your sleep pattern with others.", 0,  data.user)
    if(text != ""):
        data.user=text[0]
        config.set('main', 'user', str(data.user))    
    else:
        pass
    print "User name:",  data.user






###########################################################################
#  Gets called every second,
###########################################################################
def MeasurementTimerLoop(): #(interval: sampling*sample_smoothing*value_smoothing)
    global RecordWindowGraph    
    r=GetSensorData()     
    if(r==1): #new data got added to array
        ProcessData(False)      
        ProcessAlarm()
        
        WriteIntoRecordFile()
        
        GraphLen=records.Index/data.arrXscale[data.Xscale]
        data.GraphStart=max(GraphLen-779, 0)
        
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        RecordWindowGraph.update() #show RecordWindowGraph          
    
        if(records.Index==records.MaxArrayDataLen-1): 
            print "Stopped, array is full"
            QtGui.QMessageBox.warning(RecordWindow, 'Memory is full!', "You reached the end of the recording capacity (24h)!\nThe recording stops now.", QtGui.QMessageBox.Ok)
            RecordWindow.StopMeasurement()
    RecordWindow.UpdateScreen()        

  





###########################################################################
# Start the measurement
###########################################################################
def InitMeasurement():
    global app,  RecordWindowGraph

    data.sample_index=0

    records.User=data.user
    records.SWversion=data.version
    records.Sampling_interval=data.Sampling_interval #todo: change names to new system
    records.Sample_smoothing=data.sample_smoothing
    records.Value_smoothing=data.Value_smoothing
    records.Trigger=data.Trigger #save Trigger level on start
    records.statusID="OK"

    PrintRecordData()
    
    #go offline if requested
    if(data.QuickMode==False):
        data.interval=data.Sampling_interval #normal
        if(data.OfflineMode=="True"): #change to offline mode
            try:
                cmd="dbus-send --system --dest=com.nokia.mce --type=method_call /com/nokia/mce/request com.nokia.mce.request.req_device_mode_change string:\"offline\""
                os.popen(cmd)
            except:
                pass
        data.Xscale=1 #set to default zoom level
    else:  
        data.interval=10 #Fast mode for testing
        data.Xscale=0 #set to max zoom level
    

    data.run=2 #change to recording
    app.timer_GetData = QtCore.QTimer()
    QtCore.QObject.connect(app.timer_GetData, QtCore.SIGNAL("timeout()"), MeasurementTimerLoop)     

    data.StartTime=time.localtime()
    data.StartTimeSec=Tuple2Seconds(data.StartTime)
   
    if(data.debug==False): app.timer_GetData.start(data.interval)   
    else: app.timer_GetData.start(1)    #run very fast for testing
    
    CreateRecordFile(data.interval)
   
    data.ChangesInGraph=True #Make sure it only gets refreshed once    
    RecordWindowGraph.update() #show RecordWindowGraph        
    RecordWindow.UpdateScreen()

    print "Start timers (Interval:", data.interval ,"ms)..."
    print "Sampling interval:", data.interval, "ms"
    print "sample_smoothing:", data.sample_smoothing, "x"
    print "value_smoothing:", data.Value_smoothing, "x"
    print "=> log interval:", data.interval*data.Value_smoothing, "ms"
    print "Derivation:", data.Derivation, "(substracts value", data.Derivation, "- value 0)"  
    
    
    
    
    
    

def StopMeasurement():
    global app
    if(data.run==2): #in recording
        app.timer_GetData.stop()
        StopLogfile() 
        UpdateFileHeader() #rewrite log file (only do when recording is stopped!)

    data.run=0
    DeleteAlarm() #delete Alarm if existing
    
    if(data.QuickMode==False): #normal mode
        if(data.OfflineMode=="True" and data.demo!=1): #change to offline mode
            try:
                cmd="dbus-send --system --dest=com.nokia.mce --type=method_call /com/nokia/mce/request com.nokia.mce.request.req_device_mode_change string:\"normal\""
                os.popen(cmd)
            except:
                pass






############################################################################
## Stretch graph to show more details
############################################################################
def Zoom(direction):
    if(direction==+1): #zoom in
        if(data.Xscale==0): return
        ViewWindow.ui.bZoomOut.setEnabled(True)
        data.Xscale=(data.Xscale-1) % data.arrXscaleLen
    else: # -1, zoom out
        if(data.Xscale==data.arrXscaleLen-1): return
        ViewWindow.ui.bZoomIn.setEnabled(True)
        data.Xscale=(data.Xscale+1) % data.arrXscaleLen




    
    
###########################################################################
# Raise or lower the Trigger in the graph + sleep phases
###########################################################################
def MoveTrigger(direction):
    if(direction==+1): #up
        data.Trigger=data.Trigger+1
        if(data.Trigger>data.maxTrigger): data.Trigger=data.maxTrigger      
    else: #-1, down
        data.Trigger=data.Trigger-1
        if(data.Trigger<0): data.Trigger=0
    if(data.run==True): records.Trigger=data.Trigger




def About(ui):
    QtGui.QMessageBox.about(ui, 'About SleepAnalyser', "Copyright 2010 by George Ruinelli\n"+\
    "Version: "+str(data.version)+"-"+str(data.build)+"\n"+\
    "Contact: george@ruinelli.ch\nWebsite: https://garage.maemo.org/projects/sleepanalyser/\n"+\
    "\nFeedback and donations are very welcome!")




###########################################################################
#  Class for Main Window
###########################################################################
class frmStartWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # This attribute makes the whole Stacked Window thing work
        self.ui = Ui_StartWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bCollecting, QtCore.SIGNAL("clicked()"), self.StartMeasurementNormal)
        QtCore.QObject.connect(self.ui.bQuick, QtCore.SIGNAL("clicked()"), self.StartMeasurementQuick)
        QtCore.QObject.connect(self.ui.bLog, QtCore.SIGNAL("clicked()"), self.ShowViewWindow)
        QtCore.QObject.connect(self.ui.bConfig, QtCore.SIGNAL("clicked()"), self.ShowConfigWindow)
        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        
        file = open(data.app_path+"Info.htm", 'r')
        txt=""
        for line in file: txt=txt+line+"\n"
        file.close()
        self.ui.textBrowser.setHtml(txt)
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))
            self.ui.bQuick.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bCollecting.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bLog.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
            self.ui.bConfig.setIcon(QIcon(data.app_path+"img/"+"configure.png"))





    ###########################################################################
    # Show the collectring window (hera the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementNormal(self):
        print "Measure in normal Mode"
        data.QuickMode=False #run in normal mode
        self.StartMeasurement()
        
        
        
        
        
    ###########################################################################
    # Show the collectring window (hera the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementQuick(self):
        print "Measure in QuickMode"
        data.QuickMode=True #run in quick mode (for testing only)
        data.Xscale=0 #default zoom level
        self.StartMeasurement()
        
        
    def StartMeasurement(self):
        global RecordWindowGraph
        RecordWindow.show()
        RecordWindowGraph = RecordWindowCanvas(RecordWindow)
        InitRecord() #clear all old data
        InitMeasurement()
        RecordWindow.UpdateScreenStart()
        RecordWindowGraph.show()
        
        
    ###########################################################################
    # Show the log window (where the data from a file gets visualised)
    ###########################################################################
    def ShowViewWindow(self):
        global ViewWindowGraph        
        
        #reset to default log folder
        if(data.demo==True): data.logfolder=data.devlogfolder #folder on devel PC
        else: data.logfolder=data.defaultlogfolder
        LoadFileList()
        if os.path.exists(data.logfolder+data.lastlogfile):
            data.logfilename=data.lastlogfile
        else:
            l=len(data.arrLogFiles)
            if(l>0):
                data.logfilename=data.arrLogFiles[l-1] #open last (youngest) log file
            else: #no logfiles in logfolder, show "open file" dialog
                tmp=ViewWindow.ShowFileDialog()
                if(not os.path.exists(tmp)): #only continue if valid log file selected
                    StartWindow.show()
                    ViewWindow.hide()
                    return #only continue if valid log file selected
                tmp2=tmp.split("/")
                l=len(tmp2)    
                data.logfilename=tmp2[l-1]
                data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
                print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        
        ViewWindowGraph = ViewWindowCanvas(self)
        ViewWindow.ShowLogFile()
        
        if(data.logfilename!=""):
            ViewWindow.show()
        else:
            print "No valid file selected"
        
        
        
        
        
        
    def ShowConfigWindow(self):
        ConfigWindow.UpdateScreen()
        ConfigWindow.show()



        
    def SetAlarm(self):
        AlarmTimeWindow.UpdateScreen()
        AlarmTimeWindow.show()




    def UpdateScreenStart(self):
        try:   
            print "Application path:",  data.app_path
            file = open(data.app_path+"version", 'r')
            data.version = file.readline()
            data.version=data.version[:-1]
            data.build = file.readline()
            print "Application version:",  data.version,  "Build:",  data.build
        except:
            print "Version file not found"
        self.ui.lblVersion.setText("Version: "+data.version+"-"+data.build)


    def About(self):
        About(self)
        
        

    def closeEvent(self, event):
        global config
        WriteConfig()

















###########################################################################
#  Class for Collecting Window
###########################################################################
class frmRecordWindow(QMainWindow):
    def __init__(self, parent=None):        
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_RecordWindow()
        self.ui.setupUi(self)
        
        QtCore.QObject.connect(self.ui.bStop, QtCore.SIGNAL("clicked()"), self.StopMeasurement)
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            
            
            
            

    def UpdateScreenStart(self):
        t=strftime("%d. %b. %Y %H:%M:%S", data.StartTime)
        self.ui.lblStarttime.setText(t)
        self.ui.lblElapsedTime.setText("")
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        self.ui.bStop.setIcon(QIcon(data.app_path+"img/"+"krec2.png"))
        self.ui.bStop.setText("Stop")

        if(records.AlarmIsSet==True): #alarm active
            m=str(data.AlarmMinute)
            if(len(m)==1): m="0"+m
            self.ui.lblAlarmTime.setText(str(data.AlarmHour)+":"+m+" or up to "+str(data.AlarmTimeWindow)+" minutes before that")
        else: #Do only recording, no alarm
            self.ui.lblAlarmTime.setText("(No Alarm set)")
                        
        self.UpdateScreen()
        
        
        
        
        
        
    def UpdateScreen(self):
        if(data.run==2): #recording
            t=time.mktime(time.localtime())
            time_difference=t-data.StartTimeSec
            d=time.strftime("%H:%M:%S", time.gmtime(time_difference))
            self.ui.lblElapsedTime.setText(d + " ("+str(records.Index)+" entries)")
        
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)


        
        
#    def UpdateScreenStop(self):
#        self.ui.bStop.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
#        self.ui.bStop.setText("Start")
        
        
        
        
    def StopMeasurement(self):
            self.closeEvent(0)






    def ZoomIn(self):
        Zoom(+1)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()
        


    def ZoomOut(self):
        Zoom(-1)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()



    def RaiseTrigger(self):   
        MoveTrigger(+1)
        self.ui.bLowerTrigger.setEnabled(True)
        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update() 
        self.UpdateScreen()
        
        
        
    def LowerTrigger(self):     
        MoveTrigger(-1)
        self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()



    def closeEvent(self, event):
        global AskForKeepWindow,  StartWindow    
#        close=True
        if(data.QuickMode==False ): #running in normal mode
            try:
                event.ignore()  #ignore closing
            except:
                pass
            AskForKeepWindow.UpdateScreen()
            AskForKeepWindow.show()
#        else: 
#            pass #close
        else: #quick mode, delete file anyway and close
            print "Delete logfile: "+data.logfilename
            os.remove(data.logfolder+data.logfilename)
            StartWindow.show()
            self.hide()
            


    def EditNote(self):
        r=QInputDialog.getText(None,  "Add Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            records.Note=str(r[0])
            self.ui.lblNote.setText(str(records.Note))
            if(data.run==0): UpdateFileHeader() #rewrite log file (only do when recording is stopped!)



    def About(self):
        About(self)











###########################################################################
# Class for Log Window
###########################################################################
class frmViewWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_ViewWindow()
        self.ui.setupUi(self)
        
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bNext, QtCore.SIGNAL("clicked()"), self.NextLogFile)
        QtCore.QObject.connect(self.ui.bLast, QtCore.SIGNAL("clicked()"), self.LastLogFile)
        QtCore.QObject.connect(self.ui.bDelete, QtCore.SIGNAL("clicked()"), self.DeleteLogFile)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        QtCore.QObject.connect(self.ui.bOpenDate, QtCore.SIGNAL("clicked()"), self.OpenDate)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
        QtCore.QObject.connect(self.ui.bLeft, QtCore.SIGNAL("clicked()"), self.goLeft)
        QtCore.QObject.connect(self.ui.bRight, QtCore.SIGNAL("clicked()"), self.goRight)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionShowRecordData, QtCore.SIGNAL("triggered()"), self.ShowRecordData)
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bLast.setIcon(QIcon(data.app_path+"img/"+"arrow-left-double.png"))
            self.ui.bNext.setIcon(QIcon(data.app_path+"img/"+"arrow-right-double.png"))
            self.ui.bDelete.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"document-open-folder.png"))
            self.ui.bOpenDate.setIcon(QIcon(data.app_path+"img/"+"date.png"))
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bLeft.setIcon(QIcon(data.app_path+"img/"+"go-previous.png"))
            self.ui.bRight.setIcon(QIcon(data.app_path+"img/"+"go-next.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))

    def UpdateScreen(self):
#        shortfolder=string.replace(data.logfolder, data.homedir, "~/")
#        shortfolder=string.replace(shortfolder, "//", "/") #workaround
#        self.ui.lblLogfile.setText(data.logfilename+" (in "+shortfolder+")")
        self.ui.lblLogfile.setText(data.logfilename)

        d0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.StartTime_tuple)
        EndTime_tuple=Seconds2Tuple(Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000)
        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", EndTime_tuple)):
            d1=strftime("%H:%M:%S", EndTime_tuple) #on same day, only show date once
        else:
            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", EndTime_tuple) 
        
        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
        diff_tuple=Seconds2Tuple(diff_sec)
        diff=strftime("%H:%M:%S", EndTime_tuple)
        if(diff_sec>86400): d="1 day, " #longer than 24h
        else: d=""
        self.ui.lblTimes.setText(str(d0)+"  -  "+str(d1)+"  (Duration: "+d+str(diff)+")")

        self.ui.lblNote.setText(str(records.Note))

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)
        
        if(len(data.arrLogFiles)<2): #disable last/next buttons if only one file in folder
            self.ui.bNext.setEnabled(False)
            self.ui.bLast.setEnabled(False)
        else:
            self.ui.bNext.setEnabled(True)
            self.ui.bLast.setEnabled(True)

        ScrollbarStartPercent,ScrollbarEndPercent= ScrollbarData()
        ScrollbarStartPercent=round(ScrollbarStartPercent,  0)
        ScrollbarEndPercent=round(ScrollbarEndPercent,  0)
#        print "UpdateScreen,  Scrollbar percent:",  ScrollbarStartPercent,  ScrollbarEndPercent
        if(ScrollbarStartPercent>0): self.ui.bLeft.setEnabled(True)
        else: self.ui.bLeft.setEnabled(False)
        if(ScrollbarEndPercent<100): self.ui.bRight.setEnabled(True)
        else: self.ui.bRight.setEnabled(False)
    
    
    
            
    #ToDo: do not show record with useless data 
    #Todo Show text in graph instead message box
    def ShowLogFile(self):
        r=LoadRecord()    
        if(r==1): #file too long            
            QtGui.QMessageBox.warning(self, 'Record too long!', "The record is too long. The end will be cut off!", QtGui.QMessageBox.Ok)
        elif(r==2): #file does not exist
            self.OpenRecord()
            return
#        elif(r==3): #file is corrupted => draw empty graph
#            QtGui.QMessageBox.warning(self, 'File corrupded!', "The data in the record file\n"+data.logfilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==8): #file header is corrupted
#            QtGui.QMessageBox.warning(self, 'File header corrupted!', "The header data in the record file\n"+data.logfilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==4): #file is empty
#            QtGui.QMessageBox.warning(self, 'File empty!', "The data in the record file\n"+data.logfilename+"\ncontains no data!", QtGui.QMessageBox.Ok)
        elif(r==9): #Record too old
            QtGui.QMessageBox.warning(self, 'File version too old!', "Can not load record \n"+data.logfilename+"\nas it got created from SleepAnalyser version <1.6 and is not compatible with the current SleepAnalyser version.", QtGui.QMessageBox.Ok)
        LoadFileList()
        ProcessData(True)
        data.Xscale=1 #default zoom level
        
        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
        
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        ViewWindowGraph = ViewWindowCanvas(self)
        ViewWindowGraph.show()
        self.UpdateScreen()
        
        
        
    def NextLogFile(self):
        
        data.arrLogFileIndex=(data.arrLogFileIndex+1)  % len(data.arrLogFiles)
        data.logfilename=data.arrLogFiles[data.arrLogFileIndex]
        self.ShowLogFile()


    def LastLogFile(self):
        data.arrLogFileIndex=(data.arrLogFileIndex-1)  % len(data.arrLogFiles)
        data.logfilename=data.arrLogFiles[data.arrLogFileIndex]
        self.ShowLogFile()


        
    def ZoomIn(self):
        Zoom(+1)
        self.ui.bZoomOut.setEnabled(True)
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        
#        print "records.LastIndex:",  records.LastIndex, data.arrXscale[data.Xscale]*775
        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
#        print "GraphStart:",  data.GraphStart
        
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()




    def ZoomOut(self):
        Zoom(-1)
        self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)
        
        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
        
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()
        
        

    def RaiseTrigger(self):   
        MoveTrigger(+1)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update() 
        self.UpdateScreen()
        
    def LowerTrigger(self):     
        MoveTrigger(-1)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()
    
        
        
        
    def goLeft(self):
        data.GraphStart=max(data.GraphStart-600, 0)
        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()
        
        
    def goRight(self):
        print "records.LastIndex:",  records.LastIndex, data.arrXscale[data.Xscale]*775
        MaxGraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
        data.GraphStart=min(MaxGraphStart,  data.GraphStart+600)
        print "goRight MaxGraphStart:",  MaxGraphStart
        print "GraphStart:",  data.GraphStart

        data.ChangesInGraph=True #Make sure it only gets refreshed once
        self.update()
        self.UpdateScreen()
        
        

    def DeleteLogFile(self):
        reply = QtGui.QMessageBox.question(self, 'Delete Record?', "Are you sure you want to delete this record:\n"+data.logfilename, QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)
        if reply == QtGui.QMessageBox.Yes:
            print "Delete logfile: "+data.logfilename
            os.remove(data.logfolder+data.logfilename)    
            
            LoadFileList()
            l=len(data.arrLogFiles)
            if(l>0):
                data.logfilename=data.arrLogFiles[l-1] #open last (youngest) log file
            else: #no logfiles in logfolder, show "open file" dialog
                tmp=self.ShowFileDialog()
                if(tmp==""): #only continue if valid log file selected
                    self.closeEvent("") #no file selected, close
                    return
                tmp2=tmp.split("/")
                l=len(tmp2)    
                data.logfilename=tmp2[l-1]
                data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
                print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
            self.ShowLogFile()
            self.UpdateScreen()



    def OpenRecord(self):
        tmp=self.ShowFileDialog()
        if(not os.path.exists(tmp)): #only continue if valid log file selected
            return 
        tmp2=tmp.split("/")
        l=len(tmp2)    
        data.logfilename=tmp2[l-1]
        data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
        print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        self.ShowLogFile()
    
    
    
    
    def ShowFileDialog(self):        
        data.logfolder=string.replace(data.logfolder, "~",  data.homedir) #workaround
        print "Open file dialog in:",  data.logfolder
        tmp= QFileDialog.getOpenFileName(None,"Open SleepAnalyser Log File", data.logfolder, "Log Files (*.csv)");
        print "Selected file:",  tmp
        return tmp
        
    def OpenDate(self):
        DateWindow.show()
        DateWindow.DateChanged()


    def closeEvent(self, event):
        ViewWindow.close()
        StartWindow.show()
        

    def EditNote(self):
        r=QInputDialog.getText(None,  "Edit Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            records.Note=str(r[0])
            self.ui.lblNote.setText(str(records.Note))
            UpdateFileHeader() #rewrite log file (only do when recording is stopped!)



    def About(self):
        About(self)


    def ShowRecordData(self):
        shortfolder=string.replace(data.logfolder, data.homedir, "~/")
        shortfolder=string.replace(shortfolder, "//", "/") #workaround
        
        d0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.StartTime_tuple)
        EndTime_tuple=Seconds2Tuple(Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000)
        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", EndTime_tuple)):
            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", EndTime_tuple) #on same day, only show date once
        else:
            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", EndTime_tuple) 
        
        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
        diff_tuple=Seconds2Tuple(diff_sec)
        diff=strftime("%H:%M:%S", EndTime_tuple)
        if(diff_sec>86400): d="1 day, " #longer than 24h
        else: d=""
        
        if(records.AlarmIsSet==True):
            alarm="Alarm Time Window: "+strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowStart_tuple)+" - "+ \
            strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowEnd_tuple)+"\n" +\
            "Alarm went off on: "+strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTime_tuple)+"\n"
        else:
            alarm="Alarm: No alarm was set\n"
        
        QtGui.QMessageBox.information(self, 'About this record', "File: "+str(data.logfilename+"\n") + \
        "Folder: "+str(shortfolder) +"\n" + \
        "Start: "+d0 + "\n" + \
        "End: "+d1 + "\n" + \
        "Duration: "+diff + ", " + \
        "Total "+str(records.Index)+" entries\n" +\
        "Note: "+str(records.Note)+"\n" +\
        alarm +\
        "Condition of record: "+str(records.status)+"\n" + \
        "Sampling interval: "+str(records.Sampling_interval) + ", " + \
        "Sampling smoothing: "+str(records.Sample_smoothing) + ", " + \
        "Value smoothing: "+str(records.Value_smoothing) + ", " + \
        "Trigger level: "+str(records.Trigger) + "\n" + \
        "Created with SleepAnalyser version "+str(records.SWversion))
        








###########################################################################
# Class for Configuration Window
###########################################################################
class frmConfigWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_ConfigWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
              
        QtCore.QObject.connect(self.ui.bEditUserName, QtCore.SIGNAL("clicked()"), self.EditUserName)
        QtCore.QObject.connect(self.ui.SlrDelayStart,SIGNAL("valueChanged(int)"),self.SliderMoved)
        
        

    def UpdateScreen(self):
        if(data.OfflineMode=="True"): self.ui.cOfflineMode.setChecked(True)
        else: self.ui.cOfflineMode.setChecked(False)        
        self.ui.lblUserName.setText(str(data.user))
        self.ui.lblDelayStart.setText(str(data.DelayStartTime)+" sec")
        self.ui.SlrDelayStart.setValue(int(data.DelayStartTime))


    def closeEvent(self, event):        
        if(self.ui.cOfflineMode.isChecked()==True):          
                data.OfflineMode="True"
        else: 
            data.OfflineMode="False"
        config.set('main', 'OfflineMode', str(data.OfflineMode))   
        config.set('main', 'user', str(data.user))      
        config.set('main', 'DelayStartTime', str(data.DelayStartTime))      
        
        ConfigWindow.close()
        StartWindow.show()


    def EditUserName(self):
        EditUserName()
        self.ui.lblUserName.setText(str(data.user))
        
        
    def SliderMoved(self):
        x=self.ui.SlrDelayStart.value()
        data.DelayStartTime=x
        self.ui.lblDelayStart.setText(str(data.DelayStartTime)+" sec")








###########################################################################
# Class for Date Window
###########################################################################
class frmDateWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_DateWindow()
        self.ui.setupUi(self)
#        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))

        QtCore.QObject.connect(self.ui.cal, QtCore.SIGNAL('selectionChanged()'), self.DateChanged)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        
        self.ui.cal.showToday()




    def DateChanged(self):
        date = self.ui.cal.selectedDate()
        date=str(date.toPyDate())

        print "Load list of all csv files in folder:",  data.logfolder + " with date " + date
        output = os.popen("cd "+str(data.logfolder)+"; ls -C1 *"+date+"*.csv").read() # run ls and make the output look like a file I can read
        arrLogFiles=output.split("\n")
        arrLogFiles.pop()

        amount=len(arrLogFiles)
        
        self.ui.lblResult.setText("Found "+str(amount)+" Records for this date.")
        if(amount>0): 
            self.ui.bOpen.setEnabled(True)
            data.LogFileSelectedDate=arrLogFiles[0] #select first file of that date for opening
        else: self.ui.bOpen.setEnabled(False)


    def OpenRecord(self):
        data.logfilename=data.LogFileSelectedDate
        print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        DateWindow.hide()
        ViewWindow.ShowLogFile()

    def closeEvent(self, event):        
        pass








###########################################################################
# Class for Setting the alarm time Window
###########################################################################
class frmAlarmTimeWindow(QMainWindow):
    def __init__(self, parent=None):        
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_AlarmTimeWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))

        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
                
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))

    
        

    def UpdateScreen(self):
        if(data.AlarmHour==-1 or data.AlarmMinute==-1):
            data.AlarmHour=7
            data.AlarmMinute=0
        print "Alarm time:",  data.AlarmHour, ":",  data.AlarmMinute
        self.ui.lstAlarmHour.setCurrentRow(int(data.AlarmHour))
        self.ui.lstAlarmMinute.setCurrentRow(int(data.AlarmMinute)/5)
        self.ui.lstAlarmTimeWindow.setCurrentRow(int(data.AlarmTimeWindow)/5-1)
        s="The Alarm will then go off sometimes during the time window (as soon as you move more), but latest at the set alarm time."
        self.ui.lblNote.setText(s)




    def SetAlarm(self):
        global RecordWindowGraph
        
        InitRecord() #clear all old data
        
        print "Set Alarm"
        data.QuickMode=False #run in normal mode        
        
        RecordWindowGraph = RecordWindowCanvas(RecordWindow)
                
        try:
            data.AlarmHour=int(self.ui.lstAlarmHour.currentItem().text())
            data.AlarmMinute=int(self.ui.lstAlarmMinute.currentItem().text())
            data.AlarmTimeWindow=int(self.ui.lstAlarmTimeWindow.currentItem().text())
            config.set('main', 'AlarmHour', str(data.AlarmHour))  
            config.set('main', 'AlarmMinute', str(data.AlarmMinute))  
            config.set('main', 'AlarmTimeWindow', str(data.AlarmTimeWindow))  
            
            now=int(strftime("%H", time.localtime()))*3600+int(strftime("%M", time.localtime()))*60
            al=int(data.AlarmHour)*3600+int(data.AlarmMinute)*60
            alarmdiff=al-now
            if(alarmdiff<0):  #set to tomorrow if time already is elapsed today
                alarmdiff=alarmdiff+86400

            records.AlarmTime_seconds=int((records.StartTime_seconds+alarmdiff)/60)*60 #set seconds to zero
            records.AlarmTime_tuple=Seconds2Tuple(records.AlarmTime_seconds)           
            
            records.AlarmTimeWindowEnd_tuple=records.AlarmTime_tuple
            records.AlarmTimeWindowEnd_seconds=records.AlarmTime_seconds
            
            records.AlarmTimeWindow_minutes=int(self.ui.lstAlarmTimeWindow.currentItem().text())
            
            records.AlarmTimeWindowStart_seconds=records.AlarmTimeWindowEnd_seconds-records.AlarmTimeWindow_minutes*60
            records.AlarmTimeWindowStart_tuple=Seconds2Tuple(records.AlarmTimeWindowStart_seconds)            
            
            print records.AlarmTime_tuple,  records.AlarmTimeWindow_minutes
            print records.AlarmTimeWindowStart_tuple
            print records.AlarmTimeWindowEnd_tuple
        
        except: #If nothing selected (workaround) 
            print "Selected time not valid!"           
            print sys.exc_info()  
            StartWindow.show()
            AlarmTimeWindow.hide()
            return

        DeleteAlarm() #delete old alarm if existing
        records.AlarmIsSet=True
        SetAlarm() #set alarm for the last possible time
        InitMeasurement()
        
        RecordWindow.show()
        AlarmTimeWindow.hide()
        RecordWindow.UpdateScreenStart()
        RecordWindowGraph.show()






    def closeEvent(self, event):        
        pass











###########################################################################
# Class for Setting the alarm time Window
###########################################################################
class frmAskForKeepWindow(QMainWindow):
    def __init__(self, parent=None):        
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_AskForKeepWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))

        QtCore.QObject.connect(self.ui.bstop_keepRecord, QtCore.SIGNAL("clicked()"), self.KeepRecord)
        QtCore.QObject.connect(self.ui.bstop_removeRecord, QtCore.SIGNAL("clicked()"), self.RemoveRecord)
        QtCore.QObject.connect(self.ui.bcontinue, QtCore.SIGNAL("clicked()"), self.ContinueRecording)
                
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bstop_keepRecord.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
            self.ui.bstop_removeRecord.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bcontinue.setIcon(QIcon(data.app_path+"img/"+"krec.png"))


    def UpdateScreen(self):
        s="Do you really want to stop recording?"
        if(records.AlarmIsSet==True): s=s+"\nThis will also remove the set alarm!"
        s=s+"\n\nAnd do you want to keep the record?"
        self.ui.lblNote.setText(s)
        

    def KeepRecord(self):
        global ViewWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        data.lastlogfile=data.logfilename #set to corrent file for viewwindow
        config.set('main', 'lastlogfile', str(data.lastlogfile))              
        RecordWindow.hide()
        print "Show record in ViewWindow:",  data.lastlogfile
        StartWindow.ShowViewWindow()
        self.hide()



    def RemoveRecord(self): #removed record
        global StartWindow,  RecordWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        print "Delete logfile: "+data.logfilename
        os.remove(data.logfolder+data.logfilename)
        RecordWindow.hide()
        StartWindow.show()
        self.hide()



    def ContinueRecording(self):
        self.hide()
        return

    def closeEvent(self, event):        
        event.ignore() #do not close












###########################################################################
# Class for drawing the RecordWindowGraph
###########################################################################
class RecordWindowCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)    
        self.setGeometry(9, 180, 784, data.Ysize+55)
    
    def paintEvent(self, event):
        showGraph(self,  QtGui)



    
###########################################################################
# Class for drawing the ViewWindowGraph
###########################################################################
class ViewWindowCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)    
        self.setGeometry(9, 100, 784, data.Ysize+55)
    
    def paintEvent(self, event):
        showGraph(self,  QtGui)
       
    
###########################################################################
main() # Now we can start it.
