/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Ville Tiensuu - ville.tiensuu@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtTest/QtTest>

#include "map/baselocationitem.h"
#include "map/friendlocationitem.h"
#include "map/mapcommon.h"

namespace Testfriendlocation  //  Test data for function is defined in namespace
{
    const QPointF defaultPosition(UNDEFINED, UNDEFINED);
    const QString userID("ABC123");
    const QUrl url("http://www.test.com/test.png");
    const int itemIgnoresTransformationsFlagValue = 0x20;
}

using namespace Testfriendlocation;

/**
  * @brief Class that test FriendLocationItem
  *
  * @author Ville Tiensuu - ville.tiensuu@ixonos.com
  * @author Sami Rämö - sami.ramo@ixonos.com
  */
class TestFriendLocationItem: public QObject
{
    Q_OBJECT

private slots:
    /**
      * @brief Test method for constructors.
      *
      * Creates instance of FriendLocationItem
      * Tests that Z-values are set correctly.
      * Tests that ItemIgnoresTransformations flag is set.
      * Tests that default position is set.
      * Tests that user ID is set
      */
    void constructor();

    /**
      * @brief Test grouping functionality
      */
    void grouping();

    /**
      * @brief Tests that friend location item is possible to set visible and unvisible
      */
    void setProfileImage();
};

void TestFriendLocationItem::constructor()
{
    FriendLocationItem friendLocation(userID);

    // Test Z-value
    QCOMPARE(static_cast<int>(friendLocation.zValue()), static_cast<int>(FriendLocationItemZValue));

    // Test ItemIgnoresTransformations Flags
    QGraphicsItem::GraphicsItemFlags friendLocationItemFlags = friendLocation.flags();
    QCOMPARE(friendLocationItemFlags, itemIgnoresTransformationsFlagValue);

    // test default position
    QCOMPARE(friendLocation.pos(), defaultPosition);

    // test user ID
    QCOMPARE(friendLocation.userId(), userID);

    // test isPartOfGroup default value
    QCOMPARE(friendLocation.isPartOfGroup(), false);
}

void TestFriendLocationItem::grouping()
{
    FriendLocationItem friendLocation(userID);

    // test defaults
    QCOMPARE(friendLocation.isPartOfGroup(), false);
    QCOMPARE(friendLocation.isVisible(), true);

    // group item
    friendLocation.setPartOfGroup(true);

    // flag should be set and item should be hidden
    QCOMPARE(friendLocation.isPartOfGroup(), true);
    QCOMPARE(friendLocation.isVisible(), false);

    // back to ungrouped state
    friendLocation.setPartOfGroup(false);

    // test
    QCOMPARE(friendLocation.isPartOfGroup(), false);
    QCOMPARE(friendLocation.isVisible(), true);
}

void TestFriendLocationItem::setProfileImage()
{
    FriendLocationItem friendLocation(userID);

    //default pixmap and URL should be empty
    QCOMPARE(friendLocation.pixmap().isNull(), true);
    QCOMPARE(friendLocation.profileImageUrl().isEmpty(), true);

    QPixmap friendIcon("situare_user.gif");
    friendLocation.setProfileImage(friendIcon, url);

    // picture should be set
    QCOMPARE(friendLocation.pixmap().isNull(), false);

    // Test Offset
    QCOMPARE(friendLocation.offset(),
             QPointF(-friendIcon.width() / 2, -friendIcon.height() / 2));

    // test url
    QCOMPARE(friendLocation.profileImageUrl(), url);
}

QTEST_MAIN(TestFriendLocationItem)
#include "testfriendlocationitem.moc"
