/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jukka Saastamoinen - jukka.saastamoinen@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Katri Kaikkonen - katri.kaikkonen@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QFormLayout>
#include <QLabel>
#include <QMouseEvent>
#include <QPainter>
#include <QSettings>
#include <QVBoxLayout>

#include "common.h"
#include "imagebutton.h"
#include "textmodifier.h"
#include "user/user.h"

#include "userinfo.h"

const int BACKGROUND_BOTTOM_HEIGHT = 15;
const int BACKGROUND_TOP_HEIGHT = 20;
const int BACKGROUND_WIDTH = 368;
const int ICON_HEIGHT = 24;
const int ICON_WIDTH = 24;
const int MARGIN = 5;
const int LABEL_MAX_WIDTH = BACKGROUND_WIDTH - ICON_WIDTH - 5 * MARGIN;

UserInfo::UserInfo(QWidget *parent)
    : QWidget(parent),
      m_expanded(false),
      m_updateLocation(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    QVBoxLayout *verticalLayout = new QVBoxLayout(this);
    verticalLayout->setContentsMargins(MARGIN * 2, 0, MARGIN * 2, MARGIN * 2);
    verticalLayout->setSpacing(0);
    setLayout(verticalLayout);

    QFormLayout *infoLayout = new QFormLayout();
    infoLayout->setMargin(0);
    infoLayout->setSpacing(0);

    QLabel *envelopeLabel = new QLabel();
    envelopeLabel->setPixmap(QPixmap(":/res/images/envelope.png"));
    envelopeLabel->setContentsMargins(0, 0, MARGIN, 0);
    envelopeLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *compassLabel = new QLabel();
    compassLabel->setPixmap(QPixmap(":/res/images/compass.png"));
    compassLabel->setContentsMargins(0, 0, MARGIN, 0);
    compassLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *clockLabel = new QLabel();
    clockLabel->setPixmap(QPixmap(":/res/images/clock.png"));
    clockLabel->setContentsMargins(0, 0, MARGIN, 0);
    clockLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);

    m_avatar = new ImageButton();

    m_nameLabel = new QLabel();

    m_statusTextLabel = new QLabel();
    m_statusTextLabel->setWordWrap(true);
    m_statusTextLabel->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
    m_locationLabel = new QLabel();
    m_locationLabel->setWordWrap(true);
    m_locationLabel->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
    m_updatedLabel = new QLabel();
    m_updatedLabel->setWordWrap(true);
    m_updatedLabel->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);

    infoLayout->addRow(envelopeLabel, m_statusTextLabel);
    infoLayout->addRow(compassLabel, m_locationLabel);
    infoLayout->addRow(clockLabel, m_updatedLabel);

    verticalLayout->addWidget(m_avatar, 0, Qt::AlignHCenter);
    verticalLayout->addWidget(m_nameLabel, 0, Qt::AlignHCenter);
    verticalLayout->addLayout(infoLayout);

    connect(m_avatar, SIGNAL(clicked()),
            this, SLOT(findButtonClicked()));

    setFixedWidth(BACKGROUND_WIDTH);

    this->setFont(NOKIA_FONT_SMALL);
    m_nameLabel->setFont(NOKIA_FONT_NORMAL);
    QPalette itemPalette = palette();
    itemPalette.setColor(QPalette::Foreground, COLOR_GRAY);
    setPalette(itemPalette);
    QPalette namePalette = m_nameLabel->palette();
    namePalette.setColor(QPalette::Foreground, Qt::white);
    m_nameLabel->setPalette(namePalette);

    m_backgroundTopImage.load(":/res/images/list_item_top.png");
    m_backgroundMiddleImage.load(":/res/images/list_item_middle.png");
    m_backgroundBottomImage.load(":/res/images/list_item_bottom.png");

    restoreUnsendMessage();
}

UserInfo::~UserInfo()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    if (!m_backupMessage.isEmpty()) {
        settings.setValue(USER_UNSEND_MESSAGE, m_backupMessage.toAscii());
        settings.setValue(USER_UNSEND_MESSAGE_PUBLISH, m_backupFacebookPublishPolicity);
    } else {
        settings.remove(USER_UNSEND_MESSAGE);
        settings.remove(USER_UNSEND_MESSAGE_PUBLISH);
    }
}

void UserInfo::backupUpdateLocationDialogData(const QString &status, bool publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_backupMessage = status;
    m_backupFacebookPublishPolicity = publish;
}

void UserInfo::clearUpdateLocationDialogData()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_backupMessage.clear();
    m_backupFacebookPublishPolicity = false;
}

void UserInfo::collapse()
{
    qDebug() << __PRETTY_FUNCTION__;

    setExpanded(false);
}

void UserInfo::findButtonClicked()
{
    qDebug() << __PRETTY_FUNCTION__;

    emit findUser(m_coordinates);
}

void UserInfo::messageUpdate()
{
    qDebug() << __PRETTY_FUNCTION__;

    delete m_updateLocation;
    m_updateLocation = new UpdateLocationDialog(m_backupMessage, m_backupFacebookPublishPolicity,
                                                this);

    connect(this, SIGNAL(reverseGeoReady(QString)),
            m_updateLocation, SLOT(setAddress(QString)));

    connect(m_updateLocation, SIGNAL(statusUpdate(QString, bool)),
            this, SIGNAL(statusUpdate(QString, bool)));

    connect(m_updateLocation, SIGNAL(statusUpdate(QString, bool)),
            this, SLOT(backupUpdateLocationDialogData(QString, bool)));

    connect(m_updateLocation, SIGNAL(finished(int)),
            this, SLOT(updateLocationDialogFinished(int)));

    m_updateLocation->show();

    emit requestReverseGeo();
}

void UserInfo::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << event->pos();

    m_mousePosition = event->pos();
}

void UserInfo::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << event->pos();

    const int MOUSE_PRESS_AREA_HEIGHT = 20;
    const int MOUSE_PRESS_AREA_WIDTH = 20;

    if ((abs(m_mousePosition.y() - event->pos().y()) <= MOUSE_PRESS_AREA_WIDTH)
        && (abs(m_mousePosition.x() - event->pos().x()) <= MOUSE_PRESS_AREA_HEIGHT)) {
        if (m_expanded) {
            setExpanded(false);
            m_expanded = false;
        }
        else {
            setExpanded(true);
            m_expanded = true;
        }
    }
}

void UserInfo::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << event->rect();

    QPainter painter(this);

    QRect topRect = QRect(0, MARGIN, BACKGROUND_WIDTH, BACKGROUND_TOP_HEIGHT);

    QRect middleRect = QRect(topRect.left(), topRect.bottom() + 1, BACKGROUND_WIDTH,
                             this->height() - BACKGROUND_TOP_HEIGHT - BACKGROUND_BOTTOM_HEIGHT);

    QRect bottomRect = QRect(topRect.left(), middleRect.bottom() + 1, BACKGROUND_WIDTH,
                             BACKGROUND_BOTTOM_HEIGHT);

    painter.drawPixmap(topRect, m_backgroundTopImage);
    painter.drawPixmap(middleRect, m_backgroundMiddleImage);
    painter.drawPixmap(bottomRect, m_backgroundBottomImage);
}

void UserInfo::restoreUnsendMessage()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    m_backupMessage = settings.value(USER_UNSEND_MESSAGE, EMPTY).toString();
    m_backupFacebookPublishPolicity = settings.value(USER_UNSEND_MESSAGE_PUBLISH, false).toBool();
}

void UserInfo::setAddress(const QString &address)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationLabel->setText(address);
}

void UserInfo::setCoordinates(const GeoCoordinate &coordinates)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_coordinates = coordinates;
}

void UserInfo::setMessageText(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_messageText = TextModifier::splitLongWords(m_statusTextLabel->fontMetrics(), text,
                                                 LABEL_MAX_WIDTH);

    setExpanded(false);
}

void UserInfo::setProfileImage(const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!image.isNull())
        m_avatar->setButtonIcon(image);
}

void UserInfo::setExpanded(bool expanded)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (expanded) {
        m_statusTextLabel->setText(m_messageText);
    } else {
        m_statusTextLabel->setText(TextModifier::shortenText(m_statusTextLabel->fontMetrics(),
                                                             m_messageText, LABEL_MAX_WIDTH));
    }
}

void UserInfo::setTime(const QString &time)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_updatedLabel->setText(time);
}

void UserInfo::setUserName(const QString &name)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_userName = name;

    m_nameLabel->setText(TextModifier::shortenText(m_nameLabel->fontMetrics(), m_userName,
                                                   LABEL_MAX_WIDTH));
}

void UserInfo::updateLocationDialogFinished(int reason)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(reason);

    if (m_updateLocation) {
        disconnect(this, SIGNAL(reverseGeoReady(QString)),
                m_updateLocation, SLOT(setAddress(QString)));

        disconnect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
                this, SIGNAL(statusUpdate(QString,bool)));

        disconnect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
                this, SLOT(backupUpdateLocationDialogData(QString,bool)));

        disconnect(m_updateLocation, SIGNAL(finished(int)),
                this, SLOT(updateLocationDialogFinished(int)));

        m_updateLocation->deleteLater();
        m_updateLocation = 0;
    }
}
