/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com
      Kaj Wallin - kaj.wallin@ixonos.com
      Jussi Laitinen - jussi.laitinen@ixonos.com
      Sami Rämö - sami.ramo@ixonos.com
      Ville Tiensuu - ville.tiensuu@ixonos.com
      Katri Kaikkonen - katri.kaikkonen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QtAlgorithms>
#include <QtWebKit>

#include <QAction>
#include <QApplication>
#include <QMenuBar>
#include <QMessageBox>

#include "common.h"
#include "error.h"
#include "facebookservice/facebookauthentication.h"
#include "friendlistpanel.h"
#include "fullscreenbutton.h"
#include "indicatorbuttonpanel.h"
#include "locationsearchpanel.h"
#include "logindialog.h"
#include "map/mapcommon.h"
#include "map/mapview.h"
#include "mapscale.h"
#include "panelcommon.h"
#include "routingpanel.h"
#include "searchdialog.h"
#include "settingsdialog.h"
#include "tabbedpanel.h"
#include "userinfopanel.h"
#include "zoombuttonpanel.h"


#include "mainwindow.h"

// These MUST BE HERE, compiling for Maemo fails if moved
#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#include <QtGui/QX11Info>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#endif // Q_WS_MAEMO_5

#if defined(Q_WS_MAEMO_5) & defined(ARMEL)
#include "ossoabookdialog.h"
#endif

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent),
      m_errorShown(false),
      m_loggedIn(false),
      m_refresh(false),
      m_mapCenterHorizontalShifting(0),
      m_progressIndicatorCount(0),
      m_crosshair(0),
      m_email(), ///< @todo WTF?!?!?!?
      m_password(),
      m_webView(0),
      m_fullScreenButton(0),
      m_indicatorButtonPanel(0),
      m_mapScale(0),
      m_cookieJar(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    buildMap();

    // map view is the only widget which size & location is handled automatically by the system
    // default functionality
    setCentralWidget(m_mapView);

    buildPanels();

    createMenus();
    setWindowTitle(tr("Situare"));

    // set stacking order of widgets (from top to bottom)
    // m_tabbedPanel is the topmost one
    if (m_fullScreenButton) {
        m_fullScreenButton->stackUnder(m_tabbedPanel);
        m_crosshair->stackUnder(m_fullScreenButton);
    } else {
        m_crosshair->stackUnder(m_tabbedPanel);
    }
    m_zoomButtonPanel->stackUnder(m_crosshair);
    m_indicatorButtonPanel->stackUnder(m_zoomButtonPanel);
    m_osmLicense->stackUnder(m_indicatorButtonPanel);
    m_mapScale->stackUnder(m_osmLicense);
    m_mapView->stackUnder(m_mapScale);

    grabZoomKeys(true);

    // Set default screen size
    resize(DEFAULT_SCREEN_WIDTH, DEFAULT_SCREEN_HEIGHT);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
}

MainWindow::~MainWindow()
{
    qDebug() << __PRETTY_FUNCTION__;

    grabZoomKeys(false);

    if(m_webView)
        delete m_webView;

    qDeleteAll(m_queue.begin(), m_queue.end());
    m_queue.clear();

    qDeleteAll(m_error_queue.begin(), m_error_queue.end());
    m_error_queue.clear();
}

void MainWindow::automaticUpdateDialogFinished(int result)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (result == QMessageBox::Yes) {
        readAutomaticLocationUpdateSettings();
    } else {
        QSettings settings(DIRECTORY_NAME, FILE_NAME);
        settings.setValue(SETTINGS_AUTOMATIC_UPDATE_ENABLED, false);
        readAutomaticLocationUpdateSettings();
    }

    m_automaticUpdateLocationDialog->deleteLater();
}

void MainWindow::buildCrosshair()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_crosshair = new QLabel(this);
    QPixmap crosshairImage(":/res/images/sight.png");
    m_crosshair->setPixmap(crosshairImage);
    m_crosshair->setFixedSize(crosshairImage.size());
    m_crosshair->hide();
    m_crosshair->setAttribute(Qt::WA_TransparentForMouseEvents, true);

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(moveCrosshair()));

    connect(m_mapView, SIGNAL(horizontalShiftingChanged(int)),
            this, SLOT(mapCenterHorizontalShiftingChanged(int)));
}

void MainWindow::buildFriendListPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_friendsListPanel = new FriendListPanel(this);

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_friendsListPanel, SLOT(friendInfoReceived(QList<User*>&)));

    connect(this, SIGNAL(locationItemClicked(QList<QString>)),
            m_friendsListPanel, SLOT(showFriendsInList(QList<QString>)));

    connect(m_friendsListPanel, SIGNAL(findFriend(GeoCoordinate)),
            this, SIGNAL(centerToCoordinates(GeoCoordinate)));

    connect(this, SIGNAL(friendImageReady(User*)),
            m_friendsListPanel, SLOT(friendImageReady(User*)));

    connect(m_friendsListPanel, SIGNAL(routeToFriend(const GeoCoordinate&)),
            this, SIGNAL(routeTo(const GeoCoordinate&)));

    connect(m_friendsListPanel, SIGNAL(requestContactDialog(const QString &)),
            this, SIGNAL(requestContactDialog(const QString &)));
}

void MainWindow::buildFullScreenButton()
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    m_fullScreenButton = new FullScreenButton(this);

    if (m_fullScreenButton) {
        connect(m_fullScreenButton, SIGNAL(clicked()),
                this, SLOT(toggleFullScreen()));

        connect(qApp, SIGNAL(showFullScreenButton()),
                m_fullScreenButton, SLOT(invoke()));
    }
#endif // Q_WS_MAEMO_5
}

void MainWindow::buildIndicatorButtonPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_indicatorButtonPanel = new IndicatorButtonPanel(this);

    connect(m_indicatorButtonPanel, SIGNAL(autoCenteringTriggered(bool)),
        this, SIGNAL(autoCenteringTriggered(bool)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_indicatorButtonPanel, SLOT(screenResized(QSize)));

    connect(this, SIGNAL(directionIndicatorValuesUpdate(qreal, qreal, bool)),
            m_indicatorButtonPanel, SIGNAL(directionIndicatorValuesUpdate(qreal, qreal, bool)));

    connect(m_indicatorButtonPanel, SIGNAL(draggingModeTriggered()),
            this, SIGNAL(draggingModeTriggered()));
}

void MainWindow::buildInformationBox(const QString &message, bool modal)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString errorMessage = message;

#ifdef Q_WS_MAEMO_5

    QMaemo5InformationBox *msgBox = new QMaemo5InformationBox(this);

    if(modal) {
        msgBox->setTimeout(QMaemo5InformationBox::NoTimeout);
        // extra line changes are needed to make error notes broader
        errorMessage.prepend("\n");
        errorMessage.append("\n");
    } else {
        msgBox->setTimeout(QMaemo5InformationBox::DefaultTimeout);
    }
    QLabel *label = new QLabel(msgBox);
    label->setAlignment(Qt::AlignCenter);
    label->setText(errorMessage);
    msgBox->setWidget(label);
#else
    QMessageBox *msgBox = new QMessageBox(this);
    msgBox->button(QMessageBox::Ok);
    msgBox->setText(errorMessage);
    msgBox->setModal(modal);
#endif

    queueDialog(msgBox);
}

void MainWindow::buildLocationSearchPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationSearchPanel = new LocationSearchPanel(this);

    connect(this, SIGNAL(locationDataParsed(const QList<Location>&)),
            m_locationSearchPanel, SLOT(populateLocationListView(const QList<Location>&)));

    connect(m_locationSearchPanel, SIGNAL(locationItemClicked(const GeoCoordinate&, const GeoCoordinate&)),
            this, SIGNAL(locationItemClicked(const GeoCoordinate&, const GeoCoordinate&)));

    connect(m_locationSearchPanel, SIGNAL(routeToLocation(const GeoCoordinate&)),
            this, SIGNAL(routeTo(const GeoCoordinate&)));

    connect(m_locationSearchPanel, SIGNAL(requestSearchLocation()),
            this, SLOT(startLocationSearch()));

    connect(this, SIGNAL(searchForLocation(QString)),
            m_locationSearchPanel, SLOT(prependSearchHistory(QString)));

    connect(m_locationSearchPanel, SIGNAL(searchHistoryItemClicked(QString)),
            this, SIGNAL(searchHistoryItemClicked(QString)));
}

void MainWindow::buildMap()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapView = new MapView(this);

    buildZoomButtonPanel();
    buildOsmLicense();
    buildCrosshair();
    buildFullScreenButton();
    buildIndicatorButtonPanel();
    buildMapScale();

    connect(m_mapView, SIGNAL(viewScrolled(SceneCoordinate)),
            this, SIGNAL(mapViewScrolled(SceneCoordinate)));

    connect(this, SIGNAL(centerToSceneCoordinates(SceneCoordinate)),
            m_mapView, SLOT(centerToSceneCoordinates(SceneCoordinate)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SIGNAL(mapViewResized(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawFullScreenButton(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawMapScale(QSize)));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
             this, SLOT(moveCrosshair()));

    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_mapView, SLOT(setZoomLevel(int)));

    connect(m_mapView, SIGNAL(viewZoomFinished()),
            this, SIGNAL(viewZoomFinished()));

    connect(m_mapView, SIGNAL(zoomIn()),
            this, SIGNAL(zoomIn()));
}

void MainWindow::buildMapScale()
{
    m_mapScale = new MapScale(this);
    connect(this, SIGNAL(newMapResolution(qreal)),
            m_mapScale, SLOT(updateMapResolution(qreal)));
}

void MainWindow::buildOsmLicense()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_osmLicense = new QLabel(this);
    m_osmLicense->setAttribute(Qt::WA_TranslucentBackground, true);
    m_osmLicense->setAttribute(Qt::WA_TransparentForMouseEvents, true);
    m_osmLicense->setText("<font color='black'>" + OSM_LICENSE + "</font>");
    m_osmLicense->setFont(QFont("Nokia Sans", 9));
    m_osmLicense->resize(m_osmLicense->fontMetrics().width(OSM_LICENSE),
                         m_osmLicense->fontMetrics().height());

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            this, SLOT(drawOsmLicense(QSize)));
}

void MainWindow::buildPanels()
{
    qDebug() << __PRETTY_FUNCTION__;

    buildUserInfoPanel();
    buildFriendListPanel();
    buildLocationSearchPanel();
    buildRoutingPanel();

    m_tabbedPanel = new TabbedPanel(this);

    //Save Situare related tab indexes so tabs can be enabled/disabled when logged in/out
    m_situareTabsIndexes.append(
            m_tabbedPanel->addTab(m_userInfoPanel, QIcon(":/res/images/user_info.png")));
    m_situareTabsIndexes.append(
            m_tabbedPanel->addTab(m_friendsListPanel, QIcon(":/res/images/friend_list.png")));

    m_tabbedPanel->addTab(m_locationSearchPanel, QIcon(":/res/images/location_search.png"));
    m_tabbedPanel->addTab(m_routingPanel, QIcon(":/res/images/routing.png"));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_tabbedPanel, SLOT(resizePanel(QSize)));

    connect(m_friendsListPanel, SIGNAL(openPanelRequested(QWidget*)),
            m_tabbedPanel, SLOT(openPanel(QWidget*)));

    connect(m_routingPanel, SIGNAL(openPanelRequested(QWidget*)),
            m_tabbedPanel, SLOT(openPanel(QWidget*)));

    connect(m_tabbedPanel, SIGNAL(panelClosed()),
            m_friendsListPanel, SLOT(anyPanelClosed()));

    connect(m_tabbedPanel, SIGNAL(panelOpened()),
            m_friendsListPanel, SLOT(anyPanelOpened()));

    connect(m_tabbedPanel, SIGNAL(panelClosed()),
            m_routingPanel, SLOT(clearListsSelections()));

    connect(m_tabbedPanel, SIGNAL(panelClosed()),
            m_mapView, SLOT(disableCenterShift()));

    connect(m_tabbedPanel, SIGNAL(panelOpened()),
            m_mapView, SLOT(enableCenterShift()));

    connect(m_tabbedPanel, SIGNAL(panelClosed()),
            m_userInfoPanel, SIGNAL(collapse()));

    connect(m_tabbedPanel, SIGNAL(currentChanged(int)),
            m_userInfoPanel, SIGNAL(collapse()));

    // signals for showing and hiding list item context buttons
    connect(m_friendsListPanel, SIGNAL(listItemSelectionChanged(bool)),
            m_tabbedPanel, SIGNAL(listItemSelectionChanged(bool)));

    connect(m_locationSearchPanel, SIGNAL(listItemSelectionChanged(bool)),
            m_tabbedPanel, SIGNAL(listItemSelectionChanged(bool)));

    connect(m_routingPanel, SIGNAL(listItemSelectionChanged(bool)),
            m_tabbedPanel, SIGNAL(listItemSelectionChanged(bool)));
}

void MainWindow::buildRoutingPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_routingPanel = new RoutingPanel(this);

    connect(m_routingPanel, SIGNAL(routeToCursor()),
            this, SIGNAL(routeToCursor()));

    connect(this, SIGNAL(routeParsed(Route&)),
            m_routingPanel, SLOT(setRoute(Route&)));

    connect(m_routingPanel, SIGNAL(routeWaypointItemClicked(GeoCoordinate)),
            this, SIGNAL(centerToCoordinates(GeoCoordinate)));

    connect(m_routingPanel, SIGNAL(clearRoute()),
            this, SIGNAL(clearRoute()));
}

void MainWindow::buildUserInfoPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_userInfoPanel = new UserInfoPanel(this);

    connect(this, SIGNAL(userLocationReady(User*)),
            m_userInfoPanel, SLOT(userDataReceived(User*)));

    connect(this, SIGNAL(reverseGeoReady(QString)),
            m_userInfoPanel, SIGNAL(reverseGeoReady(QString)));

    connect(this, SIGNAL(clearUpdateLocationDialogData()),
            m_userInfoPanel, SIGNAL(clearUpdateLocationDialogData()));

    connect(m_userInfoPanel, SIGNAL(findUser(GeoCoordinate)),
            this, SIGNAL(centerToCoordinates(GeoCoordinate)));

    connect(m_userInfoPanel, SIGNAL(requestReverseGeo()),
            this, SIGNAL(requestReverseGeo()));

    connect(m_userInfoPanel, SIGNAL(statusUpdate(QString,bool)),
            this, SIGNAL(statusUpdate(QString,bool)));

    connect(m_userInfoPanel, SIGNAL(refreshUserData()),
            this, SIGNAL(refreshUserData()));

    connect(m_userInfoPanel, SIGNAL(notificateUpdateFailing(QString, bool)),
            this, SLOT(buildInformationBox(QString, bool)));
}

void MainWindow::buildWebView()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!m_webView) {
        m_webView = new QWebView;

        if(!m_cookieJar)
            m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

        m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);

        connect(m_webView->page()->networkAccessManager(), SIGNAL(finished(QNetworkReply*)),
                this, SLOT(webViewRequestFinished(QNetworkReply*)));
        connect(m_webView, SIGNAL(urlChanged(const QUrl &)),
                this, SIGNAL(updateCredentials(QUrl)));
        connect(m_webView, SIGNAL(loadFinished(bool)),
                this, SLOT(loadDone(bool)));

        m_webView->hide();
    }
}

void MainWindow::buildZoomButtonPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomButtonPanel = new ZoomButtonPanel(this);

    connect(m_zoomButtonPanel->zoomInButton(), SIGNAL(clicked()),
            this, SIGNAL(zoomIn()));

    connect(m_zoomButtonPanel->zoomOutButton(), SIGNAL(clicked()),
            this, SIGNAL(zoomOut()));

    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_zoomButtonPanel, SLOT(resetButtons()));

    connect(this, SIGNAL(maxZoomLevelReached()),
            m_zoomButtonPanel, SLOT(disableZoomInButton()));

    connect(this, SIGNAL(minZoomLevelReached()),
            m_zoomButtonPanel, SLOT(disableZoomOutButton()));

    connect(m_mapView, SIGNAL(viewResized(QSize)),
            m_zoomButtonPanel, SLOT(screenResized(QSize)));

    connect(m_zoomButtonPanel, SIGNAL(draggingModeTriggered()),
            this, SIGNAL(draggingModeTriggered()));
}

void MainWindow::clearCookieJar()
{
    qDebug() << __PRETTY_FUNCTION__;

    buildWebView();

    m_webView->stop();

    if(!m_cookieJar) {
        m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));
    }
    QList<QNetworkCookie> emptyList;
    emptyList.clear();

    m_cookieJar->setAllCookies(emptyList);
    m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);
}

void MainWindow::createMenus()
{
    qDebug() << __PRETTY_FUNCTION__;

    // login/logout
    m_loginAct = new QAction(tr("Login"), this);
    connect(m_loginAct, SIGNAL(triggered()),
            this, SIGNAL(loginActionPressed()));

    // settings
    m_toSettingsAct = new QAction(tr("Settings"), this);
    connect(m_toSettingsAct, SIGNAL(triggered()),
        this, SLOT(openSettingsDialog()));

    // GPS
    m_gpsToggleAct = new QAction(tr("GPS"), this);
    m_gpsToggleAct->setCheckable(true);

    connect(m_gpsToggleAct, SIGNAL(triggered(bool)),
            this, SIGNAL(gpsTriggered(bool)));

    // build the actual menu
    m_viewMenu = menuBar()->addMenu(tr("Main"));
    m_viewMenu->addAction(m_loginAct);
    m_viewMenu->addAction(m_toSettingsAct);
    m_viewMenu->addAction(m_gpsToggleAct);
    m_viewMenu->setObjectName(tr("Menu"));
}

void MainWindow::dialogFinished(int status)
{
    qDebug() << __PRETTY_FUNCTION__;

    QDialog *dialog = m_queue.takeFirst();
    LoginDialog *loginDialog = qobject_cast<LoginDialog *>(dialog);
    SearchDialog *searchDialog = qobject_cast<SearchDialog *>(dialog);
    if(loginDialog) {
        if(status != 0) {
            buildWebView();
            loginDialog->userInput(m_email, m_password);

            QStringList urlParts;
            urlParts.append(FACEBOOK_LOGINBASE);
            urlParts.append(SITUARE_PUBLIC_FACEBOOKAPI_KEY);
            urlParts.append(INTERVAL1);
            urlParts.append(SITUARE_LOGIN_SUCCESS);
            urlParts.append(INTERVAL2);
            urlParts.append(SITUARE_LOGIN_FAILURE);
            urlParts.append(FACEBOOK_LOGIN_ENDING);

            emit saveUsername(m_email);
            m_refresh = true;
            m_webView->load(QUrl(urlParts.join(EMPTY)));
            toggleProgressIndicator(true);
        } else {
            emit cancelLoginProcess();
        }
    } else if(searchDialog) {
        if(status != 0) {
            emit searchForLocation(searchDialog->input());
        }
    }

    dialog->deleteLater();

    if(!m_error_queue.isEmpty() && m_errorShown == false) {
        showErrorInformationBox();
    } else {
        if(!m_queue.isEmpty()) {
            showInformationBox();
        }
    }
}

void MainWindow::drawFullScreenButton(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__ << size.width() << "x" << size.height();

    if (m_fullScreenButton) {
        m_fullScreenButton->move(size.width() - m_fullScreenButton->size().width(),
                                 size.height() - m_fullScreenButton->size().height());
    }
}

void MainWindow::drawMapScale(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int LEFT_SCALE_MARGIN = 10;
    const int BOTTOM_SCALE_MARGIN = 2;

    m_mapScale->move(LEFT_SCALE_MARGIN,
                     size.height() - m_mapScale->size().height() - BOTTOM_SCALE_MARGIN);
}

void MainWindow::drawOsmLicense(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__ << size.width() << "x" << size.height();

    m_osmLicense->move(size.width() - m_osmLicense->fontMetrics().width(OSM_LICENSE)
                       - PANEL_BAR_WIDTH,
                       size.height() - m_osmLicense->fontMetrics().height());
}

void MainWindow::errorDialogFinished(int status)
{
    qDebug() << __PRETTY_FUNCTION__;

    qDebug() << status;
    QDialog *dialog = m_error_queue.takeFirst();

    dialog->deleteLater();
    m_errorShown = false;

    if(!m_error_queue.isEmpty())
        showErrorInformationBox();
    else if(!m_queue.isEmpty())
        showInformationBox();
}

void MainWindow::gpsTimeout()
{
    qDebug() << __PRETTY_FUNCTION__;

    buildInformationBox(tr("GPS timeout"));
}

void MainWindow::grabZoomKeys(bool grab)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    // Can't grab keys unless we have a window id
    if (!winId())
        return;

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom)
        return;

    XChangeProperty (QX11Info::display(),
                     winId(),
                     atom,
                     XA_INTEGER,
                     32,
                     PropModeReplace,
                     reinterpret_cast<unsigned char *>(&val),
                     1);
#else
    Q_UNUSED(grab);
#endif // Q_WS_MAEMO_5
}

void MainWindow::keyPressEvent(QKeyEvent* event)
{
    qDebug() << __PRETTY_FUNCTION__;

    switch (event->key()) {
    case Qt::Key_F7:
        event->accept();
        emit zoomIn();
        break;

    case Qt::Key_F8:
        event->accept();
        emit zoomOut();
        break;
    }
    QWidget::keyPressEvent(event);
}

void MainWindow::loadCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    QStringList list = settings.value(COOKIES, EMPTY).toStringList();

    if(!list.isEmpty()) {
        QList<QNetworkCookie> cookieList;
        for(int i=0;i<list.count();i++) {
            cookieList.append(QNetworkCookie::parseCookies(list.at(i).toAscii()));
        }

        if(!m_cookieJar)
               m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

        m_cookieJar->setAllCookies(cookieList);
        m_webView->page()->networkAccessManager()->setCookieJar(m_cookieJar);
    }
}

void MainWindow::loadDone(bool done)
{
    qDebug() << __PRETTY_FUNCTION__;

    // for the first time the login page is opened, we need to refresh it to get cookies working
    if(m_refresh) {
        m_webView->reload();
        m_refresh = false;
    }

    if (done)
    {
        QWebFrame* frame = m_webView->page()->currentFrame();
        if (frame!=NULL)
        {
            // set email box
            QWebElementCollection emailCollection = frame->findAllElements("input[name=email]");

            foreach (QWebElement element, emailCollection) {
                element.setAttribute("value", m_email.toAscii());
            }
            // set password box
            QWebElementCollection passwordCollection = frame->findAllElements("input[name=pass]");
            foreach (QWebElement element, passwordCollection) {
                element.setAttribute("value", m_password.toAscii());
            }
            // find connect button
            QWebElementCollection buttonCollection = frame->findAllElements("input[name=login]");
            foreach (QWebElement element, buttonCollection)
            {
                QPoint pos(element.geometry().center());

                // send a mouse click event to the web page
                QMouseEvent event0(QEvent::MouseButtonPress, pos, Qt::LeftButton, Qt::LeftButton,
                                   Qt::NoModifier);
                QApplication::sendEvent(m_webView->page(), &event0);
                QMouseEvent event1(QEvent::MouseButtonRelease, pos, Qt::LeftButton, Qt::LeftButton,
                                   Qt::NoModifier);
                QApplication::sendEvent(m_webView->page(), &event1);
            }
        }
    }
}

void MainWindow::loggedIn(bool logged)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_loggedIn = logged;

    if(logged) {
        m_loginAct->setText(tr("Logout"));
    } else {
        clearCookieJar();
        m_email.clear();
        m_password.clear();

        m_loginAct->setText(tr("Login"));
        m_userInfoPanel->showUserInfo(false);
    }
    updateItemVisibility();
}

void MainWindow::loginFailed()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearCookieJar();
    startLoginProcess();
}

bool MainWindow::loginState()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_loggedIn;
}

void MainWindow::loginUsingCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    toggleProgressIndicator(true);

    buildWebView();
    loadCookies();

    QStringList urlParts;
    urlParts.append(FACEBOOK_LOGINBASE);
    urlParts.append(SITUARE_PUBLIC_FACEBOOKAPI_KEY);
    urlParts.append(INTERVAL1);
    urlParts.append(SITUARE_LOGIN_SUCCESS);
    urlParts.append(INTERVAL2);
    urlParts.append(SITUARE_LOGIN_FAILURE);
    urlParts.append(FACEBOOK_LOGIN_ENDING);

    m_webView->load(QUrl(urlParts.join(EMPTY)));

}

void MainWindow::mapCenterHorizontalShiftingChanged(int shifting)
{
    m_mapCenterHorizontalShifting = shifting;
    moveCrosshair();
}

void MainWindow::moveCrosshair()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_crosshair) {
        int mapHeight = m_mapView->size().height();
        int mapWidth = m_mapView->size().width();
        m_crosshair->move(mapWidth / 2 - m_crosshair->pixmap()->width() / 2
                          - m_mapCenterHorizontalShifting,
                          mapHeight / 2 - m_crosshair->pixmap()->height() / 2);
    }
}

void MainWindow::openSettingsDialog()
{
    qDebug() << __PRETTY_FUNCTION__;

    SettingsDialog *settingsDialog = new SettingsDialog(this);
    settingsDialog->enableSituareSettings((m_loggedIn && m_gpsToggleAct->isChecked()));
    connect(settingsDialog, SIGNAL(accepted()), this, SLOT(settingsDialogAccepted()));

    settingsDialog->show();
}

void MainWindow::queueDialog(QDialog *dialog)
{
    qDebug() << __PRETTY_FUNCTION__;

    // check is dialog is modal, for now all modal dialogs have hihger priority i.e. errors
    if(dialog->isModal()) {
        m_error_queue.append(dialog);
    } else {
        m_queue.append(dialog);
    }

    // show error dialog if there is only one error dialog in the queue and no error dialog is shown
    if(m_error_queue.count() == 1 && m_errorShown == false)
        showErrorInformationBox();
    else if(m_queue.count() == 1 && m_errorShown == false)
        showInformationBox();
}

void MainWindow::readAutomaticLocationUpdateSettings()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    bool automaticUpdateEnabled = settings.value(SETTINGS_AUTOMATIC_UPDATE_ENABLED, false).toBool();
    QTime automaticUpdateInterval = settings.value(SETTINGS_AUTOMATIC_UPDATE_INTERVAL, QTime())
                                      .toTime();

    if (automaticUpdateEnabled && automaticUpdateInterval.isValid()) {
        QTime time;
        emit enableAutomaticLocationUpdate(true, time.msecsTo(automaticUpdateInterval));
    } else {
        emit enableAutomaticLocationUpdate(false);
    }
}

void MainWindow::saveCookies()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(!m_cookieJar)
        m_cookieJar = new NetworkCookieJar(new QNetworkCookieJar(this));

    QList<QNetworkCookie> cookieList = m_cookieJar->allCookies();
    QStringList list;

    for(int i=0;i<cookieList.count();i++) {
        QNetworkCookie cookie = cookieList.at(i);
        QByteArray byteArray = cookie.toRawForm(QNetworkCookie::Full);
        list.append(QString(byteArray));
    }
    list.removeDuplicates();

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(COOKIES, list);
}

void MainWindow::setCrosshairVisibility(bool visibility)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (visibility) {
        m_crosshair->show();
        moveCrosshair();
    } else {
        m_crosshair->hide();
    }
}

void MainWindow::setGPSButtonEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gpsToggleAct->setChecked(enabled);
}

void MainWindow::setIndicatorButtonEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_indicatorButtonPanel->setIndicatorButtonEnabled(enabled);
}

void MainWindow::setMapViewScene(QGraphicsScene *scene)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapView->setScene(scene);
}

void MainWindow::settingsDialogAccepted()
{
    qDebug() << __PRETTY_FUNCTION__;

    readAutomaticLocationUpdateSettings();
}

void MainWindow::setUsername(const QString &username)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_email = username;
}

void MainWindow::showContactDialog(const QString &guid)
{
    qDebug() << __PRETTY_FUNCTION__;

#if defined(Q_WS_MAEMO_5) & defined(ARMEL)
    OssoABookDialog::showContactDialog(guid);
#else
    Q_UNUSED(guid);
    buildInformationBox(tr("Contact dialog works only on phone!"), true);
#endif
}

void MainWindow::showEnableAutomaticUpdateLocationDialog(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_automaticUpdateLocationDialog = new QMessageBox(QMessageBox::Question,
                                                      tr("Automatic location update"), text,
                                                      QMessageBox::Yes | QMessageBox::No |
                                                      QMessageBox::Cancel, this);
    connect(m_automaticUpdateLocationDialog, SIGNAL(finished(int)),
            this, SLOT(automaticUpdateDialogFinished(int)));

    m_automaticUpdateLocationDialog->show();
}

void MainWindow::showErrorInformationBox()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_error_queue.count()) {
        m_errorShown = true;
        QDialog *dialog = m_error_queue.first();
        connect(dialog, SIGNAL(finished(int)),
                this, SLOT(errorDialogFinished(int)));
        dialog->show();
    }
}

void MainWindow::showInformationBox()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_queue.count()) {
        QDialog *dialog = m_queue.first();
        connect(dialog, SIGNAL(finished(int)),
                this, SLOT(dialogFinished(int)));
        dialog->show();
    }
}

void MainWindow::startLocationSearch()
{
    qDebug() << __PRETTY_FUNCTION__;

    SearchDialog *searchDialog = new SearchDialog();
    queueDialog(searchDialog);
}

void MainWindow::startLoginProcess()
{
    qDebug() << __PRETTY_FUNCTION__;

    LoginDialog *loginDialog = new LoginDialog();

    emit fetchUsernameFromSettings();

    loginDialog->clearTextFields();

    if(!m_email.isEmpty())
        loginDialog->setEmailField(m_email);

    queueDialog(loginDialog);
}

void MainWindow::toggleFullScreen()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(windowState() == Qt::WindowNoState)
        showFullScreen();
    else
        showNormal();
}

void MainWindow::toggleProgressIndicator(bool value)
{
    qDebug() << __PRETTY_FUNCTION__;

#ifdef Q_WS_MAEMO_5
    if(value) {
        m_progressIndicatorCount++;
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    } else {
        if(m_progressIndicatorCount > 0)
            m_progressIndicatorCount--;

        if(m_progressIndicatorCount == 0)
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    }
#else
    Q_UNUSED(value);
#endif // Q_WS_MAEMO_5
}

void MainWindow::updateItemVisibility()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_tabbedPanel->setTabsEnabled(m_situareTabsIndexes, m_loggedIn);
}

const QString MainWindow::username()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_email;
}

void MainWindow::webViewRequestFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    // omit QNetworkReply::OperationCanceledError due to it's nature to be called when ever
    // qwebview starts to load a new page while the current page loading is not finished
    if(reply->error() != QNetworkReply::OperationCanceledError &&
       reply->error() != QNetworkReply::NoError) {
        emit error(ErrorContext::NETWORK, reply->error());
    }
}
