/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Pekka Nissinen - pekka.nissinen@ixonos.com
       Kaj Wallin - kaj.wallin@ixonos.com
       Katri Kaikkonen - katri.kaikkonen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QPainter>
#include <QSettings>

#include "common.h"
#include "indicatorbutton.h"
#include "panelcommon.h"

#include "indicatorbuttonpanel.h"

const QString DIRECTION_INDICATOR_BUTTON_POSITION = "DIRECTION_INDICATOR_POSITION";

IndicatorButtonPanel::IndicatorButtonPanel(QWidget *parent)
    : QWidget(parent),
      m_isDraggable(false)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int DIRECTION_INDICATOR_POSITION_X = 10;
    const int DIRECTION_INDICATOR_POSITION_Y = 290;

    const int INDICATOR_BUTTON_PANEL_SPACING = 5;

    const qreal OPACITY = 0.50;
    const int MARGIN_LEFT = 0;
    const int MARGIN_TOP = 3;
    const int MARGIN_RIGHT = 0;
    const int MARGIN_BOTTOM = 0;
    const int LABEL_MARGIN_TOP = 0;

    const int PANEL_WIDTH = 90;
    const int PANEL_HEIGHT = 100;

    QVBoxLayout *verticalLayout = new QVBoxLayout;
    setLayout(verticalLayout);
    verticalLayout->setContentsMargins(MARGIN_LEFT, MARGIN_TOP, MARGIN_RIGHT, MARGIN_BOTTOM);
    verticalLayout->setSpacing(INDICATOR_BUTTON_PANEL_SPACING);

    m_indicatorButton = new IndicatorButton(this);

    m_distanceTextLabel = new QLabel();
    m_distanceTextLabel->setFont(QFont(NOKIA_FONT_SMALL));
    m_distanceTextLabel->setAlignment(Qt::AlignHCenter);
    m_distanceTextLabel->setContentsMargins(MARGIN_LEFT, LABEL_MARGIN_TOP, MARGIN_RIGHT
                                            , MARGIN_BOTTOM);

    m_normalColor = new QColor(Qt::black);
    m_normalColor->setAlpha(OPACITY * 255);

    verticalLayout->addWidget(m_indicatorButton, 0, Qt::AlignHCenter);
    verticalLayout->addWidget(m_distanceTextLabel, 0, Qt::AlignHCenter);
    verticalLayout->addStretch();

    setFixedSize(PANEL_WIDTH, PANEL_HEIGHT);

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    QPoint savedLocation = settings.value(DIRECTION_INDICATOR_BUTTON_POSITION,
                                          QPoint(DIRECTION_INDICATOR_POSITION_X,
                                                 DIRECTION_INDICATOR_POSITION_Y)).toPoint();

    if ((savedLocation.x() > DEFAULT_SCREEN_WIDTH) || (savedLocation.y() > DEFAULT_SCREEN_HEIGHT)) {
        savedLocation.rx() = DIRECTION_INDICATOR_POSITION_X;
        savedLocation.ry() = DIRECTION_INDICATOR_POSITION_Y;
    }

    move(savedLocation);

    m_dragStartTimer = new QTimer(this);
    m_dragStartTimer->setSingleShot(true);
    m_dragStartTimer->setInterval(DRAG_INIT_TIME);

    m_forceReleaseTimer = new QTimer(this);
    m_forceReleaseTimer->setSingleShot(true);
    m_forceReleaseTimer->setInterval(FORCE_RELEASE_TIME);

    connect(m_dragStartTimer, SIGNAL(timeout()),
            this, SLOT(timerExpired()));

    connect(m_forceReleaseTimer, SIGNAL(timeout()),
            this, SLOT(forceMouseRelease()));

    connect(this, SIGNAL(directionIndicatorValuesUpdate(qreal, qreal, bool)),
            this, SLOT(updateValues(qreal, qreal, bool)));

    connect(m_indicatorButton, SIGNAL(autoCenteringTriggered(bool)),
            this, SIGNAL(autoCenteringTriggered(bool)));
}

IndicatorButtonPanel::~IndicatorButtonPanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    delete m_normalColor;
}

void IndicatorButtonPanel::forceMouseRelease()
{
    qDebug() << __PRETTY_FUNCTION__;

    releaseMouse();
    setDraggable(false);
}

void IndicatorButtonPanel::mouseMoveEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_isDraggable) {
        if (event->buttons() & Qt::LeftButton) {
            QPoint newLocation = mapToParent(event->pos()) - m_dragPosition;

            if (newLocation.x() < 0)
                newLocation.rx() = 0;
            else if (newLocation.x() > (m_screenSize.width() - width() - PANEL_BAR_WIDTH))
                newLocation.rx() =  m_screenSize.width() - width() - PANEL_BAR_WIDTH;

            if (newLocation.y() < 0)
                newLocation.ry() = 0;
            else if (newLocation.y() > (m_screenSize.height() - height()))
                newLocation.ry() = m_screenSize.height() - height();

            move(newLocation);
        }
    } else {
        if (!rect().contains(event->pos()))
            m_dragStartTimer->stop();
    }
}

void IndicatorButtonPanel::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_dragPosition = event->pos();
    m_dragStartTimer->start();
}

void IndicatorButtonPanel::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_dragStartTimer->stop();

    Q_UNUSED(event);
    if (m_isDraggable) {
        setDraggable(false);

        QSettings settings(DIRECTORY_NAME, FILE_NAME);
        settings.setValue(DIRECTION_INDICATOR_BUTTON_POSITION, pos());
    }
}

void IndicatorButtonPanel::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(event);

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);

    if (!m_indicatorButton->isChecked()) {
        const int EXTRA_SPACE_LEFT = -5;
        const int EXTRA_SPACE_TOP = 0;
        const int EXTRA_SPACE_RIGHT = +5;
        const int EXTRA_SPACE_BOTTOM = 0;

        const int RADIUS_WIDTH = 5;
        const int RADIUS_HEIGHT = 5;

        QPainterPath backgroundPath;
        QRect distanceLabelRect = m_distanceTextLabel->rect();

        distanceLabelRect.translate(m_distanceTextLabel->pos());
        distanceLabelRect.adjust(EXTRA_SPACE_LEFT, EXTRA_SPACE_TOP, EXTRA_SPACE_RIGHT
                                 , EXTRA_SPACE_BOTTOM);
        backgroundPath.addRoundedRect(distanceLabelRect, RADIUS_WIDTH, RADIUS_HEIGHT);
        painter.fillPath(backgroundPath, QBrush(*m_normalColor));
    }

    if (m_isDraggable) {
        const int ROUNDING_RADIUS = 9;

        QPainterPath backgroundPath;
        backgroundPath.addRoundedRect(this->rect(), ROUNDING_RADIUS, ROUNDING_RADIUS);
        painter.fillPath(backgroundPath, QBrush(Qt::Dense4Pattern));
        painter.setPen(Qt::black);
    }
}

void IndicatorButtonPanel::screenResized(const QSize &newSize)
{
    qDebug() << __PRETTY_FUNCTION__;

    int oldHeight = 0;
    int oldWidth = 0;

    if (m_screenSize.height() < 0)
        oldHeight = DEFAULT_NON_FULLSCREEN_HEIGHT;
    else
        oldHeight = m_screenSize.height();

    if (m_screenSize.width() < 0)
        oldWidth = DEFAULT_SCREEN_WIDTH;
    else
        oldWidth = m_screenSize.width();

    m_screenSize = newSize;

    QPoint resizedPosition = pos();

    if (resizedPosition.x() < 0)
        resizedPosition.rx() = 0;
    else if (resizedPosition.x() > (newSize.width() - rect().width() - PANEL_BAR_WIDTH))
        resizedPosition.rx() = newSize.width() - rect().width() - PANEL_BAR_WIDTH;

    if (resizedPosition.y() < 0)
        resizedPosition.ry() = 0;
    else if (resizedPosition.y() > (newSize.height() - rect().height()))
        resizedPosition.ry() = newSize.height() - rect().height();

    if ((pos().y() + rect().center().y()) > (oldHeight / 2))
        resizedPosition.ry() = newSize.height() - (oldHeight - pos().y());

    if ((pos().x() + rect().center().x()) > (oldWidth / 2))
        resizedPosition.rx() = newSize.width() - (oldWidth - pos().x());

    move(resizedPosition);
}

void IndicatorButtonPanel::setDraggable(bool mode, QPoint eventPosition)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_isDraggable = mode;

    if (mode) {
        emit draggingModeTriggered();
        grabMouse();
        m_forceReleaseTimer->start();
        m_dragPosition = eventPosition;
    } else {
        releaseMouse();
        m_forceReleaseTimer->stop();
        m_indicatorButton->setDown(false);
    }
    update();
}

void IndicatorButtonPanel::setIndicatorButtonEnabled(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_indicatorButton->setChecked(enabled);
    m_distanceTextLabel->setVisible(!enabled);

    update();
}

void IndicatorButtonPanel::timerExpired()
{
    qDebug() << __PRETTY_FUNCTION__;

    setDraggable(true, m_dragPosition);
}

void IndicatorButtonPanel::updateValues(qreal direction, qreal distance, bool draw)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int MAX_TO_METERS = 999.5;
    const int MAX_TO_KM_WITH_DESIMAL = 99950;
    const int MIN_TO_METERS = 10;
    const int M_TO_KM = 1000;                   ///< Meters to kilometers conversion

    QString UNIT_KILOMETER = " km";
    QString UNIT_METER = " m";

    m_indicatorButton->setDirection(direction, draw);

    m_distance = distance;

    if (m_distance < MIN_TO_METERS) {
        m_distanceText.setNum(10);
        m_distanceText.prepend("< ");
        m_distanceText.append(UNIT_METER);
    } else if (m_distance < MAX_TO_METERS) {
        m_distanceText.setNum(m_distance, 'f', 0);
        m_distanceText.append(UNIT_METER);
    } else if (m_distance < MAX_TO_KM_WITH_DESIMAL) {
        m_distanceText.setNum(m_distance / M_TO_KM, 'f', 1);
        m_distanceText.append(UNIT_KILOMETER);
    } else {
        m_distanceText.setNum(m_distance / M_TO_KM, 'f', 0);
        m_distanceText.append(UNIT_KILOMETER);
    }

    m_distanceTextLabel->setText(m_distanceText);

    update();
}
