/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com
        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef FRIENDLISTPANEL_H
#define FRIENDLISTPANEL_H

#include "panelbase.h"

class QLabel;
class QLineEdit;
class QPushButton;

class FriendListItemDelegate;
class FriendListView;
class GeoCoordinate;
class ImageButton;
class User;

/**
 * @brief Class for sliding friends list panel
 *
 * @author Kaj Wallin - kaj.wallin (at) ixonos.com
 * @author Henri Lampela - henri.lampela (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
 * @author Jussi Laitinen - jussi.laitinen (at) ixonos.com
 * @author Sami Rämö - sami.ramo (at) ixonos.com
 */
class FriendListPanel : public PanelBase
{
    Q_OBJECT

public:
    /**
     * @brief Default constructor
     *
     * @param parent
     */
    FriendListPanel(QWidget *parent = 0);

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
protected:
    /**
    * @brief Re-implemented from QWidget::hideEvent()
    *
    * Calls updateKeyboardGrabbing() and clearFriendGroupFiltering().
    *
    * @param event
    */
    void hideEvent(QHideEvent *event);

    /**
    * @brief Re-implemented from QWidget::showEvent()
    *
    * Calls updateKeyboardGrabbing().
    *
    * @param event
    */
    void showEvent(QShowEvent *event);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:
    /**
     * @brief Slot to update friend item's image
     *
     * @param user Friend
     */
    void friendImageReady(User *user);

    /**
     * @brief Slot to refresh friends list
     *
     * @param friendList
     */
    void friendInfoReceived(QList<User *> &friendList);

private:
    /**
    * @brief Set visibility for filtering text field and clearing button
    *
    * @param visible True if items should be visible, false if not
    */
    void setFilteringLayoutVisibility(bool visible);

    /**
    * @brief Takes care of grabbing and releasing the keyboard when required
    *
    * Keyboard is grabbed when MainWindow it the topmost window, panel tab is open and
    * FriendListPanel is visible. Releasing is done if the MainWindow is not the topmost window
    * or panel tab is not open or FriendListPanel is invisible.
    */
    void updateKeyboardGrabbing();

private slots:
    /**
    * @brief Called when any of the panel tabs is closed
    *
    * Does call clearFriendGroupFiltering().
    *
    * Calls updateKeyboardGrabbing() for releasing the grabbing when FriendListPanel is closed, not
    * changed to other panel. In this case the hideEvent() is not triggered.
    */
    void anyPanelClosed();

    /**
    * @brief Called when any of the panel tabs is opened
    *
    * Calls updateKeyboardGrabbing() for grabbing the keyboard when FriendListPanel is closed and
    * is the last selected tab. In this case the showEvent() is not triggered.
    */
    void anyPanelOpened();

    /**
    * @brief Slot for clearing the filtering
    *
    * Clears friend group (or single friend) filtering and calls clearTextFiltering().
    */
    void clearFiltering();

    /**
    * @brief Slot for clearing the text based filtering.
    *
    * Does clear only the text based filtering. Friend group based filtering is not affected.
    */
    void clearTextFiltering();

    /**
    * @brief Updates the filtering when filtering text value is changed
    *
    * Filtering UI elements are invoked when the text becomes not empty and hidden when text
    * becomes empty.
    *
    * Sets the new filtering text.
    *
    * @param text New text value
    */
    void filterTextChanged(const QString &text);

    /**
    * @brief Requests selected friend's contact dialog.
    */
    void requestSelectedFriendContactDialog();

    /**
    * @brief Routes to selected friend.
    *
    * Emits routeToFriend if friend is selected from list.
    */
    void routeToSelectedFriend();

    /**
     * @brief show / hide empty panel label
     *
     * @param show true if empty panel should be shown
     */
    void showEmptyPanel(bool show);

    /**
     * @brief Slot to show friends in list.
     *
     * Shows only friends that are on userIDs list
     * @param userIDs list of user ID's
     */
    void showFriendsInList(const QList<QString> &userIDs);

    /**
    * @brief Called when topmost window is changed
    *
    * Does set m_mainWindowIsTopmost and calls updateKeyboardGrabbing()
    *
    * @param mainWindowIsTopmost True if MainWindow is the topmost one
    */
    void topmostWindowChanged(bool mainWindowIsTopmost);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
     * @brief Signal for friend finding
     *
     * @param coordinates Target coordinate
     */
    void findFriend(const GeoCoordinate &coordinates);

    /**
    * @brief Requests contact dialog.
    *
    * @param facebookId contact's facebookId
    */
    void requestContactDialog(const QString &facebookId);

    /**
    * @brief Signal for routing to friend.
    *
    * @param coordinates friend's geo coordinates
    */
    void routeToFriend(const GeoCoordinate &coordinates);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_mainWindowIsTopmost;                 ///< Is the MainWindow the topmost one
    bool m_somePanelIsOpen;                     ///< Is any panel tab open

    QLabel *m_headerLabel;                      ///< Show how many friends are selected
    QLabel *m_noFriendsLabel;                   ///< Text label for empty panel

    QLineEdit *m_filterField;                   ///< Text field for the filter text

    QPushButton *m_clearTextFilteringButton;    ///< Button for clearing the text filtering

    QWidget *m_headerWidget;                    ///< Friend list header widget

    FriendListView *m_friendListView;           ///< Friend list view
    ImageButton *m_clearGroupFilteringButton;   ///< Button for clearing friend group filtering
    ImageButton *m_showContactButton;           ///< Button for showing contact dialog    
    ImageButton *m_routeButton;                 ///< Button for routing to selected friend
};

#endif // FRIENDLISTPANEL_H
