/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QPainter>

#include "gps/gpscommon.h"
#include "map/mapcommon.h"

#include "gpslocationitem.h"

const qreal RING_OUTLINE_PEN_WIDTH = 1.0;

GPSLocationItem::GPSLocationItem()
    : m_showOnNextUpdate(true)
{
    qDebug() << __PRETTY_FUNCTION__;

    setPos(QPoint(UNDEFINED, UNDEFINED));
    setZValue(GPSLocationItemZValue);

    // create a child item for the actual red position spot
    m_pixmapItem = new QGraphicsPixmapItem(QPixmap(":/res/images/gps_position.png"));
    m_pixmapItem->setParentItem(this);
    m_pixmapItem->setFlag(QGraphicsItem::ItemIgnoresTransformations);
    m_pixmapItem->setOffset(-m_pixmapItem->pixmap().width() / 2,
                            -m_pixmapItem->pixmap().height() / 2);

    setEnabled(false);
}

QRectF GPSLocationItem::boundingRect() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_boundingRect;
}

void GPSLocationItem::paint(QPainter *painter,
                            const QStyleOptionGraphicsItem *option,
                            QWidget *widget)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(option);
    Q_UNUSED(widget);

    QColor color = QColor(Qt::red);

    const int RING_OUTLINE_ALPHA = 64;
    const int RING_FILL_ALPHA = 32;

    // set outline ring
    color.setAlpha(RING_OUTLINE_ALPHA);
    QPen pen = QPen(color);
    pen.setWidthF(RING_OUTLINE_PEN_WIDTH);
    pen.setCosmetic(true);
    painter->setPen(pen);

    // set ring fill
    color.setAlpha(RING_FILL_ALPHA);
    painter->setBrush(color);

    painter->setRenderHint(QPainter::Antialiasing);

    QPointF center = QPointF();
    painter->drawEllipse(center, m_radius, m_radius);
}

void GPSLocationItem::setEnabled(bool enable)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (enable) {
        m_showOnNextUpdate = true;
    }
    else {
        m_showOnNextUpdate = false;
        hide();
    }
}

void GPSLocationItem::setAccuracyRingRadius(qreal accuracy, qreal sceneResolution)
{
    qDebug() << __PRETTY_FUNCTION__;

    qreal newRadius = accuracy / sceneResolution;

    if (m_radius != newRadius) {
        m_radius = newRadius;
        setBoundingRect(newRadius);
    }
}

void GPSLocationItem::setBoundingRect(qreal radius)
{
    qDebug() << __PRETTY_FUNCTION__;

    prepareGeometryChange();

    m_boundingRect = QRectF(-radius - RING_OUTLINE_PEN_WIDTH / 2,
                            -radius - RING_OUTLINE_PEN_WIDTH / 2,
                            2 * radius + RING_OUTLINE_PEN_WIDTH,
                            2 * radius + RING_OUTLINE_PEN_WIDTH);
}

void GPSLocationItem::updateItem(QPointF scenePosition, qreal accuracy, qreal sceneResolution)
{
    qDebug() << __PRETTY_FUNCTION__;

    setPos(scenePosition);
    setAccuracyRingRadius(accuracy, sceneResolution);

    if (m_showOnNextUpdate) {
        show();
        m_showOnNextUpdate = false;
    }

    update();
}
