/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <cmath>

#include <QDebug>
#include <QLineF>
#include <QPointF>
#include <QVariant>

#include "geocoordinate.h"
#include "map/osm.h"

#include "scenecoordinate.h"

SceneCoordinate::SceneCoordinate() :
        m_x(0),
        m_y(0)
{
    qDebug() << __PRETTY_FUNCTION__;
}

SceneCoordinate::SceneCoordinate(double x, double y) :
        m_x(x),
        m_y(y)
{
    qDebug() << __PRETTY_FUNCTION__;
}

SceneCoordinate::SceneCoordinate(const GeoCoordinate &coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    convertFrom(coordinate);
}

qreal SceneCoordinate::azimuthTo(const SceneCoordinate &to) const
{
    qDebug() << __PRETTY_FUNCTION__;

    // construct a line from this coordinate to target coordinate
    QLineF line = QLineF(this->toPointF(), to.toPointF());

    // get the angle from the line. Because QLineF::angle() returns positive value for a
    // counter-clockwise direction, and we want the positive value to be in clockwise direction,
    // the value is negated
    qreal angle = -line.angle();

    // QLineF::angle() returns value which has zero at the 3 o'clock position, and we want to have
    // the zero pointing to the north, so we have to add 90 degrees
    angle += 90;

    // QLineF::angle() returns values from -180 to 180, an we want only positive values from 0 to
    // 360 degrees, so full 360 degrees is added if the result would otherwise be negative
    if (angle < 0)
        angle += 360;

    Q_ASSERT_X(angle >= 0.0 && angle <= 360, "return value", "value is out of range");

    return angle;
}

void SceneCoordinate::convertFrom(const GeoCoordinate &coordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    // calculate x & y positions in the map (0..1)
    double worldX = static_cast<double>((coordinate.longitude() + 180.0) / 360.0);
    double worldY = static_cast<double>((1.0 - log(tan(coordinate.latitude() * M_PI / 180.0) + 1.0
                                / cos(coordinate.latitude() * M_PI / 180.0)) / M_PI) / 2.0);

    m_x = worldX * OSM_TILES_PER_SIDE * OSM_TILE_SIZE_X;
    m_y = worldY * OSM_TILES_PER_SIDE * OSM_TILE_SIZE_Y;

    normalize(m_x, OSM_MAP_MIN_PIXEL_X, OSM_MAP_MAX_PIXEL_X);
}

bool SceneCoordinate::isNull() const
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_x == 0 && m_y == 0)
        return true;

    return false;
}

void SceneCoordinate::normalize(double &value, int min, int max)
{
    qDebug() << __PRETTY_FUNCTION__;
    Q_ASSERT_X(max >= min, "parameters", "max can't be smaller than min");

    while (int(value) < min)
        value += max - min + 1;

    while (int(value) > max)
        value -= max - min + 1;
}

void SceneCoordinate::setX(double x)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_x = x;
}

void SceneCoordinate::setY(double y)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_y = y;
}

QPointF SceneCoordinate::toPointF() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return QPointF(m_x, m_y);
}

double SceneCoordinate::x() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_x;
}

double SceneCoordinate::y() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_y;
}

SceneCoordinate::operator QVariant() const
{
    return QVariant::fromValue(*this);
}

QDebug operator<<(QDebug dbg, const SceneCoordinate &coordinate)
{
    dbg.nospace() << "(" << coordinate.x() << ", " << coordinate.y() << ")";

    return dbg.space();
}

SceneCoordinate & SceneCoordinate::operator*=(double factor)
{
    m_x *= factor;
    m_y *= factor;
    return *this;
}

SceneCoordinate & SceneCoordinate::operator+=(const SceneCoordinate &coordinate)
{
    m_x += coordinate.x();
    m_y += coordinate.y();
    return *this;
}

SceneCoordinate & SceneCoordinate::operator-=(const SceneCoordinate &coordinate)
{
    m_x -= coordinate.x();
    m_y -= coordinate.y();
    return *this;
}

const SceneCoordinate operator* (double factor, const SceneCoordinate &coordinate)
{
    return SceneCoordinate(coordinate) *= factor;
}

const SceneCoordinate SceneCoordinate::operator+(const SceneCoordinate &other) const
{
  return SceneCoordinate(*this) += other;
}

const SceneCoordinate SceneCoordinate::operator-(const SceneCoordinate &other) const
{
  return SceneCoordinate(*this) -= other;
}
