/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#ifndef GEOCOORDINATE_H
#define GEOCOORDINATE_H

#include <QDebug>
#include <QMetaType>

class SceneCoordinate;

/**
* @brief Geographic coordinate
*
* @author Jussi Laitinen - jussi.laitinen@ixonos.com
* @author Sami Rämö - sami.ramo@ixonos.com
*/
class GeoCoordinate
{
public:
    /**
    * @brief Constructs a null coordinate
    */
    GeoCoordinate();

    /**
    * @brief Constructs a coordinate with given latitude and longitude values
    *
    * @param latitude Latitude value
    * @param longitude Longitude value
    */
    GeoCoordinate(double latitude, double longitude);

    /**
    * @brief Constructs a coordinate with values converted from the given SceneCoordinate
    *
    * @param coordinate Scene coordinate
    */
    GeoCoordinate(const SceneCoordinate &coordinate);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
    * @brief Distance to other coordinate
    *
    * This calculation returns the great-circle distance between the two coordinates, with an
    * assumption that the Earth is spherical. Calculation is done using haversine formula. Altitude
    * is not used in the calculation.
    *
    * @param other Coordinate where the distance is calculated from this coordinate
    * @returns Distance to other coordinate (in meters)
    */
    qreal distanceTo(const GeoCoordinate &other) const;

    /**
    * @brief Check if coordinate is (0.0, 0.0)
    *
    * @returns True if both latitude and longitude are 0.0, otherwise false
    */
    bool isNull() const;

    /**
    * @brief Check if coordinate is valid.
    *
    * Latitude and longitude values must be set, latitude must be -90..90 and longitude must
    * be -180..180 for valid coordinate.
    *
    * @return true if coordinate is valid, false otherwise
    */
    bool isValid();

    /**
    * @brief Returns the latitude value
    *
    * @returns latitude
    */
    double latitude() const;

    /**
    * @brief Returns the longitude value
    *
    * @returns longitude
    */
    double longitude() const;

    /**
    * @brief Sets the latitude
    *
    * @param latitude Latitude value
    */
    void setLatitude(double latitude);

    /**
    * @brief Sets the longitude
    *
    * @param longitude Longitude value
    */
    void setLongitude(double longitude);

private:
    /**
     * @brief Convert values from SceneCoordinate
     *
     * @param coordinate Scene coordinate
     */
    void convertFrom(const SceneCoordinate &coordinate);

    /**
    * @brief Register meta type and stream operators for using the class with QVariant and QSetting
    *
    * Registering is done only once at the first time the GeoCoordinate object is constructed.
    */
    void registerMetaType();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    /**
    * @enum ValidityFlag
    *
    * @brief Flag values for coordinate validity
    */
    enum ValidityFlag {
        NoCoordinatesSet = 0x0,
        LatitudeSet = 0x1,
        LongitudeSet = 0x2
    };

    static bool m_metaTypeIsRegistered;     ///< Is the meta type already registered?
    double m_latitude;                      ///< Latitude value
    double m_longitude;                     ///< Longitude value
    int m_validityFlags;                    ///< Coordinate validity flags

/*******************************************************************************
 * OPERATORS
 ******************************************************************************/
public:
    /**
    * @brief Subtract operator
    *
    * @returns Returns a GeoCoordinate object that is formed by subtracting coordinate2 from
    *          coordinate1. Each component is subtracted separately.
    */
    friend const GeoCoordinate operator-(const GeoCoordinate &coordinate1,
                                         const GeoCoordinate &coordinate2);

    /**
    * @brief Output operator
    *
    * @param out Output stream
    * @param coordinate Coordinate object which is written to the stream
    */
    friend QDataStream &operator<<(QDataStream &out, const GeoCoordinate &coordinate);

    /**
    * @brief Input operator
    *
    * @param in Input stream
    * @param coordinate Object where the values from the stream are saved
    */
    friend QDataStream &operator>>(QDataStream &in, GeoCoordinate &coordinate);
};

QDebug operator<<(QDebug dbg, const GeoCoordinate &coordinate);

Q_DECLARE_METATYPE(GeoCoordinate)

#endif // GEOCOORDINATE_H
