/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QGraphicsScene>
#include <QGraphicsView>
#include <QtTest/QtTest>

#include "common.h"
#include "coordinates/scenecoordinate.h"
#include "map/mapcommon.h"
#include "map/maptile.h"

#include "map/mapengine.h"

const int ZOOM_LEVEL = 14;
const SceneCoordinate DEFAULT_TEST_SCENECOORDINATE = SceneCoordinate(12345, 54321);
const SceneCoordinate NEW_TEST_SCENECOORDINATE = SceneCoordinate(50000, 40000);
const int MAP_FETCH_DELAY = 1000;

class TestMapEngine: public QObject
{
    Q_OBJECT

private slots:
    void convertTileNumberToSceneCoordinate();
    void convertTileNumberToSceneCoordinate_data();
    void setLocationNewTilesCount();
    void setLocationRemovedTilesCount();
    void usingLastLocation();
    void zoomInRemovedTilesCount();
    void zoomOutRemovedTilesCount();

signals:
    void setCenterPosition(SceneCoordinate); // a way to call a private slot
};

/**
* @brief Test converting tile numbers to scene coordinates
*
* Different zoom levels are also tested
*/
void TestMapEngine::convertTileNumberToSceneCoordinate()
{
    QFETCH(int, zoomLevel);
    QFETCH(QPoint, tileNumber);
    QFETCH(SceneCoordinate, expected);

    SceneCoordinate result = MapTile::convertTileNumberToSceneCoordinate(zoomLevel, tileNumber);
    QCOMPARE(result.x(), expected.x());
    QCOMPARE(result.y(), expected.y());
}

void TestMapEngine::convertTileNumberToSceneCoordinate_data()
{
    QTest::addColumn<int>("zoomLevel");
    QTest::addColumn<QPoint>("tileNumber");
    QTest::addColumn<SceneCoordinate>("expected");

    QTest::newRow("data1") << 18 << QPoint(0, 0) << SceneCoordinate(0, 0);
    QTest::newRow("data2") << 18 << QPoint(1, 2) << SceneCoordinate(256, 512);
    QTest::newRow("data3") << 16 << QPoint(3, 4) << SceneCoordinate(3072, 4096);
}

void TestMapEngine::setLocationNewTilesCount()
{
    MapEngine engine;
    QGraphicsView view;
    view.setScene(engine.scene());
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(SceneCoordinate)),
            &engine, SLOT(setCenterPosition(SceneCoordinate)));

    // set initial state for map engine
    engine.setZoomLevel(ZOOM_LEVEL);
    emit setCenterPosition(SceneCoordinate(0, 0));
    QTest::qWait(MAP_FETCH_DELAY);

    QSignalSpy fetchImageSpy(&engine, SIGNAL(fetchImage(int,int,int)));

    // first test, target location in the scene is empty so all tiles should be downloaded
    emit setCenterPosition(SceneCoordinate(1220*16, 1220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QCOMPARE(fetchImageSpy.count(), 5*4);
    fetchImageSpy.clear();

    // move one tile to east and south, only most right one column and most bottom one row tiles
    // should be downloaded
    emit setCenterPosition(SceneCoordinate((1220+OSM_TILE_SIZE_X)*16, (1220+OSM_TILE_SIZE_Y)*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QCOMPARE(fetchImageSpy.count(), 4 + 4);
    fetchImageSpy.clear();
}

void TestMapEngine::setLocationRemovedTilesCount()
{
    MapEngine engine;
    QGraphicsView view;
    view.setScene(engine.scene());
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(SceneCoordinate)),
            &engine, SLOT(setCenterPosition(SceneCoordinate)));

    engine.setZoomLevel(OSM_MAX_ZOOM_LEVEL);

    const int maxItemsCount = 40;

    emit setCenterPosition(SceneCoordinate(1220*16, 1220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    emit setCenterPosition(SceneCoordinate(2220*16, 2220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(SceneCoordinate(520*16, 2220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    emit setCenterPosition(SceneCoordinate(2220*16, 520*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::zoomInRemovedTilesCount()
{
    MapEngine engine;
    QGraphicsView view;
    view.setScene(engine.scene());
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(SceneCoordinate)),
            &engine, SLOT(setCenterPosition(SceneCoordinate)));

    engine.setZoomLevel(OSM_MAX_ZOOM_LEVEL);

    const int maxItemsCount = 40;

    emit setCenterPosition(SceneCoordinate(1220*16, 1220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(SceneCoordinate(520*16, 2220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    emit setCenterPosition(SceneCoordinate(2220*16, 520*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::zoomOutRemovedTilesCount()
{
    MapEngine engine;
    QGraphicsView view;
    view.setScene(engine.scene());
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(SceneCoordinate)),
            &engine, SLOT(setCenterPosition(SceneCoordinate)));

    engine.setZoomLevel(OSM_MAX_ZOOM_LEVEL);

    const int maxItemsCount = 40;

    emit setCenterPosition(SceneCoordinate(1220*16, 1220.23*16));
    QTest::qWait(MAP_FETCH_DELAY);
    emit setCenterPosition(SceneCoordinate(2220*16, 2220.23*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(SceneCoordinate(520*16, 2220*16));
    QTest::qWait(MAP_FETCH_DELAY);
    emit setCenterPosition(SceneCoordinate(2220*16, 520*16));
    QTest::qWait(MAP_FETCH_DELAY);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::usingLastLocation()
{
    // Create mapengine and start monitoring zoomLevelChanged-signal
    MapEngine *mapengine = new MapEngine;
    QGraphicsView view;
    view.setScene(mapengine->scene());
    QSignalSpy mapEngineSpy(mapengine, SIGNAL(zoomLevelChanged(int)));
    connect(this, SIGNAL(setCenterPosition(SceneCoordinate)),
            mapengine, SLOT(setCenterPosition(SceneCoordinate)));
    QVERIFY (mapEngineSpy.isValid());
    QCOMPARE (mapEngineSpy.count(), 0);

    // Write new zoomlevel and location to settings
    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(MAP_LAST_ZOOMLEVEL, ZOOM_LEVEL);
    settings.setValue(MAP_LAST_POSITION,
                      QVariant::fromValue(GeoCoordinate(DEFAULT_TEST_SCENECOORDINATE)));

    // Call mapengines init() and verify that signal will be send
    mapengine->init();
    QCOMPARE (mapEngineSpy.count(), 1);

    // Remove zoomlevel and location from settings, call init() again and check that
    // signals are sent again
    settings.remove(MAP_LAST_ZOOMLEVEL);
    settings.remove(MAP_LAST_POSITION);
    mapengine->init();
    QCOMPARE(mapEngineSpy.count(), 2);

    // Check parameters of sended signals
    QList<QVariant> parameters = mapEngineSpy.takeFirst();
    QVERIFY(parameters.at(0).toInt() == ZOOM_LEVEL);

    // Call set location with know parameter to get location changed
    // Store new location and zoomlevel to settings
    emit setCenterPosition(NEW_TEST_SCENECOORDINATE);
    delete mapengine;

    // Read settings and verify that zoomlevel is correct
    GeoCoordinate location =
        settings.value(MAP_LAST_POSITION, ERROR_VALUE_NOT_FOUND_ON_SETTINGS).value<GeoCoordinate>();
    QCOMPARE(location.latitude(), GeoCoordinate(NEW_TEST_SCENECOORDINATE).latitude());
    QCOMPARE(location.longitude(), GeoCoordinate(NEW_TEST_SCENECOORDINATE).longitude());
}

QTEST_MAIN(TestMapEngine)
#include "testmapengine.moc"
