/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtTest/QtTest>
#include <QGraphicsScene>

#include "map/mapcommon.h"
#include "map/friendgroupitem.h"
#include "map/friendlocationitem.h"

class TestFriendGroupItem: public QObject
{
    Q_OBJECT

private slots:
    void initTestCase(); // before first test
    void cleanupTestCase(); // after last test have been run
    void init(); // before each test
    void cleanup(); // after each test

    void addFriends();
    void constructor();
    void dropFriends();

private:
    FriendLocationItem *friend1;
    FriendLocationItem *friend2;
    FriendLocationItem *friend3;
    FriendLocationItem *friend4;
    FriendGroupItem *group;
    QPixmap *pixmap;
    QUrl url;
    QGraphicsScene scene;
};

void TestFriendGroupItem::initTestCase()
{
    pixmap = new QPixmap("friend_group.png");
    url = QUrl("http://www.something.com/pict.png");
}

void TestFriendGroupItem::cleanupTestCase()
{
    delete pixmap;
}

void TestFriendGroupItem::init()
{
    QString userID("ABC123");

    // create test friends
    friend1 = new FriendLocationItem(userID, this);
    friend2 = new FriendLocationItem(userID, this);
    friend3 = new FriendLocationItem(userID, this);
    friend4 = new FriendLocationItem(userID, this);
    QVERIFY(friend1 != 0);
    QVERIFY(friend2 != 0);
    QVERIFY(friend3 != 0);
    QVERIFY(friend4 != 0);

    // set pixmaps
    friend1->setProfileImage(*pixmap, url);
    friend2->setProfileImage(*pixmap, url);
    friend3->setProfileImage(*pixmap, url);
    friend4->setProfileImage(*pixmap, url);

    // add friends to scene
    scene.addItem(friend1);
    scene.addItem(friend2);
    scene.addItem(friend3);
    scene.addItem(friend4);

    // override coordinate positions with scene pixel positions so testing is more
    // convenient
    const int ITEM_SIDE = 58; // defined in baselocationitem.cpp
    friend1->setPos(100, 100);
    friend2->setPos(friend1->pos() + QPoint(0, ITEM_SIDE)); // east from friend1
    friend3->setPos(friend1->pos() + QPoint(ITEM_SIDE, 0)); // south from friend1
    friend4->setPos(friend1->pos() + QPoint(ITEM_SIDE, ITEM_SIDE)); // south-east from friend1

    // create test group
    group = new FriendGroupItem(friend1);
    QVERIFY(group != 0);
    QVERIFY(friend1->isPartOfGroup());
}

void TestFriendGroupItem::cleanup()
{
    delete group;

    // remove and delete all items from the scene
    scene.clear();
}

void TestFriendGroupItem::addFriends()
{
    group->joinFriend(friend2);
    group->joinFriend(friend3);
    QCOMPARE(friend2->isPartOfGroup(), true);
    QCOMPARE(friend3->isPartOfGroup(), true);
    QCOMPARE(friend4->isPartOfGroup(), false); // not yet part of group

    // group position should still be same as first group members
    QCOMPARE(group->pos(), QPointF(100, 100));
}

void TestFriendGroupItem::constructor()
{
    // group picture should be set
    QVERIFY(!group->pixmap().isNull());

    // group position should be same as first group members
    QCOMPARE(group->pos(), QPointF(100, 100));

    // zValue should be set
    QCOMPARE(group->zValue(), static_cast<qreal>(FriendLocationItemZValue));

    // icon offset should be set
    QCOMPARE(group->offset(), QPointF(-pixmap->width() / 2, -pixmap->height() / 2));
}

void TestFriendGroupItem::dropFriends()
{
    group->joinFriend(friend2);
    group->joinFriend(friend3);
    group->joinFriend(friend4);

    QCOMPARE(friend1->isPartOfGroup(), true);
    QCOMPARE(friend2->isPartOfGroup(), true);
    QCOMPARE(friend3->isPartOfGroup(), true);
    QCOMPARE(friend4->isPartOfGroup(), true);

    group->dropFriends();

    // no-one should be part of group anymore
    QCOMPARE(friend1->isPartOfGroup(), false);
    QCOMPARE(friend2->isPartOfGroup(), false);
    QCOMPARE(friend3->isPartOfGroup(), false);
    QCOMPARE(friend4->isPartOfGroup(), false);
}

QTEST_MAIN(TestFriendGroupItem)

#include "testfriendgroupitem.moc"
