/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtCore/QString>
#include <QtTest/QtTest>

#include "coordinates/scenecoordinate.h"
#include "map/mapcommon.h"

#include "coordinates/geocoordinate.h"

const double LATITUDE = 12.345678;
const double LONGITUDE = -89.765432;

const double ONE_SCENE_PIXEL_WIDTH_IN_DEGREES = 0.00000536441802978516;

class TestGeoCoordinate : public QObject
{
    Q_OBJECT

private Q_SLOTS:
    void constructors();
    void conversion();
    void conversion_data();
    void distanceTo();
    void distanceTo_data();
    void isNullAndIsValid();
    void isValid();
    void isValid_data();
    void settersAndGetters();
    void streamOperators();
    void subtractOperator();
};

// for formatting the output of double valuest into the test log
namespace QTest {
    template<>
    char *toString(const double &number)
    {
        QByteArray ba;
        ba += QByteArray::number(number, 'f', 9);
        return qstrdup(ba.data());
    }
}

void TestGeoCoordinate::constructors()
{
    GeoCoordinate coordinate;
    QVERIFY(coordinate.isNull());

    GeoCoordinate coordinate2(LATITUDE, LONGITUDE);
    QCOMPARE(coordinate2.latitude(), LATITUDE);
    QCOMPARE(coordinate2.longitude(), LONGITUDE);

    // NOTE: constructor with conversion from GeoCoordinate is tested in conversion() test slot
}

void TestGeoCoordinate::conversion()
{
    // allow rounding error of one tenth of one scene pixel
    const double MAX_ERROR = ONE_SCENE_PIXEL_WIDTH_IN_DEGREES * 0.1;

    QFETCH(SceneCoordinate, sceneCoordinate);
    QFETCH(GeoCoordinate, result);

    GeoCoordinate geoCoordinate(sceneCoordinate);

    QVERIFY(qAbs(geoCoordinate.latitude() - result.latitude()) < MAX_ERROR);
    QVERIFY(qAbs(geoCoordinate.longitude() - result.longitude()) < MAX_ERROR);
}

void TestGeoCoordinate::conversion_data()
{
    QTest::addColumn<SceneCoordinate>("sceneCoordinate");
    QTest::addColumn<GeoCoordinate>("result");

    QTest::newRow("top left") << SceneCoordinate(OSM_MAP_MIN_PIXEL_X, OSM_MAP_MIN_PIXEL_Y)
                              << GeoCoordinate(OSM_MAX_LATITUDE, MIN_LONGITUDE);

    const double LAST_SCENE_HORIZONTAL_PIXEL_LONGITUDE = MAX_LONGITUDE
                                                         - ONE_SCENE_PIXEL_WIDTH_IN_DEGREES;

    QTest::newRow("bottom right") << SceneCoordinate(OSM_MAP_MAX_PIXEL_X, OSM_MAP_MAX_PIXEL_Y)
                                  << GeoCoordinate(-OSM_MAX_LATITUDE,
                                                   LAST_SCENE_HORIZONTAL_PIXEL_LONGITUDE);
}

void TestGeoCoordinate::distanceTo()
{
    QFETCH(GeoCoordinate, from);
    QFETCH(GeoCoordinate, to);
    QFETCH(qreal, expectedDistance);

    QCOMPARE(from.distanceTo(to), expectedDistance);
}

void TestGeoCoordinate::distanceTo_data()
{
    const qreal EARTH_RADIUS = 6371010; // in meters

    QTest::addColumn<GeoCoordinate>("from");
    QTest::addColumn<GeoCoordinate>("to");
    QTest::addColumn<qreal>("expectedDistance");

    QTest::newRow("longitude") << GeoCoordinate(0, -90)
                               << GeoCoordinate(0, 90)
                               << M_PI * 2 * EARTH_RADIUS / 2;

    QTest::newRow("latitude") << GeoCoordinate(-45, 0)
                              << GeoCoordinate(45, 0)
                              << M_PI * 2 * EARTH_RADIUS / 4;

    QTest::newRow("both") << GeoCoordinate(-25, -135)
                          << GeoCoordinate(25, 45)
                          << M_PI * 2 * EARTH_RADIUS / 2;
}


void TestGeoCoordinate::isNullAndIsValid()
{
    // coordinate created with default constructor
    GeoCoordinate coordinate;
    QVERIFY(coordinate.isNull());
    QVERIFY(!coordinate.isValid());

    // only latitude is set
    coordinate.setLatitude(LATITUDE);
    QVERIFY(!coordinate.isNull());
    QVERIFY(!coordinate.isValid());

    // another coordinate created with default constructor
    GeoCoordinate coordinate2;
    QVERIFY(coordinate2.isNull());
    QVERIFY(!coordinate2.isValid());

    // only longitude is set
    coordinate2.setLongitude(LONGITUDE);
    QVERIFY(!coordinate2.isNull());
    QVERIFY(!coordinate2.isValid());

    // also latitude is set
    coordinate2.setLatitude(LATITUDE);
    QVERIFY(!coordinate2.isNull());
    QVERIFY(coordinate2.isValid());

    // null coordinate created with latitude and longitude parameters
    GeoCoordinate coordinate3(0, 0);
    QVERIFY(coordinate3.isNull());
    QVERIFY(coordinate3.isValid());

    // non-null coordinate created from scene coordinate
    GeoCoordinate coordinate4 = GeoCoordinate(SceneCoordinate(1, 1));
    QVERIFY(!coordinate4.isNull());
    QVERIFY(coordinate4.isValid());

    // non-null coordinate created with latitude and longitude parameters
    GeoCoordinate coordinate5(LATITUDE, LONGITUDE);
    QVERIFY(!coordinate5.isNull());
    QVERIFY(coordinate5.isValid());
}

void TestGeoCoordinate::isValid()
{
    QFETCH(GeoCoordinate, coordinate);
    QFETCH(bool, expectedValidity);

    QCOMPARE(coordinate.isValid(), expectedValidity);
}

void TestGeoCoordinate::isValid_data()
{
    QTest::addColumn<GeoCoordinate>("coordinate");
    QTest::addColumn<bool>("expectedValidity");

    QTest::newRow("null") << GeoCoordinate(0, 0) << true;
    QTest::newRow("min latitude") << GeoCoordinate(-90, 0) << true;
    QTest::newRow("latitude too small") << GeoCoordinate(-90.000000001, 0) << false;
    QTest::newRow("max latitude") << GeoCoordinate(90, 0) << true;
    QTest::newRow("latitude too big") << GeoCoordinate(90.000000001, 0) << false;
    QTest::newRow("min longitude") << GeoCoordinate(0, -180) << true;
    QTest::newRow("longitude too small") << GeoCoordinate(0, -180.0000000001) << false;
    QTest::newRow("max longitude") << GeoCoordinate(0, 180) << true;
    QTest::newRow("longitude too big") << GeoCoordinate(0, 180.0000000001) << false;
}

void TestGeoCoordinate::settersAndGetters()
{
    GeoCoordinate coordinate;
    QCOMPARE(coordinate.latitude(), (double)0);
    QCOMPARE(coordinate.longitude(), (double)0);

    coordinate.setLatitude(LATITUDE);
    coordinate.setLongitude(LONGITUDE);

    QCOMPARE(coordinate.latitude(), LATITUDE);
    QCOMPARE(coordinate.longitude(), LONGITUDE);
}

void TestGeoCoordinate::streamOperators()
{
    // test data
    GeoCoordinate original(LATITUDE, LONGITUDE);
    GeoCoordinate originalInverted(LONGITUDE, LATITUDE);

    // create data output stream and write coordinates to it
    QByteArray array;
    QDataStream outStream(&array,QIODevice::WriteOnly);
    outStream << original << originalInverted;

    // use another stream with same byte array for reading the values
    QDataStream inStream(&array,QIODevice::ReadOnly);
    GeoCoordinate another;
    GeoCoordinate anotherInverted;
    inStream >> another >> anotherInverted;

    QCOMPARE(another.latitude(), original.latitude());
    QCOMPARE(another.longitude(), original.longitude());
    QCOMPARE(anotherInverted.latitude(), originalInverted.latitude());
    QCOMPARE(anotherInverted.longitude(), originalInverted.longitude());
}

void TestGeoCoordinate::subtractOperator()
{
    // test operator-
    GeoCoordinate first(100, 50);
    GeoCoordinate second(45, 15);

    GeoCoordinate result = first - second;

    QCOMPARE(result.latitude(), 55.0);
    QCOMPARE(result.longitude(), 35.0);

    // make sure the returned object is not either of the parameters
    QVERIFY(&result != &first);
    QVERIFY(&result != &second);
}

QTEST_APPLESS_MAIN(TestGeoCoordinate);

#include "testgeocoordinate.moc"
