/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QAbstractButton>
#include <QButtonGroup>
#include <QDebug>
#include <QPropertyAnimation>
#include <QRegion>
#include <QSignalTransition>
#include <QStackedWidget>
#include <QStateMachine>

#include "listitemcontextbuttonbar.h"
#include "panelbar.h"
#include "panelbase.h"
#include "panelcontentstack.h"
#include "panelcontextbuttonbar.h"
#include "paneltabbar.h"

#include "tabbedpanel.h"

const int PANEL_CONTEXT_BUTTON_BAR_LEFT_X = 1;
const int PANEL_TAB_BAR_TOP_SPACING = 8;

TabbedPanel::TabbedPanel(QWidget *parent)
    : QWidget(parent),
      m_open(false),
      m_closeRequestPending(false)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int PANEL_LEFT_X = 0;
    const int PANEL_TOP_Y = 0;

    resize(PANEL_BAR_TABBED_WIDTH + PANEL_WIDTH, PANEL_HEIGHT);
    move(PANEL_CLOSED_X, PANEL_TOP_PADDING);

    // --- TABS ---
    m_panelTabBar = new PanelTabBar(this);
    m_panelTabBar->move(PANEL_LEFT_X, PANEL_TAB_BAR_TOP_SPACING);

    connect(m_panelTabBar, SIGNAL(currentChanged(int)),
            this, SLOT(setCurrentIndex(int)));

    connect(m_panelTabBar, SIGNAL(sizeChangeRequested()),
            this, SLOT(calculateMask()));

    connect(m_panelTabBar, SIGNAL(tabCloseRequested(int)),
             this, SLOT(closePanel()));

    connect(this, SIGNAL(panelClosed()),
            m_panelTabBar, SLOT(deselectTabs()));

    // --- BAR ---
    m_panelBar = new PanelBar(this);
    m_panelBar->move(PANEL_TAB_BAR_WIDTH, PANEL_TOP_Y);

    // --- GENERIC PANEL CONTEXT BUTTON BAR ---
    m_panelContextButtonBar = new PanelContextButtonBar(this);
    m_panelContextButtonBar->move(PANEL_CONTEXT_BUTTON_BAR_LEFT_X, PANEL_HEIGHT);

    connect(m_panelContextButtonBar, SIGNAL(barHidden()),
            this, SLOT(closePanel()));

    connect(m_panelContextButtonBar, SIGNAL(positionChangeRequested()),
            this, SLOT(repositionContextButtonBar()));

    // --- LIST ITEM RELATED CONTEXT BUTTONS BAR ---
    m_itemContextButtonBar = new ListItemContextButtonBar(this);
    m_itemContextButtonBar->hide();

    connect(this, SIGNAL(listItemSelectionChanged(bool)),
            m_itemContextButtonBar, SLOT(onListItemSelectionChanged(bool)));

    // --- PANEL CONTENT ---
    m_panelContentStack = new PanelContentStack(this);
    m_panelContentStack->move(PANEL_TAB_BAR_WIDTH + PANEL_BAR_WIDTH, PANEL_TOP_Y);
    m_panelContentStack->stackUnder(m_itemContextButtonBar);

    // --- PANEL ANIMATION ---
    QStateMachine *panelStateMachine = new QStateMachine(this);

    m_stateClosed = new QState(panelStateMachine);
    m_stateOpened = new QState(panelStateMachine);

    QPropertyAnimation *panelAnimation = new QPropertyAnimation(this, "pos", this);

    panelStateMachine->setInitialState(m_stateClosed);

    QSignalTransition *panelTransitionOpen = m_stateClosed->addTransition(this,
                                                                          SIGNAL(toggleState()),
                                                                          m_stateOpened);
    panelTransitionOpen->addAnimation(panelAnimation);

    QSignalTransition *panelTransitionClose = m_stateOpened->addTransition(this,
                                                                           SIGNAL(toggleState()),
                                                                           m_stateClosed);
    panelTransitionClose->addAnimation(panelAnimation);

    connect(panelAnimation, SIGNAL(finished()),
            this, SLOT(stateChanged()));

    QPoint closedPosition(PANEL_CLOSED_X, PANEL_TOP_PADDING);
    m_stateClosed->assignProperty(this, "pos", closedPosition);

    QPoint openedPosition(PANEL_OPENED_X, PANEL_TOP_PADDING);
    m_stateOpened->assignProperty(this, "pos", openedPosition);

    panelStateMachine->start();
}

int TabbedPanel::addTab(QWidget *widget, const QIcon& icon)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int APPEND_INDEX = -1;

    return insertTab(APPEND_INDEX, widget, icon);
}

void TabbedPanel::calculateMask()
{
    qDebug() << __PRETTY_FUNCTION__;

    QRect panelTabBarRect = m_panelTabBar->rect();
    QRect panelContextButtonBarRect = m_panelContextButtonBar->rect();
    int panelContextButtonBarY = height() - panelContextButtonBarRect.height();

    if (!m_open)
        panelContextButtonBarY = height();

    QRegion panelTabBarRegion(0, PANEL_TAB_BAR_TOP_SPACING,
                         panelTabBarRect.width(), panelTabBarRect.height());
    QRegion panelContextButtonBarRegion(0, panelContextButtonBarY,
                                      panelContextButtonBarRect.width(),
                                      panelContextButtonBarRect.height());
    QRegion panelContentRegion(panelTabBarRect.right() + 1, 0,
                               PANEL_WIDTH + PANEL_BAR_WIDTH, height());
    QRegion panelRegion = panelTabBarRegion + panelContentRegion + panelContextButtonBarRegion;

    setMask(panelRegion);
}

void TabbedPanel::closePanel()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_open) {
        m_open = false;

        if (m_panelContextButtonBar->isBarVisible()) {
            m_closeRequestPending = true;
            m_panelContextButtonBar->hideContextButtonBar();
        } else {
            emit toggleState();
        }
    } else if (m_closeRequestPending) {
        m_closeRequestPending = false;
        emit toggleState();
    }
}

int TabbedPanel::insertTab(int index, QWidget *widget, const QIcon& icon)
{
    qDebug() << __PRETTY_FUNCTION__;

    index = m_panelContentStack->insertWidget(index, widget);
    m_panelTabBar->insertTab(index, icon);

    return index;
}

void TabbedPanel::openPanel(QWidget *widget)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (widget) {
        m_panelTabBar->selectTab(m_panelContentStack->indexOf(widget));
    } else if (!m_open) {
        if (!m_closeRequestPending) {
            m_open = true;
            emit toggleState();
        }
    }
}

void TabbedPanel::removeTab(int index)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (QWidget *widget = m_panelContentStack->widget(index)) {
        m_panelContentStack->removeWidget(widget);
        m_panelTabBar->removeTab(index);
    }
}

void TabbedPanel::repositionContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_panelContextButtonBar->move(PANEL_CONTEXT_BUTTON_BAR_LEFT_X, height());

    calculateMask();
}

void TabbedPanel::resizePanel(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    resize(PANEL_BAR_TABBED_WIDTH + PANEL_WIDTH,
           size.height() - PANEL_TOP_PADDING - PANEL_BOTTOM_PADDING);

    if (!m_open)
        move(size.width() - PANEL_TAB_BAR_WIDTH - PANEL_BAR_WIDTH, PANEL_TOP_PADDING);
    else
        move(size.width() - PANEL_TAB_BAR_WIDTH - PANEL_BAR_WIDTH - PANEL_WIDTH, PANEL_TOP_PADDING);

    m_panelBar->resizeBar(size);

    m_panelContextButtonBar->move(PANEL_CONTEXT_BUTTON_BAR_LEFT_X, size.height());

    m_panelContentStack->resizeContentStack(size);

    QPoint closedPosition(size.width() - PANEL_TAB_BAR_WIDTH - PANEL_BAR_WIDTH, PANEL_TOP_PADDING);
    m_stateClosed->assignProperty(this, "pos", closedPosition);

    QPoint openedPosition(size.width() - PANEL_TAB_BAR_WIDTH - PANEL_BAR_WIDTH - PANEL_WIDTH,
                          PANEL_TOP_PADDING);
    m_stateOpened->assignProperty(this, "pos", openedPosition);

    calculateMask();
}

void TabbedPanel::setCurrentIndex(int index)
{
    qDebug() << __PRETTY_FUNCTION__;

    if ((index < m_panelContentStack->count()) && (index >= 0)) {
        m_panelContentStack->setCurrentIndex(index);

        if (!m_open)
            openPanel();

        m_panelContextButtonBar->setContextButtons(
                static_cast<PanelBase *>(m_panelContentStack->widget(index))->genericPanelButtons());

        QWidget *itemContextButtons = static_cast<PanelBase *>(m_panelContentStack->widget(index))->itemButtons();
        m_itemContextButtonBar->setContextButtons(itemContextButtons);

        emit currentChanged(index);
    }
}

void TabbedPanel::setTabsEnabled(const QList<int> &tabIndexes, bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;

    QButtonGroup *tabs = m_panelTabBar->tabs();

    foreach (int tabIndex, tabIndexes) {
        QAbstractButton *tabButton = tabs->button(tabIndex);

        if (tabButton) {
            if (tabButton->isChecked())
                closePanel();

            tabButton->setEnabled(enabled);
        }
    }
}

void TabbedPanel::stateChanged()
{
    qDebug() << __PRETTY_FUNCTION__;

    calculateMask();

    if (m_open) {
        m_panelContextButtonBar->showContextButtonBar();
        emit panelOpened();
    } else {
        emit panelClosed();
    }
}
