/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Pekka Nissinen - pekka.nissinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QPainter>
#include <QPropertyAnimation>
#include <QSignalTransition>
#include <QStateMachine>

#include "panelcontextbuttonbar.h"

const int PANEL_CONTEXT_BUTTON_BAR_WIDTH = 78;

PanelContextButtonBar::PanelContextButtonBar(QWidget *parent)
    : QWidget(parent),
      m_visible(false),
      m_contextButtons(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    setFixedWidth(PANEL_CONTEXT_BUTTON_BAR_WIDTH);

    m_barTile.load(":/res/images/panel_context_button_bar_tile.png");
    m_barTop.load(":/res/images/panel_context_button_bar_top.png");

    // --- BAR ANIMATION ---
    QStateMachine *stateMachine = new QStateMachine(this);

    m_stateHidden = new QState(stateMachine);
    m_stateVisible = new QState(stateMachine);

    QPropertyAnimation *animation = new QPropertyAnimation(this, "pos", this);

    stateMachine->setInitialState(m_stateHidden);

    QSignalTransition *showSignalTransition = m_stateHidden->addTransition(this,
                                                                           SIGNAL(toggleState()),
                                                                           m_stateVisible);
    showSignalTransition->addAnimation(animation);

    QSignalTransition *hideSignalTransition = m_stateVisible->addTransition(this,
                                                                            SIGNAL(toggleState()),
                                                                            m_stateHidden);
    hideSignalTransition->addAnimation(animation);

    connect(animation, SIGNAL(finished()),
            this, SLOT(contextButtonBarStateChanged()));

    QPoint hiddenPosition(pos().x(), pos().y());
    m_stateHidden->assignProperty(this, "pos", hiddenPosition);

    QPoint visiblePosition(pos().x(), pos().y() - height());
    m_stateVisible->assignProperty(this, "pos", visiblePosition);

    stateMachine->start();
}

void PanelContextButtonBar::contextButtonBarStateChanged()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_visible)
        emit barVisible();
    else
        emit barHidden();
}

void PanelContextButtonBar::hideContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_visible) {
        m_visible = false;
        emit toggleState();
    }
}

bool PanelContextButtonBar::isBarVisible() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_visible;
}

void PanelContextButtonBar::move(int x, int y)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_visible)
        QWidget::move(x, y - height());
    else
        QWidget::move(x, y);

    QPoint hiddenPosition(x, y);
    m_stateHidden->assignProperty(this, "pos", hiddenPosition);

    QPoint visiblePosition(x, y - height());
    m_stateVisible->assignProperty(this, "pos", visiblePosition);
}

void PanelContextButtonBar::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(event);

    const int CONTEXT_BUTTON_BAR_RECT_X = 0;
    const int CONTEXT_BUTTON_BAR_TOP_HEIGHT = 32;
    const int CONTEXT_BUTTON_BAR_TOP_X = 0;
    const int CONTEXT_BUTTON_BAR_TOP_Y = 0;

    QPainter painter(this);

    m_barRect.setRect(CONTEXT_BUTTON_BAR_RECT_X, CONTEXT_BUTTON_BAR_TOP_HEIGHT,
                      PANEL_CONTEXT_BUTTON_BAR_WIDTH, height() - CONTEXT_BUTTON_BAR_TOP_HEIGHT);

    painter.drawPixmap(CONTEXT_BUTTON_BAR_TOP_X, CONTEXT_BUTTON_BAR_TOP_Y, m_barTop);
    painter.drawTiledPixmap(m_barRect, m_barTile);
}

void PanelContextButtonBar::setContextButtons(QWidget *contextButtons)
{
    qDebug() << __PRETTY_FUNCTION__;

    // Hide previous buttons (if any)
    if (m_contextButtons)
        m_contextButtons->setParent(0);

    m_contextButtons = contextButtons;

    m_contextButtons->setParent(this);
    m_contextButtons->setVisible(true);

    setFixedHeight(m_contextButtons->height());

    emit positionChangeRequested();
}

void PanelContextButtonBar::showContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_visible) {
        m_visible = true;
        emit toggleState();
    }
}
