/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Pekka Nissinen - pekka.nissinen@ixonos.com
       Kaj Wallin - kaj.wallin@ixonos.com
       Katri Kaikkonen - katri.kaikkonen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef INDICATORBUTTONPANEL_H
#define INDICATORBUTTONPANEL_H

#include <QGraphicsItem>
#include <QGridLayout>
#include <QLabel>
#include <QMouseEvent>
#include <QTimer>
#include <QWidget>

#include "indicatorbutton.h"

/**
 * @brief Draggable panel for indicator button
 *
 * @author Katri Kaikkonen - katri.kaikkonen (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
 * @author Kaj Wallin - kaj.wallin (at) ixonos.com
 */
class IndicatorButtonPanel : public QWidget
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     *
     * @param parent Parent
     */
    IndicatorButtonPanel(QWidget *parent = 0);

    /**
     * @brief Desctructor
     * Destroys m_normalColor
     */
    ~IndicatorButtonPanel();

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
protected:
    /**
     * @brief Move event for the indicator button panel
     *
     * @param event Event
     */
    void mouseMoveEvent(QMouseEvent *event);

    /**
     * @brief Press event for the indicator button panel
     *
     * @param event Event
     */
    void mousePressEvent(QMouseEvent *event);

    /**
     * @brief Event handler for mouse release events
     *
     * @param event Mouse event
     */
    void mouseReleaseEvent(QMouseEvent *event);

    /**
     * @brief Event handler for paint events
     *
     * Paints the panel
     * @param event Paint event
     */
    void paintEvent(QPaintEvent *event);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:
    /**
     * @brief Called when direction and distance from current map center point to current GPS
     *        location is changed
     *
     * @param direction Direction in degrees
     * @param distance Distance in meters
     * @param draw Should the indicator triangle be drawn or not
     */
    void updateValues(qreal direction, qreal distance, bool draw);

    /**
     * @brief Set indicator button enabled.
     *
     * @param enabled true if enabled, false otherwise
     */
    void setIndicatorButtonEnabled(bool enabled);

    /**
     * @brief Toggle indicator button panel draggability
     *
     * @param mode True if draggable, otherwise false
     * @param eventPosition Start position of the dragging
     */
    void setDraggable(bool mode, QPoint eventPosition = QPoint(0,0));

    /**
     * @brief Slot to redraw the panel after window resize event
     *
     * @param size Size of the new screen
     */
    void screenResized(const QSize &size);

private slots:
    /**
     * @brief Safeguard slot to release mouse grab if something goes horribly wrong
     */
    void forceMouseRelease();

    /**
     * @brief Slot that handles drag initialization once timer has timed out
     */
    void timerExpired();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
     * @brief Automatic centering setting changed by user
     *
     * @param enabled True if automatic centering is enabled, otherwise false
     */
    void autoCenteringTriggered(bool enabled);

    /**
     * @brief Signal when direction and distance from current map center point to current GPS
     *        location is changed
     *
     * @param direction Direction in degrees
     * @param distance Distance in meters
     * @param draw Should the indicator triangle be drawn or not
     */
    void directionIndicatorValuesUpdate(qreal direction, qreal distance, bool draw);

    /**
     * @brief Dragging mode triggered.
     */
    void draggingModeTriggered();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_isDraggable;                     ///< Boolean for tracking the draggability state

    qreal m_distance;                       ///< Distance to the GPS position

    QColor *m_normalColor;                  ///< Normal background color

    QLabel *m_distanceTextLabel;            ///< Pointer to distanceTextLabel

    QPoint m_dragPosition;                  ///< Location from where the widget is grabbed

    QSize m_screenSize;                     ///< Store for the screen size

    QString m_distanceText;                 ///< Text description of the distance

    QTimer *m_dragStartTimer;               ///< Timer to init draggability of the zoom panel
    QTimer *m_forceReleaseTimer;            ///< Timer to run forceMouseRelease;

    IndicatorButton *m_indicatorButton;     ///< Indicator button
};

#endif // INDICATORBUTTONPANEL_H
