/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QFontMetrics>
#include <QPainter>

#include "../common.h"
#include "extendedlistitemstore.h"
#include "listcommon.h"
#include "textmodifier.h"

#include "extendedlistitem.h"

ExtendedListItem::ExtendedListItem()
    : m_selected(false),
      m_expandedHeight(ITEM_MIN_HEIGHT),
      m_normalHeight(ITEM_MIN_HEIGHT),
      m_subItemTextWidth(SUBITEM_TEXT_MAX_WIDTH)

{
    qDebug() << __PRETTY_FUNCTION__;

    setSize(QSize(ITEM_WIDTH, ITEM_MIN_HEIGHT));

    m_subItemStoreList = new QList<ExtendedListItemStore *>();
    setData(SUBITEM_STORE_INDEX, qVariantFromValue((void *) m_subItemStoreList));
}

ExtendedListItem::~ExtendedListItem()
{
    qDebug() << __PRETTY_FUNCTION__;

    clearSubItems();
    delete m_subItemStoreList;
}

void ExtendedListItem::addSubItem(const QString &text, const QPixmap &icon)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPixmap p = QPixmap(ICON_WIDTH, ICON_HEIGHT);
    QPainter painter(&p);
    painter.setFont(NOKIA_FONT_SMALL);
    QFontMetrics textMetrics = painter.fontMetrics();

    QString expandedText = TextModifier::splitLongWords(textMetrics, text, m_subItemTextWidth);

    ExtendedListItemStore *itemStore = new ExtendedListItemStore(expandedText);
    itemStore->setIcon(icon);
    itemStore->setShortenedText(shortenText(text, m_subItemTextWidth, ListItem::TEXT_SIZE_SMALL));
    itemStore->setTextRect(calculateExpandedTextRect(expandedText));

    m_subItemStoreList->append(itemStore);
}

QRect ExtendedListItem::boundingRect(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPixmap p = QPixmap(ICON_WIDTH, ICON_HEIGHT);
    QPainter painter(&p);
    painter.setFont(NOKIA_FONT_SMALL);

    QFontMetrics textMetrics = painter.fontMetrics();
    QRect textRect;
    int rowHeight = textMetrics.ascent() + textMetrics.descent();
    textRect.setSize(QSize(m_subItemTextWidth, rowHeight));

    QStringList words = text.split(QRegExp("\\b"));
    QString rowText;
    foreach (QString word, words) {

        if (word.contains("\n")) {
            foreach (QChar c, word) {
                if (c == '\n')
                    textRect.setHeight(textRect.height() + rowHeight);
            }
            rowText.clear();
        } else {
            rowText.append(word);
            if (textMetrics.width(rowText) > m_subItemTextWidth) {
                textRect.setHeight(textRect.height() + rowHeight);
                rowText = word.simplified();
            }
        }
    }

    return textRect;
}

QRect ExtendedListItem::calculateExpandedTextRect(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int TEXT_BOTTOM_MARGIN = 2;

    QRect expandedTextRect = boundingRect(text);

    m_normalHeight += SUBITEM_TEXT_ROW_HEIGHT + TEXT_BOTTOM_MARGIN;
    m_expandedHeight += expandedTextRect.height() + TEXT_BOTTOM_MARGIN;
    setSize(QSize(ITEM_WIDTH, m_normalHeight));

    return expandedTextRect;
}

void ExtendedListItem::clearSubItems()
{
    qDebug() << __PRETTY_FUNCTION__;

    qDeleteAll(m_subItemStoreList->begin(), m_subItemStoreList->end());
    m_subItemStoreList->clear();

    m_expandedHeight = ITEM_MIN_HEIGHT;
    m_normalHeight = ITEM_MIN_HEIGHT;
}

void ExtendedListItem::setSubitemTextWidth(int width)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_subItemTextWidth = width;
}

void ExtendedListItem::setSelected(bool selected)
{
    qDebug() << __PRETTY_FUNCTION__;

    QListWidgetItem::setSelected(selected);
    m_selected = selected;
    setData(ITEM_EXPANDED_INDEX, m_selected);

    if (m_selected)
        setData(ITEM_SIZE_HINT_INDEX, QSize(ITEM_WIDTH, m_expandedHeight));
    else
        setData(ITEM_SIZE_HINT_INDEX, QSize(ITEM_WIDTH, m_normalHeight));
}

bool ExtendedListItem::toggleSelection()
{
    qDebug() << __PRETTY_FUNCTION__;

    setSelected(!m_selected);
    return m_selected;
}
