/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

      Henri Lampela - henri.lampela@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include "parser.h"

#include <QtAlgorithms>
#include <QDebug>
#include <QNetworkReply>
#include <QPixmap>
#include <QStringList>
#include <QtGlobal>

#include "error.h"
#include "network/networkaccessmanager.h"
#include "situarecommon.h"
#include "ui/avatarimage.h"

#include "situareservice.h"

SituareService::SituareService(QObject *parent)
        : QObject(parent),
        m_user(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_networkManager = new NetworkAccessManager(this);
    connect(m_networkManager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)), Qt::QueuedConnection);

    m_imageFetcher = new ImageFetcher(new NetworkAccessManager(this), this);
    connect(this, SIGNAL(fetchImage(QUrl)),
            m_imageFetcher, SLOT(fetchImage(QUrl)));
    connect(m_imageFetcher, SIGNAL(imageReceived(QUrl,QPixmap)),
            this, SLOT(imageReceived(QUrl, QPixmap)));
    connect(m_imageFetcher, SIGNAL(error(int, int)),
            this, SIGNAL(error(int, int)));
}

SituareService::~SituareService()
{
    qDebug() << __PRETTY_FUNCTION__;

    if(m_user) {
        delete m_user;
        m_user = 0;
    }

    qDeleteAll(m_friendsList.begin(), m_friendsList.end());
    m_friendsList.clear();
}

void SituareService::fetchLocations()
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie = formCookie(API_KEY, m_credentials.expires(), m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);

    QUrl url = formUrl(SITUARE_URL, GET_LOCATIONS);
    sendRequest(url, COOKIE, cookie);
}

void SituareService::reverseGeo(const GeoCoordinate &coordinates)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie = formCookie(API_KEY, m_credentials.expires(),m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);

    QString urlParameters = formUrlParameters(coordinates);
    urlParameters.append(JSON_FORMAT);
    QUrl url = formUrl(SITUARE_URL, REVERSE_GEO, urlParameters);

    sendRequest(url, COOKIE, cookie);
}

void SituareService::updateLocation(const GeoCoordinate &coordinates, const QString &status,
                                    const bool &publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString urlParameters = formUrlParameters(coordinates, status, publish);
    QUrl url = formUrl(SITUARE_URL, UPDATE_LOCATION, urlParameters);

    QString cookie = formCookie(API_KEY, m_credentials.expires(), m_credentials.userID(),
                                m_credentials.sessionKey(), m_credentials.sessionSecret(),
                                m_credentials.sig(), EN_LOCALE);

    sendRequest(url, COOKIE, cookie);
}

QString SituareService::formCookie(const QString &apiKeyValue, QString expiresValue,
                                   QString userValue, QString sessionKeyValue,
                                   QString sessionSecretValue, const QString &signatureValue,
                                   const QString &localeValue)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString cookie;
    QString apiKey;
    QString user;
    QString expires;
    QString sessionKey;
    QString sessionSecret;
    QString locale;
    QString variable;
    QString signature = EQUAL_MARK;
    QStringList variableList;

    signature.append(signatureValue);
    apiKey.append(apiKeyValue);
    apiKey.append(UNDERLINE_MARK);

    user.append(USER);
    user.append(EQUAL_MARK);
    expires.append(EXPIRES);
    expires.append(EQUAL_MARK);
    sessionKey.append(SESSION_KEY);
    sessionKey.append(EQUAL_MARK);
    sessionSecret.append(SESSION_SECRET);
    sessionSecret.append(EQUAL_MARK);
    locale.append(LOCALE);
    locale.append(EQUAL_MARK);
    locale.append(localeValue);

    variableList.append(expires.append(expiresValue.append(BREAK_MARK)));
    variableList.append(sessionKey.append(sessionKeyValue.append(BREAK_MARK)));
    variableList.append(user.append(userValue).append(BREAK_MARK));
    variableList.append(sessionSecret.append(sessionSecretValue.append(BREAK_MARK)));

    cookie.append(BREAK_MARK);

    foreach(variable, variableList) {
        cookie.append(apiKey);
        cookie.append(variable);
    }
    apiKey.remove(UNDERLINE_MARK);
    cookie.append(apiKey);
    cookie.append(signature);
    cookie.append(BREAK_MARK);
    cookie.append(locale);

    qDebug() << cookie;

    return cookie;
}

QUrl SituareService::formUrl(const QString &baseUrl, const QString &phpScript,
                             QString urlParameters)
{
    qDebug() << __PRETTY_FUNCTION__;
    QString urlString;

    urlString.append(baseUrl);
    urlString.append(phpScript);
    if(!urlParameters.isEmpty())
        urlString.append(urlParameters);

    QUrl url = QUrl(urlString);

    qDebug() << url;

    return url;
}

QString SituareService::formUrlParameters(const GeoCoordinate &coordinates, QString status,
                                          bool publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    // one scene pixel is about 5.4e-6 degrees, the integer part is max three digits and one
    // additional digit is added for maximum precision
    const int COORDINATE_PRECISION = 10;

    QString parameters;

    parameters.append(QUESTION_MARK);
    parameters.append(LATITUDE);
    parameters.append(EQUAL_MARK);
    parameters.append(QString::number(coordinates.latitude(), 'f', COORDINATE_PRECISION));
    parameters.append(AMBERSAND_MARK);
    parameters.append(LONGTITUDE);
    parameters.append(EQUAL_MARK);
    parameters.append(QString::number(coordinates.longitude(), 'f', COORDINATE_PRECISION));

    parameters.append(AMBERSAND_MARK);
    parameters.append(PUBLISH);
    parameters.append(EQUAL_MARK);

    if(publish)
        parameters.append(PUBLISH_TRUE);
    else
        parameters.append(PUBLISH_FALSE);

    if(!status.isEmpty()) {
        parameters.append(AMBERSAND_MARK);
        parameters.append(DATA);
        parameters.append(EQUAL_MARK);
        parameters.append(status);
    }

    return parameters;
}

void SituareService::sendRequest(const QUrl &url, const QString &cookieType, const QString &cookie)
{
    qDebug() << __PRETTY_FUNCTION__;

    QNetworkRequest request;

    request.setUrl(url);
    request.setAttribute(QNetworkRequest::CacheSaveControlAttribute, false);
    request.setRawHeader(cookieType.toAscii(), cookie.toUtf8());

    QNetworkReply *reply = m_networkManager->get(request, true);

    m_currentRequests.append(reply);
}

void SituareService::requestFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Reply from situare
    if (m_currentRequests.contains(reply)) {

        qDebug() << "BytesAvailable: " << reply->bytesAvailable();

        if (reply->error()) {
            emit error(ErrorContext::NETWORK, reply->error());
        } else {
            QByteArray replyArray = reply->readAll();
            qDebug() << "Reply from: " << reply->url() << "reply " << replyArray;

            if(replyArray == ERROR_LAT.toAscii()) {
                qDebug() << "Error: " << ERROR_LAT;
                emit error(ErrorContext::SITUARE, SituareError::UPDATE_FAILED);
            } else if(replyArray == ERROR_LON.toAscii()) {
                qDebug() << "Error: " << ERROR_LON;
                emit error(ErrorContext::SITUARE, SituareError::UPDATE_FAILED);
            } else if(replyArray.contains(ERROR_SESSION.toAscii())) {
                qDebug() << "Error: " << ERROR_SESSION;
                emit error(ErrorContext::SITUARE, SituareError::SESSION_EXPIRED);
            } else if(replyArray.startsWith(OPENING_BRACE_MARK.toAscii())) {
                qDebug() << "JSON string";
                parseUserData(replyArray);
            } else if(replyArray.isEmpty()) {
                if(reply->url().toString().contains(UPDATE_LOCATION.toAscii())) {
                    emit updateWasSuccessful();
                } else {
                    // session credentials are invalid
                    emit error(ErrorContext::SITUARE, SituareError::SESSION_EXPIRED);
                }
            } else {
                // unknown reply
                emit error(ErrorContext::SITUARE, SituareError::ERROR_GENERAL);
            }
        }
        m_currentRequests.removeAll(reply);
        reply->deleteLater();
    }
}

void SituareService::credentialsReady(const FacebookCredentials &credentials)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_credentials = credentials;
}

void SituareService::parseUserData(const QByteArray &jsonReply)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_defaultImage = false;

    QJson::Parser parser;
    bool ok;

    QVariantMap result = parser.parse (jsonReply, &ok).toMap();
    if (!ok) {
        emit error(ErrorContext::SITUARE, SituareError::INVALID_JSON);
        return;
    } else {

        if(result.contains("ErrorCode")) {
            QVariant errorVariant = result.value("ErrorCode");
            emit error(ErrorContext::SITUARE, errorVariant.toInt());
            return;
        } else if(result.contains("user")) {

            QVariant userVariant = result.value("user");
            QMap<QString, QVariant> userMap = userVariant.toMap();

            GeoCoordinate coordinates(userMap["latitude"].toReal(), userMap["longitude"].toReal());

            QUrl imageUrl = userMap[NORMAL_SIZE_PROFILE_IMAGE].toUrl();

            if(imageUrl.isEmpty()) {
                // user doesn't have profile image, so we need to get him a silhouette image
                m_defaultImage = true;
            }

            QString address = userMap["address"].toString();
            if(address.isEmpty()) {
                QStringList location;
                location.append(QString::number(coordinates.latitude()));
                location.append(QString::number(coordinates.longitude()));
                address = location.join(", ");
            }

            User user = User(address, coordinates, userMap["name"].toString(),
                          userMap["note"].toString(), imageUrl, userMap["timestamp"].toString(),
                          true, userMap["uid"].toString());

            QList<User> tmpFriendsList;

            foreach (QVariant friendsVariant, result["friends"].toList()) {
              QMap<QString, QVariant> friendMap = friendsVariant.toMap();
              QVariant distance = friendMap["distance"];
              QMap<QString, QVariant> distanceMap = distance.toMap();

              GeoCoordinate coordinates(friendMap["latitude"].toReal(),friendMap["longitude"].toReal());

              QUrl imageUrl = friendMap["profile_pic"].toUrl();

              if(imageUrl.isEmpty()) {
                  // friend doesn't have profile image, so we need to get him a silhouette image
                  m_defaultImage = true;
              }

              QString address = friendMap["address"].toString();
              if(address.isEmpty()) {
                  QStringList location;
                  location.append(QString::number(coordinates.latitude()));
                  location.append(QString::number(coordinates.longitude()));
                  address = location.join(", ");
              }

              User buddy = User(address, coordinates, friendMap["name"].toString(),
                               friendMap["note"].toString(), imageUrl,
                               friendMap["timestamp"].toString(),
                               false, friendMap["uid"].toString(), distanceMap["units"].toString(),
                               distanceMap["value"].toDouble());

              tmpFriendsList.append(buddy);
            }

            QList<QUrl> imageUrlList; // url list for images

            // set unchanged profile images or add new images to imageUrlList for downloading
            if(m_user) {
                if(m_user->profileImageUrl() != user.profileImageUrl()) {
                    if(!user.profileImageUrl().isEmpty())
                        imageUrlList.append(user.profileImageUrl());
                } else {
                    user.setProfileImage(m_user->profileImage());
                }
            } else {
                if(!user.profileImageUrl().isEmpty())
                    imageUrlList.append(user.profileImageUrl());
            }

            // clear old user object
            if(m_user) {
                delete m_user;
                m_user = 0;
            }

            // create new user object from temporary user object
            m_user = new User(user);

            // set unchanged profile images or add new images to imageUrlList for downloading
            if(!m_friendsList.isEmpty()) {
                foreach(User tmpBuddy, tmpFriendsList) {
                    if(!tmpBuddy.profileImageUrl().isEmpty()) {
                        bool found = false;
                        foreach(User *buddy, m_friendsList) {
                            if(tmpBuddy.profileImageUrl() == buddy->profileImageUrl()) {
                                tmpBuddy.setProfileImage(buddy->profileImage());
                                found = true;
                                break;
                            }
                        }
                        if(!found && !tmpBuddy.profileImageUrl().isEmpty())
                            imageUrlList.append(tmpBuddy.profileImageUrl());
                    }
                }
            } else {
                foreach(User buddy, tmpFriendsList) {
                    if(!buddy.profileImageUrl().isEmpty())
                        imageUrlList.append(buddy.profileImageUrl());
                }
            }

            // clear old friendlist
            qDeleteAll(m_friendsList.begin(), m_friendsList.end());
            m_friendsList.clear();

            // populate new friendlist with temporary friendlist's data
            foreach(User tmpFriendItem, tmpFriendsList) {
                User *friendItem = new User(tmpFriendItem);
                m_friendsList.append(friendItem);
            }
            tmpFriendsList.clear();

            emit userDataChanged(m_user, m_friendsList);

            // set silhouette image to imageUrlList for downloading
            if(m_defaultImage)
                imageUrlList.append(QUrl(SILHOUETTE_URL));

            addProfileImages(imageUrlList);
            imageUrlList.clear();
        } else {
            QVariant address = result.value("address");
            if(!address.toString().isEmpty()) {
                emit reverseGeoReady(address.toString());
            } else {
                QStringList coordinates;
                coordinates.append(result.value("lat").toString());
                coordinates.append(result.value("lon").toString());

                emit error(ErrorContext::SITUARE, SituareError::ADDRESS_RETRIEVAL_FAILED);
                emit reverseGeoReady(coordinates.join(", "));
            }
        }
    }
}

void SituareService::imageReceived(const QUrl &url, const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;
    qDebug() << "Image URL: " << url << " size :" << image.size();

    // assign facebook silhouette image to all who doesn't have a profile image
    if(url == QUrl(SILHOUETTE_URL)) {
        if(m_user->profileImageUrl().isEmpty()) {
            m_user->setProfileImage(AvatarImage::create(image, AvatarImage::Large));
            emit imageReady(m_user);
        }
        foreach(User *friendItem, m_friendsList) {
            if(friendItem->profileImageUrl().isEmpty()) {
                friendItem->setProfileImage(AvatarImage::create(image, AvatarImage::Small));
                emit imageReady(friendItem);
            }
        }
    }

    if (m_user->profileImageUrl() == url) {
        m_user->setProfileImage(AvatarImage::create(image, AvatarImage::Large));
        emit imageReady(m_user);
    }

    foreach(User *friendItem, m_friendsList) {
        if(friendItem->profileImageUrl() == url) {
            friendItem->setProfileImage(AvatarImage::create(image, AvatarImage::Small));
            emit imageReady(friendItem);
        }
    }
}

void SituareService::addProfileImages(const QList<QUrl> &imageUrlList)
{
    qDebug() << __PRETTY_FUNCTION__;

    foreach(QUrl url, imageUrlList) {
        emit fetchImage(url);
    }
}

void SituareService::clearUserData()
{
    qDebug() << __PRETTY_FUNCTION__;

    qDeleteAll(m_friendsList.begin(), m_friendsList.end());
    m_friendsList.clear();

    if(m_user) {
        delete m_user;
        m_user = 0;
    }
    emit userDataChanged(m_user, m_friendsList);
}
