/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QTimer>

#include "networkhandler.h"
#include "networkreply.h"

#include "networkaccessmanager.h"

NetworkAccessManager::NetworkAccessManager(QObject *parent)
    : QObject(parent),
      m_connected(false),
      m_networkHandler(0),
      m_networkAccessManagerPrivate(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_networkHandler = NetworkHandler::instance();
    m_networkAccessManagerPrivate = new QNetworkAccessManager(this);

    connect(m_networkHandler, SIGNAL(connected()),
            this, SLOT(connected()));
    connect(m_networkHandler, SIGNAL(disconnected()),
            this, SLOT(disconnected()));

    connect(m_networkAccessManagerPrivate, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(downloadFinished(QNetworkReply*)));
}

QAbstractNetworkCache *NetworkAccessManager::cache() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_networkAccessManagerPrivate->cache();
}

void NetworkAccessManager::connected()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_connected = true;

    //Loop through all requests and calls get method.
    foreach (const QNetworkRequest &request, m_requestQueue) {
        QNetworkReply *reply = m_networkAccessManagerPrivate->get(request);
        m_temporaryReplyQueue.insert(request.url().toString(), reply);
    }

    m_requestQueue.clear();
}

void NetworkAccessManager::disconnected()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_connected = false;
}

void NetworkAccessManager::downloadFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString key = m_temporaryReplyQueue.key(reply, "");

    //Replace offline reply object's content with server reply's content
    if (!key.isEmpty()) {
        QNetworkReply *offlineReply = m_offlineReplyQueue.value(key, 0);

        if (offlineReply) {
            memmove(offlineReply, reply, sizeof(QNetworkReply));
            m_offlineReplyQueue.remove(key);
            m_temporaryReplyQueue.remove(key);
            emit finished(offlineReply);
        }
    }
    //Forward online request's reply
    else {
        emit finished(reply);
    }
}

QNetworkReply *NetworkAccessManager::get(const QNetworkRequest &request, bool onlineRequestsOnly)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Disconnected from network, queue request and return empty reply.
    if (!m_networkHandler->isConnected()) {
        m_requestQueue.append(request);
        m_networkHandler->connect();
        QNetworkReply *reply = new NetworkReply(request, this);

        if (onlineRequestsOnly) {
            ((NetworkReply*)reply)->setErrorCode(QNetworkReply::UnknownNetworkError);
            emit finished(reply);
        }
        else {
            m_offlineReplyQueue.insert(request.url().toString(), reply);
        }

        return reply;
    }
    //Connected, use normal get method.
    else {
        return m_networkAccessManagerPrivate->get(request);
    }
}

QNetworkReply *NetworkAccessManager::post(const QNetworkRequest &request, QByteArray &data,
                                          bool onlineRequestsOnly)
{
    qDebug() << __PRETTY_FUNCTION__;

    //Disconnected from network, queue request and return empty reply.
    if (!m_networkHandler->isConnected()) {
        m_requestQueue.append(request);
        m_networkHandler->connect();
        QNetworkReply *reply = new NetworkReply(request, this);

        if (onlineRequestsOnly) {
            ((NetworkReply*)reply)->setErrorCode(QNetworkReply::UnknownNetworkError);
            emit finished(reply);
        }
        else {
            m_offlineReplyQueue.insert(request.url().toString(), reply);
        }

        return reply;
    }
    //Connected, use normal post method.
    else {
        return m_networkAccessManagerPrivate->post(request, data);
    }
}

bool NetworkAccessManager::isConnected()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_connected;
}

void NetworkAccessManager::setCache(QAbstractNetworkCache *cache)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_networkAccessManagerPrivate->setCache(cache);
}
