/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#ifndef FRIENDITEMSHANDLER_H
#define FRIENDITEMSHANDLER_H

#include <QLinkedList>
#include <QObject>

class BaseLocationItem;
class FriendGroupItem;
class FriendLocationItem;
class MapScene;
class User;

/**
 * @brief Handler for friend and friend group items
 *
 * Handles all friend map items. Colliding items and groups are grouped. All items are owned
 * by MapScene.
 *
 * @author Sami Rämö - sami.ramo@ixonos.com
 * @author Ville Tiensuu - ville.tiensuu@ixonos.com
 */
class FriendItemsHandler : public QObject
{
    Q_OBJECT

public:
    /**
      * @brief Constructor
      *
      * @param mapScene MapScene object
      * @param parent Parent QObject
      */
    FriendItemsHandler(MapScene *mapScene, QObject *parent = 0);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
private:
    /**
      * @brief Add new FriendLocationItem
      *
      * New FriendLocationItem is created, values are set and item is added to map.
      * Item is also added to m_friendItems list.
      *
      * @param friendData Data for new friend
      */
    void addFriendItem(User *friendData);

    /**
      * @brief Group colliding friends
      *
      * Call checkFriendForCollidingFriends() for all visible FriendLocationItem items.
      */
    void checkAllFriendsForCollidingFriends();

    /**
      * @brief Check single FriendLocationItem for colliding FriendLocationItem items
      *
      * Check FriendLocationItem against another visible FriendLocationItem items. If collision is
      * found, then new FriendGroupItem is made and colliding FriendLocationItem items are joined
      * to this new FriendGroupItem.
      *
      * @param item FriendLocationItem to be checked
      */
    void checkFriendForCollidingFriends(FriendLocationItem *item);

    /**
    * @brief clean old friend data from m_mapScene and m_friendItems
    *
    * @param friendsList QList item of friend information
    */
    void cleanOldFriendData(const QList<User *> &friendsList);

    /**
    * @brief Check if items collide
    *
    * Does check if items sceneTransformedBoundingRect() does intersect. If item1's rect is max
    * half of the rect width from the vertical limits of the map (from inside), then rect is
    * translated to opposite side of the map and intersections are tested there too.
    *
    * @param item1 First item
    * @param item2 Secont item
    * @return True if collision was found, otherwise false
    */
    bool collides(BaseLocationItem *item1, BaseLocationItem *item2);

    /**
      * @brief Delete FriendLocationItem
      *
      * Drops item from all groups, removes it from scene and deletes the item.
      *
      * @param item Item to be deleted
      */
    void deleteFriendItem(FriendLocationItem *item);

    /**
    * @brief Destructs all current group items
    */
    void destructGroups();

    /**
      * @brief Update FriendLocationItem data
      *
      * Position and image are updated.
      *
      * @param friendItem Item to be updated
      * @param friendData New data for the item
      */
    void updateFriendItem(FriendLocationItem *friendItem, User *friendData);

    /**
    * @brief updates data member m_friendItems from given parameter
    *
    * @param friendsList QList item of friend information
    */
    void updateFriendItemList(const QList<User *> &friendsList);

    /**
    * @brief updates data member m_friendItems values that differs from given parameter
    *
    * @param friendsList QList item of friend information
    */
    void updateFriendLocationsAndImages(const QList<User *> &friendsList);

private slots:

    /**
    * @brief Slot updating friend item's profile image
    *
    * @param user Friend
    */
    void friendImageReady(User *user);

    /**
    * @brief Slot for upgrading friend items and groups
    *
    * Does add and/or remove FriendLocationItem items if there are new friends
    * or old ones deleted in the given list. Remaining FriensLocationItems are updated.
    *
    * @param friendsList QList item of friend information
    */
    void friendListUpdated(QList<User *> &friendsList);

    /**
      * @brief Destroys all current groups and runs grouping again
      *
      * Calls destructGroups() and checkAllFriendsForCollidingFriends().
      *
      * @param zoomLevel Current view zoom level, used for friend item rect scaling
      */
    void refactorFriendItems(int zoomLevel);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signal is emitted when location item is clicked.
    *
    * @param userIDs list of friends user IDs in the group
    */
    void locationItemClicked(const QList<QString> &userIDs);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QLinkedList<FriendGroupItem *> m_friendGroupItems; ///< All FriendGroupItem items
    QLinkedList<FriendLocationItem *> m_friendItems; ///< List of friendLocationItems
    MapScene *m_mapScene; ///< Pointer to MapScene
    int m_zoomLevel; ///< Current view zoom level used for calculation of items bounding rect
};

#endif // FRIENDITEMSHANDLER_H
