/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>

#include "liblocationwrapper.h"
#include "geopositioninfo.h"
#include "coordinates/geocoordinate.h"

const int INTERVAL_1S = 1500;       ///< Maximum value for 1 sec interval
const int INTERVAL_2S = 2500;       ///< Maximum value for 2 sec interval
const int INTERVAL_5S = 7500;       ///< Maximum value for 5 sec interval
const int INTERVAL_10S = 15000;     ///< Maximum value for 10 sec interval
const int INTERVAL_20S = 25000;     ///< Maximum value for 20 sec interval
const int INTERVAL_30S = 45000;     ///< Maximum value for 30 sec interval
const int INTERVAL_60S = 90000;     ///< Maximum value for 60 sec interval

LiblocationWrapper::LiblocationWrapper(QObject *parent)
    : QObject(parent),
      m_control(0),
      m_device(0),
      m_lastKnownPosition(GeoPositionInfo()),
      m_state(LiblocationWrapper::Undefined)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_control = location_gpsd_control_get_default();
    m_device = (LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
}

LiblocationWrapper::~LiblocationWrapper()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_device)
        g_object_unref(m_device);
    if (m_control)
        g_object_unref(m_control);
}

void LiblocationWrapper::changed(LocationGPSDevice *device, gpointer data)
{
    qDebug() << __PRETTY_FUNCTION__;

    const int METRES_COEFFICIENT = 100; // Coefficient to get metres from centimetres

    if (!device || !data)
        return;

    GeoPositionInfo positionInfo;
    LiblocationWrapper *wrapper;
    wrapper = (LiblocationWrapper*)data;

    if (device && wrapper->isRunning()) {
        if (device->fix) {

            if (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
                positionInfo.setTimestamp(device->fix->time);
                positionInfo.setAccuracy(true, device->fix->eph / METRES_COEFFICIENT);
            }
            else {
                positionInfo.setAccuracy(false, device->fix->eph / METRES_COEFFICIENT);
            }

            if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {
                GeoCoordinate coordinate;

                coordinate.setLatitude(device->fix->latitude);
                coordinate.setLongitude(device->fix->longitude);
                positionInfo.setCoordinate(coordinate);
            }
        }
        wrapper->setLastKnownPosition(positionInfo);
        emit wrapper->locationChanged(positionInfo);
    }
}

void LiblocationWrapper::error(LocationGPSDevice *device, gpointer data)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!device || !data)
        return;

    LiblocationWrapper *wrapper;
    wrapper = (LiblocationWrapper*)data;

    wrapper->errorMessage(tr("Error in GPS"));
}

bool LiblocationWrapper::isRunning()
{
    qDebug() << __PRETTY_FUNCTION__;

    return (state() == LiblocationWrapper::Running);
}

GeoPositionInfo LiblocationWrapper::lastKnownPosition() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_lastKnownPosition;
}

void LiblocationWrapper::setLastKnownPosition(const GeoPositionInfo &positionInfo)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_lastKnownPosition = positionInfo;
}

void LiblocationWrapper::init(int updateInterval)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (state() == LiblocationWrapper::Undefined) {

        int locationInterval = LOCATION_INTERVAL_DEFAULT;

        if ((updateInterval > 0) && (updateInterval <= INTERVAL_1S))
            locationInterval = LOCATION_INTERVAL_1S;
        else if (updateInterval <= INTERVAL_2S)
            locationInterval = LOCATION_INTERVAL_2S;
        else if (updateInterval <= INTERVAL_5S)
            locationInterval = LOCATION_INTERVAL_5S;
        else if (updateInterval <= INTERVAL_10S)
            locationInterval = LOCATION_INTERVAL_10S;
        else if (updateInterval <= INTERVAL_20S)
            locationInterval = LOCATION_INTERVAL_20S;
        else if (updateInterval <= INTERVAL_30S)
            locationInterval = LOCATION_INTERVAL_30S;
        else if (updateInterval <= INTERVAL_60S)
            locationInterval = LOCATION_INTERVAL_60S;
        else
            locationInterval = LOCATION_INTERVAL_120S;


        g_object_set(G_OBJECT(m_control), "preferred-method", LOCATION_METHOD_USER_SELECTED,
                "preferred-interval", locationInterval, ((void *)0));

        g_signal_connect(G_OBJECT(m_device), "error", G_CALLBACK(&error),
                         static_cast<void*>(this));

        g_signal_connect(G_OBJECT(m_device), "changed", G_CALLBACK(&changed),
                         static_cast<void*>(this));

        setState(LiblocationWrapper::Initialized);
    }
}

void LiblocationWrapper::startUpdates()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!isRunning()) {
        setState(LiblocationWrapper::Running);
        location_gpsd_control_start(m_control);
    }
}

void LiblocationWrapper::setState(int state)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_state = state;
}

int LiblocationWrapper::state()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_state;
}

void LiblocationWrapper::stopUpdates()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (isRunning()) {
        setState(LiblocationWrapper::Initialized);
        location_gpsd_control_stop(m_control);
    }
}
