/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Pekka Nissinen - pekka.nissinen@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QAbstractState>

#include "sidepanel.h"

#include "panelcommon.h"
#include "panelsidebar.h"
#include "panelsliderbar.h"
#include "sidepanelbase.h"

SidePanel::SidePanel(QWidget *parent) :
    QWidget(parent),
    m_isOpen(false),
    currentType(SidePanel::None)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_panelVBox = new QVBoxLayout(this);
    m_panelVBox->setMargin(0);
    m_panelVBox->setSpacing(0);

    m_panelBase = new SidePanelBase(this);
    m_panelBase->setLayout(m_panelVBox);

    m_panelStateMachine = new QStateMachine(this);
    m_panelStateClosed = new QState(m_panelStateMachine);
    m_panelStateOpened = new QState(m_panelStateMachine);

    m_panelStateMachine->setInitialState(m_panelStateClosed);

    m_panelTransitionOpen = m_panelStateClosed->addTransition(this, SIGNAL(toggleState()),
                                                              m_panelStateOpened);
    m_panelTransitionOpen->addAnimation(new QPropertyAnimation(this, "pos", this));

    m_panelTransitionClose = m_panelStateOpened->addTransition(this, SIGNAL(toggleState()),
                                                               m_panelStateClosed);
    m_panelTransitionClose->addAnimation(new QPropertyAnimation(this, "pos", this));

    connect(m_panelStateClosed, SIGNAL(entered()),
            this, SLOT(stateChangedToClosed()));
    connect(m_panelStateOpened, SIGNAL(entered()),
            this, SLOT(stateChangedToOpen()));

    hide();
}

void SidePanel::openPanel()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    if(!m_isOpen)
        emit toggleState();
}

void SidePanel::closePanel()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    if(m_isOpen)
        emit toggleState();
}

void SidePanel::resizePanel(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(currentType == SidePanel::UserPanel) {
        this->resize(USERPANEL_WIDTH + SLIDER_BAR_WIDTH + SLIDER_BUTTON_OFFSET,
                     size.height() - PANEL_TOP_PADDING - PANEL_BOTTOM_PADDING);

        m_panelBase->resize(USERPANEL_WIDTH,
                            size.height() - PANEL_TOP_PADDING - PANEL_BOTTOM_PADDING);

        m_panelSlidingBar->resizeSliderBar(size);
    } else if(currentType == SidePanel::FriendPanel) {
        this->resize(SLIDER_WIDTH + FRIENDPANEL_WIDTH,
                     size.height() - PANEL_TOP_PADDING - PANEL_BOTTOM_PADDING);

        if(!m_isOpen)
            this->move(size.width() - SLIDER_BUTTON_OFFSET - PANEL_PEEK_AMOUNT, PANEL_TOP_PADDING);
        else
            this->move(size.width() - SLIDER_BUTTON_OFFSET - SLIDER_BAR_WIDTH - FRIENDPANEL_WIDTH,
                       PANEL_TOP_PADDING);

        m_panelBase->resize(FRIENDPANEL_WIDTH,
                            size.height() - PANEL_TOP_PADDING - PANEL_BOTTOM_PADDING);

        m_panelSlidingBar->resizeSliderBar(size);

        m_panelStateClosed->assignProperty(this, "pos",
                QPoint(size.width() - SLIDER_BUTTON_OFFSET - PANEL_PEEK_AMOUNT, PANEL_TOP_PADDING));
        m_panelStateOpened->assignProperty(this, "pos",
                QPoint(size.width() - SLIDER_BUTTON_OFFSET - SLIDER_BAR_WIDTH - FRIENDPANEL_WIDTH,
                       PANEL_TOP_PADDING));
    } else {
        qFatal("Illegal panel type in SidePanel::screenResized");
    }
}

void SidePanel::setType(SidePanel::PanelType type)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(currentType == SidePanel::None) {
        if(type == SidePanel::UserPanel) {
            currentType = type;
            setObjectName("UserPanel");

            this->resize(USERPANEL_WIDTH + SLIDER_BAR_WIDTH + SLIDER_BUTTON_OFFSET,
                         USERPANEL_HEIGHT);
            this->move(USERPANEL_CLOSED_X, PANEL_TOP_PADDING);

            m_panelBase->resize(USERPANEL_WIDTH, SLIDER_HEIGHT);

            m_panelSlidingBar = new PanelSliderBar(this, LEFT);
            m_panelSlidingBar->move(USERPANEL_WIDTH - SLIDER_BUTTON_OFFSET, 0);

            connect(m_panelSlidingBar, SIGNAL(clicked()),
                    this, SIGNAL(toggleState()));

            m_panelStateClosed->assignProperty(this, "pos",
                                               QPoint(USERPANEL_CLOSED_X, PANEL_TOP_PADDING));
            m_panelStateOpened->assignProperty(this, "pos",
                                               QPoint(USERPANEL_OPENED_X, PANEL_TOP_PADDING));
        } else if (type == SidePanel::FriendPanel) {
            currentType = type;
            setObjectName("FriendPanel");

            this->resize(SLIDER_BUTTON_OFFSET + SLIDER_BAR_WIDTH + FRIENDPANEL_WIDTH,
                         FRIENDPANEL_HEIGHT);
            this->move(FRIENDPANEL_CLOSED_X, PANEL_TOP_PADDING);

            m_panelBase->resize(FRIENDPANEL_WIDTH, FRIENDPANEL_HEIGHT);
            m_panelBase->move(SLIDER_BUTTON_OFFSET + SLIDER_BAR_WIDTH, 0);

            m_panelSlidingBar = new PanelSliderBar(this, RIGHT);

            connect(m_panelSlidingBar, SIGNAL(clicked()),
                    this, SIGNAL(toggleState()));

            m_panelStateClosed->assignProperty(this, "pos",
                                               QPoint(FRIENDPANEL_CLOSED_X, PANEL_TOP_PADDING));
            m_panelStateOpened->assignProperty(this, "pos",
                                               QPoint(FRIENDPANEL_OPENED_X, PANEL_TOP_PADDING));
        } else {
            qWarning() << "In" << __PRETTY_FUNCTION__ << ":" << type;
            qFatal("Illegal type set for SidePanel::setType");
        }
        m_panelStateMachine->start();
        show();
    } else {
        qWarning() << "Illegally tried to set type multiple times at SidePanel::setType";
    }
}

void SidePanel::stateChangedToClosed()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_isOpen = false;
    emit panelClosed();

    m_panelSlidingBar->changeDirection(PanelSliderBar::Closed);
}

void SidePanel::stateChangedToOpen()
{
    qDebug() << __PRETTY_FUNCTION__;
    
    m_isOpen = true;
    emit panelOpened();

    m_panelSlidingBar->changeDirection(PanelSliderBar::Open);
}
