/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QtAlgorithms>
#include <QDebug>
#include <QString>
#include <QStringList>
#include <QUrl>
#include <QHash>
#include <QHashIterator>
#include <QRect>

#include "common.h"
#include "frienditemshandler.h"
#include "gpslocationitem.h"
#include "mapcommon.h"
#include "mapfetcher.h"
#include "mapscene.h"
#include "maptile.h"
#include "network/networkaccessmanager.h"
#include "ownlocationitem.h"
#include "user/user.h"

#include "mapengine.h"

MapEngine::MapEngine(QObject *parent)
    : QObject(parent),
      m_autoCenteringEnabled(false),
      m_zoomedIn(false),
      m_zoomLevel(DEFAULT_ZOOM_LEVEL),
      m_centerTile(QPoint(UNDEFINED, UNDEFINED)),
      m_lastManualPosition(QPoint(0, 0)),
      m_tilesGridSize(QSize(0, 0)),
      m_viewSize(QSize(DEFAULT_SCREEN_WIDTH, DEFAULT_SCREEN_HEIGHT))
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapScene = new MapScene(this);

    m_mapFetcher = new MapFetcher(NetworkAccessManager::instance(), this);
    connect(this, SIGNAL(fetchImage(int, int, int)),
            m_mapFetcher, SLOT(enqueueFetchMapImage(int, int, int)));
    connect(m_mapFetcher, SIGNAL(mapImageReceived(int, int, int, QPixmap)),
            this, SLOT(mapImageReceived(int, int, int, QPixmap)));
    connect(m_mapFetcher, SIGNAL(error(int)),
            this, SIGNAL(error(int)));

    m_ownLocation = new OwnLocationItem();
    m_ownLocation->hide(); // hide until first location info is received
    m_mapScene->addItem(m_ownLocation);

    m_gpsLocationItem = new GPSLocationItem();
    m_mapScene->addItem(m_gpsLocationItem);

    m_friendItemsHandler = new FriendItemsHandler(m_mapScene, this);
    connect(this, SIGNAL(zoomLevelChanged(int)),
            m_friendItemsHandler, SLOT(refactorFriendItems(int)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            m_friendItemsHandler, SLOT(friendListUpdated(QList<User*>&)));

    connect(this, SIGNAL(friendsLocationsReady(QList<User*>&)),
            this, SLOT(friendsPositionsUpdated()));

    connect(m_friendItemsHandler, SIGNAL(locationItemClicked(QList<QString>)),
            this, SIGNAL(locationItemClicked(QList<QString>)));
}

MapEngine::~MapEngine()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(MAP_LAST_POSITION,
                      convertSceneCoordinateToLatLon(m_zoomLevel, m_sceneCoordinate));
    settings.setValue(MAP_LAST_ZOOMLEVEL, m_zoomLevel);
}

QRect MapEngine::calculateTileGrid(QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPoint tileCoordinate = convertSceneCoordinateToTileNumber(m_zoomLevel, sceneCoordinate);

    QPoint topLeft;
    topLeft.setX(tileCoordinate.x() - (m_tilesGridSize.width() / 2));
    topLeft.setY(tileCoordinate.y() - (m_tilesGridSize.height() / 2));

    return QRect(topLeft, m_tilesGridSize);
}

QPointF MapEngine::centerGeoCoordinate()
{
    qDebug() << __PRETTY_FUNCTION__;

    return convertSceneCoordinateToLatLon(m_zoomLevel, m_sceneCoordinate);
}

QPoint MapEngine::convertLatLonToSceneCoordinate(QPointF latLonCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    qreal longitude = latLonCoordinate.x();
    qreal latitude = latLonCoordinate.y();

    if ((longitude > MAX_LONGITUDE) || (longitude < MIN_LONGITUDE))
        return QPoint(UNDEFINED, UNDEFINED);
    if ((latitude > MAX_LATITUDE) || (latitude < MIN_LATITUDE))
        return QPoint(UNDEFINED, UNDEFINED);

    qreal z = static_cast<qreal>(MapEngine::tilesPerSide(MAX_MAP_ZOOM_LEVEL));

    qreal x = static_cast<qreal>((longitude + 180.0) / 360.0);
    qreal y = static_cast<qreal>((1.0 - log(tan(latitude * M_PI / 180.0) + 1.0
                                / cos(latitude * M_PI / 180.0)) / M_PI) / 2.0);

    return QPointF(x * z * TILE_SIZE_X, y * z * TILE_SIZE_Y).toPoint();
}

QPointF MapEngine::convertSceneCoordinateToLatLon(int zoomLevel, QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    double tileFactor = 1 << (MAX_MAP_ZOOM_LEVEL - zoomLevel);
    double xFactor = (sceneCoordinate.x() / (TILE_SIZE_X*tileFactor));
    double yFactor = (sceneCoordinate.y() / (TILE_SIZE_Y*tileFactor));

    tileFactor = 1 << zoomLevel;
    double longitude = xFactor / tileFactor * 360.0 - 180;

    double n = M_PI - 2.0 * M_PI * yFactor / tileFactor;
    double latitude = 180.0 / M_PI * atan(0.5 * (exp(n) - exp(-n)));

    return QPointF(longitude, latitude);
}

QPoint MapEngine::convertSceneCoordinateToTileNumber(int zoomLevel, QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    int pow = 1 << (MAX_MAP_ZOOM_LEVEL - zoomLevel);
    int x = static_cast<int>(sceneCoordinate.x() / (TILE_SIZE_X * pow));
    int y = static_cast<int>(sceneCoordinate.y() / (TILE_SIZE_Y * pow));

    return QPoint(x, y);
}

QPoint MapEngine::convertTileNumberToSceneCoordinate(int zoomLevel, QPoint tileNumber)
{
    qDebug() << __PRETTY_FUNCTION__;

    int pow = 1 << (MAX_MAP_ZOOM_LEVEL - zoomLevel);
    int x = tileNumber.x() * TILE_SIZE_X * pow;
    int y = tileNumber.y() * TILE_SIZE_Y * pow;

    return QPoint(x, y);
}

bool MapEngine::disableAutoCentering(QPoint sceneCoordinate)
{
    if (isAutoCenteringEnabled()) {
        int zoomFactor = (1 << (MAX_MAP_ZOOM_LEVEL - m_zoomLevel));

        QPoint oldPixelValue = QPoint(m_lastManualPosition.x() / zoomFactor,
                                      m_lastManualPosition.y() / zoomFactor);

        QPoint newPixelValue = QPoint(sceneCoordinate.x() / zoomFactor,
                                      sceneCoordinate.y() / zoomFactor);

        if ((abs(oldPixelValue.x() - newPixelValue.x()) > AUTO_CENTERING_DISABLE_DISTANCE) ||
            (abs(oldPixelValue.y() - newPixelValue.y()) > AUTO_CENTERING_DISABLE_DISTANCE))
            return true;
    }

    return false;
}

void MapEngine::friendsPositionsUpdated()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapScene->spanItems(m_zoomLevel, m_sceneCoordinate, m_viewSize);
}

void MapEngine::getTiles(QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_viewTilesGrid = calculateTileGrid(sceneCoordinate);
    updateViewTilesSceneRect();
    m_mapScene->setTilesGrid(m_viewTilesGrid);

    int topLeftX = m_viewTilesGrid.topLeft().x();
    int topLeftY = m_viewTilesGrid.topLeft().y();
    int bottomRightX = m_viewTilesGrid.bottomRight().x();
    int bottomRightY = m_viewTilesGrid.bottomRight().y();

    int tileMaxVal = tileMaxIndex(m_zoomLevel);

    for (int x = topLeftX; x <= bottomRightX; ++x) {
        for (int y = topLeftY; y <= bottomRightY; ++y) {

            // map doesn't span in vertical direction, so y index must be inside the limits
            if (y >= MAP_TILE_MIN_INDEX && y <= tileMaxVal) {
                if (!m_mapScene->tileInScene(tilePath(m_zoomLevel, x, y)))
                    emit fetchImage(m_zoomLevel, normalize(x, MAP_TILE_MIN_INDEX, tileMaxVal), y);
            }
        }
    }
}

void MapEngine::gpsPositionUpdate(QPointF position, qreal accuracy)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_gpsLocationItem->updatePosition(convertLatLonToSceneCoordinate(position), accuracy);
    m_mapScene->spanItems(m_zoomLevel, m_sceneCoordinate, m_viewSize);

    if (m_autoCenteringEnabled)
        setViewLocation(position);
}

qreal MapEngine::greatCircleDistance(QPointF firstLocation, QPointF secondLocation)
{
    qDebug() << __PRETTY_FUNCTION__;

    const qreal TORAD = (M_PI/180);

    qreal dLat = (secondLocation.y() - firstLocation.y())*TORAD;
    qreal dLon = (secondLocation.x() - firstLocation.x())*TORAD;
    qreal a = pow(sin(dLat/2),2) + cos(firstLocation.y()*TORAD) * cos(secondLocation.y()*TORAD)
              * pow(sin(dLon/2),2);
    qreal c = 2 * atan2(sqrt(a), sqrt(1-a));

    return (EARTH_RADIUS * c);
}

void MapEngine::init()
{
    qDebug() << __PRETTY_FUNCTION__;

    QPointF startLocation;
    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    if (settings.value(MAP_LAST_POSITION, ERROR_VALUE_NOT_FOUND_ON_SETTINGS).toString()
        == ERROR_VALUE_NOT_FOUND_ON_SETTINGS || settings.value(MAP_LAST_ZOOMLEVEL,
        ERROR_VALUE_NOT_FOUND_ON_SETTINGS).toString() == ERROR_VALUE_NOT_FOUND_ON_SETTINGS) {

        startLocation = QPointF(DEFAULT_LONGITUDE, DEFAULT_LATITUDE);
        m_zoomLevel = qBound(MIN_VIEW_ZOOM_LEVEL, DEFAULT_START_ZOOM_LEVEL, MAX_MAP_ZOOM_LEVEL);
    } else {
        m_zoomLevel = settings.value(MAP_LAST_ZOOMLEVEL, ERROR_VALUE_NOT_FOUND_ON_SETTINGS).toInt();
        startLocation = settings.value(MAP_LAST_POSITION,
                                       ERROR_VALUE_NOT_FOUND_ON_SETTINGS).toPointF();
    }

    emit zoomLevelChanged(m_zoomLevel);
    setViewLocation(QPointF(startLocation.x(), startLocation.y()));
}

bool MapEngine::isAutoCenteringEnabled()
{
    return m_autoCenteringEnabled;
}

bool MapEngine::isCenterTileChanged(QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPoint centerTile = convertSceneCoordinateToTileNumber(m_zoomLevel, sceneCoordinate);
    QPoint temp = m_centerTile;
    m_centerTile = centerTile;

    return (centerTile != temp);
}

qreal MapEngine::sceneResolution()
{
    qDebug() << __PRETTY_FUNCTION__;

    const int SHIFT = 200;
    const int KM_TO_M = 1000;
    qreal scale = (1 << (MAX_MAP_ZOOM_LEVEL - m_zoomLevel));
    QPointF centerCoordinate = centerGeoCoordinate();
    QPoint shiftedSceneCoordinate = QPoint(m_sceneCoordinate.x() + SHIFT*scale
                                           , m_sceneCoordinate.y());
    QPointF shiftedCoordinate = convertSceneCoordinateToLatLon(m_zoomLevel, shiftedSceneCoordinate);
    qreal dist = greatCircleDistance(centerCoordinate, shiftedCoordinate) * KM_TO_M;
    return (dist / SHIFT);
}

void MapEngine::mapImageReceived(int zoomLevel, int x, int y, const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;

    // add normal tile inside the world
    QPoint tileNumber(x, y);
    m_mapScene->addTile(zoomLevel, tileNumber, image, m_zoomLevel);

    // note: add 1 so odd width is rounded up and even is rounded down
    int tilesGridWidthHalf = (m_viewTilesGrid.width() + 1) / 2;

    // duplicate to east side? (don't need to duplicate over padding)
    if (tileNumber.x() < (tilesGridWidthHalf - GRID_PADDING)) {
        QPoint adjustedTileNumber(tileNumber.x() + tileMaxIndex(zoomLevel) + 1, tileNumber.y());
        m_mapScene->addTile(zoomLevel, adjustedTileNumber, image, m_zoomLevel);
    }

    // duplicate to west side? (don't need to duplicate over padding)
    if (tileNumber.x() > (tileMaxIndex(zoomLevel) - tilesGridWidthHalf + GRID_PADDING)) {
        QPoint adjustedTileNumber(tileNumber.x() - tileMaxIndex(zoomLevel) - 1, tileNumber.y());
        m_mapScene->addTile(zoomLevel, adjustedTileNumber, image, m_zoomLevel);
    }
}

int MapEngine::normalize(int value, int min, int max)
{
    qDebug() << __PRETTY_FUNCTION__;
    Q_ASSERT_X(max >= min, "parameters", "max can't be smaller than min");

    while (value < min)
        value += max - min + 1;

    while (value > max)
        value -= max - min + 1;

    return value;
}

void MapEngine::receiveOwnLocation(User *user)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(user) {
        QPoint newPosition = convertLatLonToSceneCoordinate(user->coordinates());
        if (m_ownLocation->pos().toPoint() != newPosition) {
            m_ownLocation->setPos(newPosition);
        }

        if (!m_ownLocation->isVisible())
            m_ownLocation->show();
    } else {
        m_ownLocation->hide();
    }

    m_mapScene->spanItems(m_zoomLevel, m_sceneCoordinate, m_viewSize);
}

QGraphicsScene* MapEngine::scene()
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_mapScene;
}

void MapEngine::setAutoCentering(bool enabled)
{
    m_autoCenteringEnabled = enabled;
}

void MapEngine::setGPSEnabled(bool enabled)
{
    m_gpsLocationItem->setEnabled(enabled);
}

void MapEngine::setLocation(QPoint sceneCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    // jump to opposite side of the world if world horizontal limit is exceeded
    if (sceneCoordinate.x() < MAP_MIN_PIXEL_X)
        sceneCoordinate.setX(sceneCoordinate.x() + MAP_PIXELS_X);
    else if (sceneCoordinate.x() > MAP_MAX_PIXEL_X)
        sceneCoordinate.setX(sceneCoordinate.x() - MAP_PIXELS_X);

    // don't allow vertical scene coordinates go out of the map
    sceneCoordinate.setY(qBound(MAP_MIN_PIXEL_Y, sceneCoordinate.y(), MAP_MAX_PIXEL_Y));

    if (disableAutoCentering(sceneCoordinate))
        emit mapScrolledManually();

    m_sceneCoordinate = sceneCoordinate;
    emit locationChanged(m_sceneCoordinate);

    if (isCenterTileChanged(sceneCoordinate)) {
        getTiles(sceneCoordinate);
        m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    }

    m_mapScene->spanItems(m_zoomLevel, m_sceneCoordinate, m_viewSize);
    emit newMapResolution(sceneResolution());
}

void MapEngine::setZoomLevel(int newZoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomLevel = newZoomLevel;
    zoomed();
}

void MapEngine::setTilesGridSize(const QSize &viewSize)
{
    qDebug() << __PRETTY_FUNCTION__;

    // there must be scrolling reserve of at least half tile added to tile amount
    // calculated from view size
    const qreal SCROLLING_RESERVE = 0.5;

    // converting scene tile to tile number does cause grid centering inaccuracy of one tile
    const int CENTER_TILE_INACCURACY = 1;

    int gridWidth = ceil(qreal(viewSize.width()) / TILE_SIZE_X + SCROLLING_RESERVE)
                    + CENTER_TILE_INACCURACY + (GRID_PADDING * 2);
    int gridHeight = ceil(qreal(viewSize.height()) / TILE_SIZE_Y + SCROLLING_RESERVE)
                     + CENTER_TILE_INACCURACY + (GRID_PADDING * 2);

    m_mapFetcher->setDownloadQueueSize(gridWidth * gridHeight);

    m_tilesGridSize.setHeight(gridHeight);
    m_tilesGridSize.setWidth(gridWidth);
}

void MapEngine::setViewLocation(QPointF latLonCoordinate)
{
    qDebug() << __PRETTY_FUNCTION__;

    QPoint sceneCoordinate = convertLatLonToSceneCoordinate(latLonCoordinate);

    m_lastManualPosition = sceneCoordinate;

    setLocation(sceneCoordinate);
}

int MapEngine::tileMaxIndex(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    // subtract one because first tile index is zero
    return tilesPerSide(zoomLevel) - 1;
}

QString MapEngine::tilePath(int zoomLevel, int x, int y)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString tilePathString(QString::number(zoomLevel) + "/");
    tilePathString.append(QString::number(x) + "/");
    tilePathString.append(QString::number(y));

    return tilePathString;
}

int MapEngine::tilesPerSide(int zoomLevel)
{
    return (1 << zoomLevel);
}

void MapEngine::updateViewTilesSceneRect()
{
    qDebug() << __PRETTY_FUNCTION__;

    const QPoint ONE_TILE = QPoint(1, 1);
    const QPoint ONE_PIXEL = QPoint(1, 1);

    QPoint topLeft = convertTileNumberToSceneCoordinate(m_zoomLevel, m_viewTilesGrid.topLeft());
    // one tile - one pixel is added because returned coordinates are pointing to upper left corner
    // of the last tile.
    QPoint bottomRight = convertTileNumberToSceneCoordinate(m_zoomLevel,
                                                            m_viewTilesGrid.bottomRight()
                                                             + ONE_TILE) - ONE_PIXEL;

    m_mapScene->tilesSceneRectUpdated(QRect(topLeft, bottomRight));
}

void MapEngine::viewResized(const QSize &size)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_viewSize = size;
    setTilesGridSize(m_viewSize);

    emit locationChanged(m_sceneCoordinate);
    getTiles(m_sceneCoordinate);
    m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    m_mapScene->setSceneVerticalOverlap(m_viewSize.height(), m_zoomLevel);
}

void MapEngine::viewZoomFinished()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomedIn) {
        m_zoomedIn = false;
        m_mapScene->removeOutOfViewTiles(m_viewTilesGrid, m_zoomLevel);
    }

    if (m_zoomLevel == MAX_MAP_ZOOM_LEVEL)
        emit maxZoomLevelReached();
    else if (m_zoomLevel == MIN_VIEW_ZOOM_LEVEL)
        emit minZoomLevelReached();
}

void MapEngine::zoomed()
{
    emit zoomLevelChanged(m_zoomLevel);
    m_mapScene->setTilesDrawingLevels(m_zoomLevel);
    m_mapScene->setZoomLevel(m_zoomLevel);
    getTiles(m_sceneCoordinate);
    m_mapScene->setSceneVerticalOverlap(m_viewSize.height(), m_zoomLevel);
    m_mapScene->spanItems(m_zoomLevel, m_sceneCoordinate, m_viewSize);
    emit newMapResolution(sceneResolution());
}

void MapEngine::zoomIn()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomLevel < MAX_MAP_ZOOM_LEVEL) {
        m_zoomLevel++;
        m_zoomedIn = true;
        zoomed();
    }
}

void MapEngine::zoomOut()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_zoomLevel > MIN_VIEW_ZOOM_LEVEL) {
        m_zoomLevel--;
        zoomed();
    }
}
