/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QVBoxLayout>
#include <QPushButton>
#include <QPainter>
#include <QDebug>
#include <QPaintEvent>
#include <QLabel>
#include <QPixmap>
#include <QFormLayout>
#include <QSpacerItem>
#include <QStylePainter>
#include <math.h>

#include "friendlistitem.h"
#include "../user/user.h"
#include "imagebutton.h"
#include "../common.h"

const int BACKGROUND_BOTTOM_HEIGHT = 15;
const int BACKGROUND_TOP_HEIGHT = 20;
const int BACKGROUND_WIDTH = 368;
const int ICON_HEIGHT = 24;     ///< Icon height
const int ICON_WIDTH = 24;       ///< Icon width
const int IMAGE_HEIGHT = 64; ///< Friend image height
const int IMAGE_WIDTH = 64;  ///< Friend image width
const int ITEM_MIN_HEIGHT = 141; ///< Minimum height for item
const int MARGIN = 5;   ///< Icon margin
const int MOUSE_PRESS_AREA_WIDTH = 20;  ///< Area width for item height toggling
const int MOUSE_PRESS_AREA_HEIGHT = 20; ///< Area height for item height toggling

/**
* @var NAME_LABEL_MAX_WIDTH
*
* @brief Name label's maximum width
*/
const int NAME_LABEL_MAX_WIDTH = BACKGROUND_WIDTH - 3*MARGIN - IMAGE_WIDTH;

/**
* @var LABEL_MAX_WIDTH
*
* @brief All label's maximum width
*/
const int LABEL_MAX_WIDTH = BACKGROUND_WIDTH - 3 * MARGIN - IMAGE_WIDTH - MARGIN - ICON_WIDTH;

const int WALK_DISTANCE = 5;        ///< Walk distance limit for distance icon
const int CAR_DISTANCE = 500;       ///< Car distance limit for distance icon
const int AEROPLANE_DISTANCE = 5000;///< Aeroplane distance limit for distance icon

FriendListItem::FriendListItem(QWidget *parent)
    : QWidget(parent)
    , m_expanded(false)
    , m_user(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setContentsMargins(MARGIN, 0, MARGIN*2, MARGIN*2);
    layout->setSpacing(0);
    setLayout(layout);

    QHBoxLayout *topLayout = new QHBoxLayout();
    topLayout->setMargin(0);
    topLayout->setSpacing(0);

    QVBoxLayout *distanceLayout = new QVBoxLayout();
    distanceLayout->setMargin(0);
    distanceLayout->setSpacing(0);

    QHBoxLayout *bottomLayout = new QHBoxLayout();
    bottomLayout->setMargin(0);
    bottomLayout->setSpacing(0);

    QFormLayout *infoLayout = new QFormLayout();
    infoLayout->setMargin(0);
    infoLayout->setSpacing(0);
    infoLayout->setLabelAlignment(Qt::AlignTop);

    QLabel *clockLabel = new QLabel();
    clockLabel->setPixmap(QPixmap(":/res/images/clock.png"));
    clockLabel->setContentsMargins(0, 0, MARGIN, 0);
    clockLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *envelopeLabel = new QLabel();
    envelopeLabel->setPixmap(QPixmap(":/res/images/envelope.png"));
    envelopeLabel->setContentsMargins(0, 0, MARGIN, 0);
    envelopeLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *compassLabel = new QLabel();
    compassLabel->setPixmap(QPixmap(":/res/images/compass.png"));
    compassLabel->setContentsMargins(0, 0, MARGIN, 0);
    compassLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);

    m_nameLabel = new QLabel();
    m_nameLabel->setFixedHeight(IMAGE_HEIGHT);

    m_distanceTextLabel = new QLabel();
    m_distanceTextLabel->setFixedHeight(ICON_HEIGHT);

    m_distanceImageLabel = new QLabel();
    m_distanceImageLabel->setFixedSize(ICON_WIDTH, ICON_HEIGHT);

    m_findButton = new ImageButton(this);

    m_updatedLabel = new QLabel();
    m_updatedLabel->setWordWrap(true);
    m_statusTextLabel = new QLabel();
    m_statusTextLabel->setWordWrap(true);
    m_locationLabel = new QLabel();
    m_locationLabel->setWordWrap(true);

    distanceLayout->addWidget(m_distanceImageLabel, 0, Qt::AlignRight);
    distanceLayout->addWidget(m_distanceTextLabel, 0, Qt::AlignRight);

    infoLayout->addRow(envelopeLabel, m_statusTextLabel);
    infoLayout->addRow(compassLabel, m_locationLabel);
    infoLayout->addRow(clockLabel, m_updatedLabel);

    topLayout->addWidget(m_findButton);
    topLayout->addWidget(m_nameLabel, 1);
    topLayout->addLayout(distanceLayout);

    bottomLayout->addSpacing(IMAGE_WIDTH);
    bottomLayout->addLayout(infoLayout);

    layout->addLayout(topLayout, 0);
    layout->addLayout(bottomLayout, 1);

    setMinimumSize(BACKGROUND_WIDTH, ITEM_MIN_HEIGHT);
    setMaximumWidth(BACKGROUND_WIDTH);

    setFont(NOKIA_FONT_SMALL);
    m_nameLabel->setFont(NOKIA_FONT_NORMAL);
    QPalette itemPalette = palette();
    itemPalette.setColor(QPalette::Foreground, COLOR_GRAY);
    setPalette(itemPalette);
    QPalette namePalette = m_nameLabel->palette();
    namePalette.setColor(QPalette::Foreground, Qt::white);
    m_nameLabel->setPalette(namePalette);

    m_backgroundTopImage.load(":/res/images/list_item_top.png");
    m_backgroundMiddleImage.load(":/res/images/list_item_middle.png");
    m_backgroundBottomImage.load(":/res/images/list_item_bottom.png");

    connect(m_findButton, SIGNAL(clicked()),
        this, SLOT(findButtonClicked()));
}

void FriendListItem::setData(User *user)
{
    qDebug() << __PRETTY_FUNCTION__;

    if(user) {
        m_user = user;

        if (!m_user->profileImage().isNull())
            m_findButton->setButtonIcon(m_user->profileImage());

        QString unit;
        double value;
        user->distance(value, unit);
        m_distanceTextLabel->setText(QString::number(value) + " " + unit);
        setDistanceIcon(value, unit);

        shortenTexts();
        setText(false);
    }
}

void FriendListItem::setDistanceIcon(double value, const QString &unit)
{
    QPixmap distanceImage;

    if ((unit == "m") || (value < WALK_DISTANCE))
        distanceImage.load(":/res/images/walk_icon_gray.png");
    else if (value < CAR_DISTANCE)
        distanceImage.load(":/res/images/car_icon_gray.png");
    else if (value < AEROPLANE_DISTANCE)
        distanceImage.load(":/res/images/aeroplane_icon_gray.png");
    else
        distanceImage.load(":/res/images/rocket_icon_gray.png");

    m_distanceImageLabel->setPixmap(distanceImage);
}

void FriendListItem::shortenTexts()
{
    qDebug() << __PRETTY_FUNCTION__;

    QFontMetrics nameLabelMetrics = m_nameLabel->fontMetrics();
    QFontMetrics otherLabelsMetrics = m_updatedLabel->fontMetrics();

    QString name = m_user->name();
    QString updated = m_user->timestamp();
    QString statusText = m_user->note();
    QString location = m_user->address();

    int nameIndex = name.indexOf('\n');
    int updatedIndex = updated.indexOf('\n');
    int statusTextIndex = statusText.indexOf('\n');
    int locationIndex = location.indexOf('\n');

    if (nameIndex > 0) {
        name.truncate(nameIndex);
        name.append("...");
    }
    if (updatedIndex > 0) {
        updated.truncate(updatedIndex);
        updated.append("...");
    }
    if (statusTextIndex > 0) {
        statusText.truncate(statusTextIndex);
        statusText.append("...");
    }
    if (locationIndex > 0) {
        location.truncate(locationIndex);
        location.append("...");
    }

    int distanceLabelWidth = otherLabelsMetrics.width(m_distanceTextLabel->text());
    m_shortenedName = nameLabelMetrics.elidedText(name, Qt::ElideRight, NAME_LABEL_MAX_WIDTH
                                                  - distanceLabelWidth);
    m_shortenedUpdated = otherLabelsMetrics.elidedText(updated, Qt::ElideRight, LABEL_MAX_WIDTH);
    m_shortenedStatusText = otherLabelsMetrics.elidedText(statusText, Qt::ElideRight,
                                                          LABEL_MAX_WIDTH);
    m_shortenedLocation = otherLabelsMetrics.elidedText(location, Qt::ElideRight, LABEL_MAX_WIDTH);
}

void FriendListItem::setText(bool expanded)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (expanded) {
        setUpdatesEnabled(false);
        m_nameLabel->setText(m_shortenedName);
        m_updatedLabel->setText(m_user->timestamp());
        m_statusTextLabel->setText(m_user->note());
        m_locationLabel->setText(m_user->address());
        setUpdatesEnabled(true);
    }
    else {
        setUpdatesEnabled(false);
        m_nameLabel->setText(m_shortenedName);
        m_updatedLabel->setText(m_shortenedUpdated);
        m_statusTextLabel->setText(m_shortenedStatusText);
        m_locationLabel->setText(m_shortenedLocation);
        setUpdatesEnabled(true);
    }
}

void FriendListItem::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mousePosition = event->pos();
}

void FriendListItem::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    if ((abs(m_mousePosition.y() - event->pos().y()) <= MOUSE_PRESS_AREA_WIDTH) &&
        (abs(m_mousePosition.x() - event->pos().x()) <= MOUSE_PRESS_AREA_HEIGHT)) {
        if (m_expanded) {
            setText(false);
            m_expanded = false;
        }
        else {
            setText(true);
            m_expanded = true;
        }
    }
}

void FriendListItem::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(event);

    QPainter painter(this);

    QRect topRect = QRect(0, 0, BACKGROUND_WIDTH, BACKGROUND_TOP_HEIGHT);
    QRect middleRect = QRect(0, topRect.bottom(), BACKGROUND_WIDTH,
                             height() - BACKGROUND_TOP_HEIGHT - BACKGROUND_BOTTOM_HEIGHT);
    QRect bottomRect = QRect(topRect.left(), middleRect.bottom(), BACKGROUND_WIDTH,
                             BACKGROUND_BOTTOM_HEIGHT);

    painter.drawPixmap(topRect, m_backgroundTopImage);
    painter.drawPixmap(middleRect, m_backgroundMiddleImage);
    painter.drawPixmap(bottomRect, m_backgroundBottomImage);
}

void FriendListItem::findButtonClicked()
{
    qDebug() << __PRETTY_FUNCTION__;

    emit findFriend(m_user->coordinates());
}
