/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef LIBLOCATIONWRAPPER_H
#define LIBLOCATIONWRAPPER_H

#include <QObject>
#include "geopositioninfo.h"

extern "C"
{
    #include <location/location-gps-device.h>
    #include <location/location-gpsd-control.h>
}

/**
* @brief LiblocationWrapper class is a wrapper for liblocation library.
*/
class LiblocationWrapper : public QObject
{
    Q_OBJECT

public:
    /**
    * @brief Constructor.
    */
    LiblocationWrapper(QObject *parent);

    /**
    * @brief Destructor.
    *
    * Releases m_control and m_device objects.
    */
    ~LiblocationWrapper();

    /**
    * GPS state.
    */
    enum State {Undefined, Initialized, Running};

/******************************************************************************
* MEMBER FUNCTIONS AND SLOTS
******************************************************************************/
    /**
    * @brief Checks if GPS is running.
    *
    * @return true if running, false otherwise
    */
    bool isRunning();

    /**
    * @brief Returns last known position.
    *
    * @return GeoPositionInfo
    */
    GeoPositionInfo lastKnownPosition() const;

    /**
    * @brief Starts GPS updates.
    */
    void startUpdates();

    /**
    * @brief Sets last known position.
    *
    * @param positionInfo GeoPositionInfo
    */
    void setLastKnownPosition(const GeoPositionInfo &positionInfo);

    /**
    * @brief Initializes GPS with update interval.
    *
    * @param updateInterval update interval in milliseconds
    */
    void init(int updateInterval);

    /**
    * @brief Stops GPS updates.
    */
    void stopUpdates();

private:
    /**
    * @brief Called from liblocation when GPS location changes.
    *
    * @param device LocationGPSDevice
    * @param data qpointer
    */
    static void changed(LocationGPSDevice *device, gpointer data);

    /**
    * @brief Called from liblocation when there is an error on GPS.
    *
    * @param device LocationGPSDevice
    * @param data qpointer
    */
    static void error(LocationGPSDevice *device, gpointer data);

    /**
    * @brief Sets state.
    *
    * @param state State
    */
    void setState(int state);

    /**
    * Returns state.
    *
    * State state
    */
    int state();

/******************************************************************************
* SIGNALS
******************************************************************************/
signals:
    /**
    * @brief Signal for location change.
    *
    * @param positionInfo GeoPositionInfo
    */
    void locationChanged(const GeoPositionInfo &positionInfo);

    /**
    * @brief Signal for error.
    *
    * @param message error message
    */
    void errorMessage(const QString &message);


/******************************************************************************
* DATA MEMBERS
******************************************************************************/
private:
    LocationGPSDControl *m_control;         ///< LocationGPSDControl pointer
    LocationGPSDevice *m_device;            ///< LocationGPSDevice pointer
    GeoPositionInfo m_lastKnownPosition;    ///< GeoPositionInfo data
    int m_state;                            ///< GPS state
};

#endif // LIBLOCATIONWRAPPER_H
