/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAPVIEW_H
#define MAPVIEW_H

#include <QGraphicsView>

class QPropertyAnimation;

/**
* @brief Map view widget
*
* @author Sami Rämö - sami.ramo (at) ixonos.com
* @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
*/
class MapView : public QGraphicsView
{
    Q_OBJECT

    /**
    * @brief View scaling
    *
    * @property viewScale
    */
    Q_PROPERTY(qreal viewScale READ viewScale WRITE setViewScale)

public:
    /**
    * @brief Constructor
    *
    * @param parent Parent
    */
    MapView(QWidget *parent = 0);

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
protected:
    /**
    * @brief Called when view is resized.
    *
    * @param event resize event
    */
    void resizeEvent(QResizeEvent *event);

private:
    /**
    * @brief Event handler for mouse move events
    *
    * Does calculate mouse movement delta from last event position and new view center
    * based on that delta. Saves current event position for next round. Emits viewScrolled
    * signal and doesn't actually scroll the view.
    * @param event Mouse event
    */
    void mouseMoveEvent(QMouseEvent *event);

    /**
    * @brief Event handler for mouse press events
    *
    * Saves inial values for mouse and scene location for dragging view.
    * @param event Mouse event
    */
    void mousePressEvent(QMouseEvent *event);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:
    /**
    * @brief Slot for centering view to new location
    *
    * @param sceneCoordinate Scene coordinates of the new center point
    */
    void centerToSceneCoordinates(QPoint sceneCoordinate);

    /**
    * @brief Set zoom level of the view
    *
    * @param zoomLevel Zoom level
    */
    void setZoomLevel(int zoomLevel);

private:
    /**
    * @brief Set new view scale
    *
    * @param viewScale New scaling factor
    */
    void setViewScale(qreal viewScale);

    /**
    * @brief Get current view scale
    *
    * @return Current view scaling factor
    */
    qreal viewScale();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signal for view resize events.
    *
    * Signal is emitted when view has been resized.
    * @param size view size
    */
    void viewResized(const QSize &size);

    /**
    * @brief Signal for view scroll events
    *
    * Signal is emitted when view is scrolled.
    * @param sceneCoordinate Scene coordinates of the new center point of the view
    */
    void viewScrolled(QPoint sceneCoordinate);

    /**
    * @brief Signal for informing that zooming animation is finished
    */
    void viewZoomFinished();

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QPoint m_mousePosition;               ///< Previous mouse event position
    QPoint m_scenePosition;               ///< New center position
    QPropertyAnimation *m_zoomAnimation;  ///< Zoom animation
};

#endif // MAPVIEW_H
