/*
 * This file is part of simple-fmtx-widget
 *
 * Copyright (C) 2009 Pekka Rönkkö. All rights reserved.
 *
 * This code is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the root
 * directory.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libosso.h>
#include "simple-fmtx-widget.h"

HD_DEFINE_PLUGIN_MODULE (SimpleFmtxWidget, simple_fmtx_widget_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

#define SIMPLE_FMTX_WIDGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                        SIMPLE_TYPE_FMTX_WIDGET, SimpleFmtxWidgetPrivate))

typedef struct _SimpleFmtxWidgetPrivate SimpleFmtxWidgetPrivate;
struct _SimpleFmtxWidgetPrivate
{
    gboolean fmtx_state;
    osso_context_t *osso_context;
    GtkWidget *button;
    GtkWidget *label;
};
typedef enum
{
    GET,
    SET,
} PropertyCommand;
/*****************************************************************************************/
static gpointer get_set(PropertyCommand commad, const gchar* val, gboolean state_val);
static void set_freq_label(SimpleFmtxWidget *self);
static gboolean get_current_state(SimpleFmtxWidget *self);
static void button_clicked (HildonButton *button, gpointer user_data);
static gboolean event_box_clicked(GtkWidget *widget,
                           GdkEventButton *event,
                           gpointer user_data);
static GtkWidget *build_ui (SimpleFmtxWidget *self);
/*****************************************************************************************/

static gboolean get_current_state(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);
    
    gchar* current_state = (gchar*)get_set(GET, "state", FALSE);

    if (g_strcmp0(current_state, "enabled") == 0)
    {
        priv->fmtx_state = TRUE;
        hildon_button_set_title((HildonButton*)priv->button, "OFF");
        g_free(current_state);
        return TRUE;
    }
    else if (g_strcmp0(current_state, "disabled") == 0)
    {
        priv->fmtx_state = FALSE;
	    hildon_button_set_title((HildonButton*)priv->button, "ON");
        g_free(current_state);
        return FALSE;
    }
    g_free(current_state);
    return FALSE;
}
static void set_freq_label(SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

    gchar* freq = NULL;
    guint32 f = (guint32)get_set(GET, "frequency", FALSE);
    gdouble f_d = (gdouble)f/1000;
    freq = g_strdup_printf("%.2f",f_d);
    gtk_label_set_text(GTK_LABEL(priv->label), freq);
    g_free(freq);

}
static void button_clicked (HildonButton *button, gpointer user_data)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (user_data);

    if (priv->fmtx_state == TRUE)
    {
	    get_set(SET, "state", FALSE);
    }
    else if (priv->fmtx_state == FALSE)
    {
	    get_set(SET, "state", TRUE);
    }
        get_current_state(user_data);
}
static gboolean event_box_clicked(GtkWidget *widget,
                           GdkEventButton *event,
                           gpointer user_data)
{
   SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (user_data);

   if (priv->osso_context)
   {
	    osso_cp_plugin_execute(priv->osso_context, "libcpfmtx.so", NULL, TRUE);
	    get_current_state(user_data);
        set_freq_label(user_data);
   }
   return TRUE;
}
static GtkWidget *build_ui (SimpleFmtxWidget *self)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (self);

    GtkWidget *hbox = gtk_hbox_new(0, FALSE);
    priv->label = gtk_label_new("--.--");
    priv->button = hildon_button_new(HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

    GtkWidget *event_box = gtk_event_box_new();
    g_signal_connect (event_box, "button-press-event", G_CALLBACK (event_box_clicked), self);

    hildon_button_set_title((HildonButton*)priv->button, "");
    g_signal_connect (priv->button, "clicked", G_CALLBACK (button_clicked), self);


    gtk_container_add(GTK_CONTAINER (event_box), priv->label);
    gtk_container_add(GTK_CONTAINER (hbox), event_box);
    gtk_container_add(GTK_CONTAINER (hbox), priv->button);

    gtk_widget_set_size_request(priv->button, 100, 100);
    gtk_widget_set_size_request(priv->label, 100, 100);

    gtk_widget_show_all(GTK_WIDGET (hbox));

    return GTK_WIDGET(hbox);
}
static void simple_fmtx_widget_plugin_init (SimpleFmtxWidget *desktop_plugin)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (desktop_plugin);
    priv->osso_context = osso_initialize(PACKAGE, PACKAGE_VERSION, TRUE, NULL);
    GtkWidget *contents = build_ui (desktop_plugin);
    set_freq_label(desktop_plugin);
    get_current_state(desktop_plugin);
    gtk_container_add (GTK_CONTAINER (desktop_plugin), contents);
}
static void simple_fmtx_widget_plugin_finalize(GObject *object)
{
    SimpleFmtxWidgetPrivate *priv = SIMPLE_FMTX_WIDGET_GET_PRIVATE (object);
    osso_deinitialize(priv->osso_context);
    priv->osso_context = NULL;
}
static void simple_fmtx_widget_plugin_class_init (SimpleFmtxWidgetClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    object_class->finalize = simple_fmtx_widget_plugin_finalize;
    g_type_class_add_private (class, sizeof (SimpleFmtxWidgetPrivate));
}
static void simple_fmtx_widget_plugin_class_finalize (SimpleFmtxWidgetClass *class)
{
}
static gpointer get_set(PropertyCommand commad, const gchar* val, gboolean state_val)
{
    DBusGConnection *connection;
    GError *error = NULL;
    DBusGProxy *proxy = NULL;
    GValue value = { 0 };
    gchar* command = NULL;

    connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (connection == NULL || error)
    {
        if (error)
        {
            g_clear_error (&error);
        }
        return 0;
    }

    proxy = dbus_g_proxy_new_for_name (connection, "com.nokia.FMTx",
                                       "/com/nokia/fmtx/default",
                                       "org.freedesktop.DBus.Properties");

    if (proxy == NULL)
    {
        return NULL;
    }

    switch (commad)
    {
        case SET:
            command = g_strdup("Set");
            break;
        case GET:
            command = g_strdup("Get");
            break;
        default:
            break;
    }

    if (g_strcmp0(val, "frequency") == 0)
    {
        dbus_g_proxy_call (proxy, command,
                           &error,
                           G_TYPE_STRING,
                           "org.freedesktop.DBus.Properties",
                           G_TYPE_STRING,
                           val,
                           G_TYPE_INVALID,
                           G_TYPE_VALUE,
                           &value,
                           G_TYPE_INVALID);

        if (error)
        {
            g_clear_error (&error);
            g_value_unset (&value);
            g_free(command);
            return NULL;
        }

        guint32 retval = g_value_get_uint(&value);
        g_value_unset (&value);
        g_free(command);
        return (gpointer)retval;
    }
    else if (g_strcmp0(val, "state") == 0)
    {
	    if (commad == GET)
	    {
	    	dbus_g_proxy_call (proxy, command,
	    	                   &error,
	    	                   G_TYPE_STRING,
	    	                   "org.freedesktop.DBus.Properties",
	    	                   G_TYPE_STRING,
	    	                   val,
	    	                   G_TYPE_INVALID,
	    	                   G_TYPE_VALUE,
	    	                   &value,
	    	                   G_TYPE_INVALID);

	    	if (error)
	    	{
	    	    g_clear_error (&error);
	    	    g_value_unset (&value);
	    	    g_free(command);
	    	    return NULL;
	    	}

	    	gchar* retval = g_strdup(g_value_get_string(&value));
	    	g_value_unset (&value);
	    	g_free(command);
	    	return (gchar*)retval;
	    }
	    else if (commad == SET)
	    {
	    	g_value_init(&value, G_TYPE_STRING);
		    switch (state_val)
		    {
		    case FALSE:
		        	g_value_set_string(&value, "disabled");
		        break;
		    case TRUE:
		       	g_value_set_string(&value, "enabled");
		        break;
		    default:
		        break;
		    }
		    dbus_g_proxy_call (proxy, command,
		                       &error,
		                       G_TYPE_STRING,
		                       "org.freedesktop.DBus.Properties",
		                       G_TYPE_STRING,
		                       val,
		                       G_TYPE_VALUE,
		                       &value,
		                       G_TYPE_INVALID,
		                       G_TYPE_INVALID);

		    if (error)
		    {
		        hildon_banner_show_information(NULL, NULL, error->message);
		        g_clear_error (&error);
		        g_value_unset (&value);
		        g_free(command);
		        return NULL;
		    }
	    }
    }
    g_free(command);
    return NULL;
}

