#include <QFont>
#include <QPainter>
#include <QListIterator>
#include <QPen>

#include "expensesgraph.h"
#include "expense.h"

#define Y_LABEL_WIDTH 30
#define Y_LABEL_HEIGHT 30
#define X_LABEL_WIDTH 30
#define X_LABEL_HEIGHT 10

#define Y_DIVISIONS 10

ExpensesGraph::ExpensesGraph(QWidget *parent)
        : QWidget(parent)
{
        _padding = 10;
        _accumulated = true;
        _month = NULL;
}

void ExpensesGraph::setPadding(int padding)
{
        _padding = padding;
        repaint();
}

void ExpensesGraph::setMonthData(MonthData *month)
{
        _month = month;
}

void ExpensesGraph::setAccumulated(bool accumulated)
{
        _accumulated = accumulated;
        repaint();
}

bool ExpensesGraph::accumulated()
{
        return _accumulated;
}

void ExpensesGraph::changeView()
{
        setAccumulated(!_accumulated);
}

QSize ExpensesGraph::minimumSizeHint() const
{
        return QSize(200, 100);
}

QSize ExpensesGraph::sizeHint() const
{
        return QSize(400, 200);
}

void ExpensesGraph::paintEvent (QPaintEvent *)
{
        QPainter painter(this);
        int aux;

        xStart = _padding + Y_LABEL_WIDTH + _padding;
        xEnd = width() - _padding;
        sceneWidth = xEnd - xStart;

        /* watch out with the reversed Y */
        yStart = height() - (_padding + X_LABEL_HEIGHT + _padding);
        yEnd = _padding;
        sceneHeight = yStart - yEnd;


        QFont font = painter.font();
        font.setPointSize(10);
        painter.setFont(font);
        painter.fillRect(0, 0, width(), height(), QColor(Qt::black));

        drawScene(painter);

        if (_month) {
                /* labels in x axis */
                xStepsPerLabel = _month->days() / qMin(sceneWidth / X_LABEL_WIDTH, _month->days());
                /* labels in y axis. not more than 10 */
                yStepsPerLabel = Y_DIVISIONS / qMin(sceneHeight / Y_LABEL_HEIGHT, Y_DIVISIONS);

                dailyExpense = _month->budget() / _month->days();
                if (_accumulated) {
                        if (_month->budget() >= _month->totalExpense()) {
                                yMaxValue = _month->budget();
                        } else {
                                aux = _month->totalExpense() / 100;
                                yMaxValue = (aux + 1) * 100;
                        }
                } else {
                        aux = qMax(_month->maxExpense(), dailyExpense) / 100;
                        yMaxValue = (aux + 1) * 100;
                }
                yScale = sceneHeight / (double)yMaxValue;

                xStep = sceneWidth / (double)(_month->days()-1);
                yStep = sceneHeight / (double)Y_DIVISIONS;

                painter.setRenderHint(QPainter::Antialiasing);
                drawGrid(painter);

                drawReference(painter);
                drawExpenses(painter);
        }
}

void ExpensesGraph::drawScene (QPainter &painter)
{
        QPen pen;

        painter.save();
        pen.setColor(Qt::white);
        painter.setPen(pen);
        painter.drawLine(xStart, yStart, xStart, yEnd);
        painter.drawLine(xStart, yStart, xEnd, yStart);

        painter.restore ();
}

void ExpensesGraph::drawGrid (QPainter &painter)
{
        int i;
        double y, x;
        double yLabelValue, yInc;
        QPen pen;

        painter.save();

        pen.setStyle(Qt::DotLine);
        pen.setColor(Qt::gray);
        pen.setWidth(1);
        painter.setPen(pen);

        /* vertical divisions  and y labels*/
        y = yStart;
        yLabelValue = 0;
        yInc = yMaxValue / (double)Y_DIVISIONS;
        for (i = 0; i <= Y_DIVISIONS; i+=yStepsPerLabel) {
                y -= yStepsPerLabel * yStep;
                yLabelValue += yStepsPerLabel * yInc;
                if (i != Y_DIVISIONS - 1) {
                        painter.drawLine (xStart + 2, y, xEnd, y);
                }
                painter.setPen(Qt::white);
                painter.drawText (_padding, y - (Y_LABEL_HEIGHT / 2), Y_LABEL_WIDTH, Y_LABEL_HEIGHT, 
                                  Qt::AlignRight | Qt::AlignVCenter,
                                  QString("%1").arg(qRound(yLabelValue)));
                painter.setPen(pen);
        }
        /* horizontal divisions and x labels*/
        painter.setPen(Qt::white);
        painter.drawText (xStart - (X_LABEL_WIDTH / 2),
                          yStart + _padding,
                          X_LABEL_WIDTH,
                          X_LABEL_HEIGHT,
                          Qt::AlignCenter,
                          QString("1"));
        painter.setPen(pen);
        x = xStart + (xStep *xStepsPerLabel);
        for (i = xStepsPerLabel + 1; i<=_month->days(); i+=xStepsPerLabel) {
                if (i != _month->days()) {
                        painter.drawLine (qRound(x), yStart, qRound(x), yEnd);
                }
                painter.setPen(Qt::white);
                painter.drawText (qRound(x) - (X_LABEL_WIDTH / 2),
                                  yStart + _padding,
                                  X_LABEL_WIDTH,
                                  X_LABEL_HEIGHT,
                                  Qt::AlignCenter,
                                  QString("%1").arg(i));
                painter.setPen(pen);
                x += xStep * xStepsPerLabel;
        }

        /* current */
        if (_month->isCurrent()) {
                i = QDate::currentDate().day() - 1;
                painter.setPen (Qt::yellow);
                painter.drawLine (qRound(xStart + (i * xStep)),
                                  yStart,
                                  qRound(xStart + (i * xStep)),
                                  yEnd);
        }


        painter.restore();
}

void ExpensesGraph::drawReference (QPainter &painter)
{
        painter.save();
        painter.setPen(Qt::green);

        if (_accumulated) {
                painter.drawLine (xStart + 1,
                                  qRound(yStart - (dailyExpense * yScale)),
                                  xEnd,
                                  qRound(yStart - (_month->budget() * yScale)));
        } else {
                painter.drawLine (xStart + 1,
                                  qRound(yStart - (dailyExpense * yScale)),
                                  xEnd,
                                  qRound(yStart - (dailyExpense * yScale)));
        }

        painter.restore();
}

void ExpensesGraph::drawExpenses (QPainter &painter)
{
        QPainterPath path;
        int i;
        double x, y;

        x = xStart + 1;
        y = yStart;
        path.moveTo (x, y);

        for (i=1; i<=_month->days(); i++) {
                if (_accumulated) {
                        y -= _month->summary()[i] * yScale;
                } else {
                        y = yStart - (_month->summary()[i] * yScale);
                }
                if (i == 1) {
                        path.moveTo (x, y);
                } else {
                        path.lineTo (x, y);
                }
                x += xStep;
        }

        QPen pen;
        painter.save();
        pen.setColor(Qt::red);
        pen.setWidth(2);
        painter.setPen(pen);
        painter.drawPath(path);
        painter.restore();
}
