#!/usr/bin/env python

import os

SCORE_FILE=os.environ['HOME']+'/.shisensho-scores-v2'
SCORE_LIMIT=15
VERSION='0.7'
VERSION_DATE='2013-04-23'

TILES_NAMES=['Classic', 'Modern']
TILES_NAMES_SCORE=['classic', 'modern']
TILES_DEF=[['/usr/share/pixmaps/osso-mahjong/tiles.png', 43, 2, range(36)], # 43 are usable
           ['/opt/shisensho/tiles-modern.png', 9, 4, range(36)]]
#TILES_DEF=[['osso/tiles.png', 43, 2, range(36)], # 43 are usable
#           ['tiles-modern.png', 9, 4, range(36)]]

DIF_NAMES=['Hard', 'Easy']
DIF_NAMES_SCORE=['hard', 'easy']
DIF_DEF=[[[14, 7, 36], [40, 50]],
         [[13, 6, 28], [45, 58]]]

FULLSCREEN=True

import pygtk
pygtk.require('2.0')
import gtk
import gobject
import pango
import operator
import time
import string
import random
#import code ## only for debug:  code.InteractiveConsole(locals()).interact()
try:
    import hildon
except ImportError:
    FULLSCREEN=False

def group_by(list, keyfn):
    group={}
    for item in list:
        key = keyfn(item)
        if group.has_key(key):
            group[key].append(item)
        else:
            group[key]=[item]
    return group


class Colors:
    def __init__(self, colormap):
        self.bg=colormap.alloc_color(gtk.gdk.color_parse('#060'))
        self.bg2=colormap.alloc_color(gtk.gdk.color_parse('#050'))
        self.label=colormap.alloc_color(gtk.gdk.color_parse('#FFF'))
        self.pause=colormap.alloc_color(gtk.gdk.color_parse('#FFB'))
        self.line=colormap.alloc_color(gtk.gdk.color_parse('#000'))
        self.mark=colormap.alloc_color(gtk.gdk.color_parse('#00F'))
        self.path=colormap.alloc_color(gtk.gdk.color_parse('#F00'))

class ShishenShoModel:
    def __init__(self, width, height, count):
        # note: not all width, height pairs satisfy (width*height)%2==0
        if not (width*height)%2==0:
            raise 'error: width*height must be even!'
        self.width = width; self.height = height; self.count = count
        self.newgame()
    def newgame(self):
        self.board=[None]*self.height
        for i in range(len(self.board)):
            self.board[i]=[0]*self.width
        self.area=self.width*self.height
        available=range(self.area)
        while True:
            elem=random.randint(0, self.count-1)
            address1=available.pop(random.randint(0,len(available)-1))
            address2=available.pop(random.randint(0,len(available)-1))
            self.board[address1 / self.width][address1 % self.width]=elem
            self.board[address2 / self.width][address2 % self.width]=elem
            if len(available)==0:
                break
        self.starttime=time.time()
        self.timeadd=0.0
        self.gamedone=False
        self.moves=0
    def pause(self):
        if self.starttime!=None and not self.gamedone:
            self.timeadd+=time.time()-self.starttime
            self.starttime=None
    def resume(self):
        if self.starttime==None and not self.gamedone:
            self.starttime=time.time()
    def get_time(self):
        if self.starttime!=None:
            return self.timeadd+(time.time()-self.starttime)
        else:
            return self.timeadd
    def is_pause(self):
        return self.starttime==None
    def is_done(self):
        return self.gamedone
    def empty_at(self,x,y):
        return self.board[y][x]==None
    def cell_at(self,x,y):
        return self.board[y][x]
    def can_move(self,x1,y1,x2,y2):
        if self.gamedone: return False
        if x1==x2 and y1==y2: return False
        if self.cell_at(x1,y1)==None: return False
        if self.cell_at(x1,y1)!=self.cell_at(x2,y2): return False
        empty='empty'
        fill='fill'
        choose=[empty,fill]
        tmpboard=[[empty]*(self.width+2)]
        for y in range(self.height):
            tmpboard.append([empty]+map(lambda x: choose[int(x!=None)], self.board[y])+[empty])
        tmpboard.append([empty]*(self.width+2))
        x1+=1; y1+=1; x2+=1; y2+=1
        tmpboard[y1][x1]=0
        tmpboard[y2][x2]=empty
        tmpxlen=len(tmpboard[0])
        tmpylen=len(tmpboard)
        dirboard=[[None]*(self.width+2) for x in range(self.height+2)]
        for distance in [1,2,3]:
            for y in range(tmpylen):
                for x in range(tmpxlen):
                    if tmpboard[y][x]==distance-1:
                        for xx in range(x+1,tmpxlen):
                            a=tmpboard[y][xx]
                            if a==fill: break
                            if a==empty:
                                tmpboard[y][xx]=distance
                                dirboard[y][xx]=(x,y)
                        for xx in range(x-1,-1,-1):
                            a=tmpboard[y][xx]
                            if a==fill: break
                            if a==empty:
                                tmpboard[y][xx]=distance
                                dirboard[y][xx]=(x,y)
                        for yy in range(y+1,tmpylen):
                            a=tmpboard[yy][x]
                            if a==fill: break
                            if a==empty:
                                tmpboard[yy][x]=distance
                                dirboard[yy][x]=(x,y)
                        for yy in range(y-1,-1,-1):
                            a=tmpboard[yy][x]
                            if a==fill: break
                            if a==empty:
                                tmpboard[yy][x]=distance
                                dirboard[yy][x]=(x,y)
#        print "\n".join(map(lambda line: ''.join(map(lambda item: '%5s' % item, line)), tmpboard))
#        print 'can move res:', tmpboard[y2][x2]
#        print "\n".join(map(lambda line: ''.join(map(lambda item: '%3s%2s' % (item or (0,0)), line)), dirboard))
        canmove=tmpboard[y2][x2]!=empty
        if canmove:
            path=[(x2-1,y2-1)]
            while (x2,y2)!=(x1,y1):
                x2,y2=dirboard[y2][x2]
                path.append((x2-1,y2-1))
            return path
        return False
    def move(self, x1, y1, x2, y2):
        path=self.can_move(x1, y1, x2, y2)
        if path:
            removed=self.board[y1][x1]
            self.board[y1][x1]=None
            self.board[y2][x2]=None
            self.moves+=1
            if 2*self.moves == self.area:
                self.timeadd+=time.time()-self.starttime
                self.starttime=None
                self.gamedone=True
            return path,removed
        else:
            return False
    def moves_left(self):
        return self.area-2*self.moves

class ShishenShoRenderer:
    def __init__(self):
        self.tileset=None
    def load_tileset(self, tileset_file, cells_per_imagew, cells_per_imageh, use_cells, scalecellw, scalecellh):
        self.tileset=gtk.gdk.pixbuf_new_from_file(tileset_file)
        self.tileset=self.tileset.scale_simple(scalecellw*cells_per_imagew,
                                               scalecellh*cells_per_imageh,
                                               gtk.gdk.INTERP_BILINEAR)
        self.cellw=self.tileset.get_width()/cells_per_imagew
        self.cellh=self.tileset.get_height()/cells_per_imageh
        self.cells_per_imagew=cells_per_imagew
        self.cells_per_imageh=cells_per_imageh
        self.use_cells=use_cells
        self.shiftx=self.cellw*3/5
        self.shifty=self.cellh*3/5
    def decodexy(self, widget, model, ax, ay):
        ax-=self.shiftx; ay-=self.shifty
        ax/=self.cellw; ay/=self.cellh
        if ax<0 or ax>=model.width or ay<0 or ay>=model.height:
            return False
        else:
            return (ax,ay)
    def draw(self, widget, model, markedcell=None, path=[]):
        def posx(x):
            return self.shiftx+x*self.cellw
        def posy(y):
            return self.shifty+y*self.cellh
        def drawcell(gc,x,y,cell):
            ucell=self.use_cells[cell]
            cellx=ucell % self.cells_per_imagew
            celly=ucell / self.cells_per_imagew
            widget.window.draw_pixbuf(gc, self.tileset,
                                      cellx*self.cellw, celly*self.cellh,
                                      posx(x), posy(y),
                                      self.cellw, self.cellh)
        if not self.tileset: return
        gc=widget.window.new_gc()
        gcm=widget.window.new_gc(foreground=widget.my_colors.mark,
                                 line_width=3, cap_style=gtk.gdk.CAP_ROUND, join_style=gtk.gdk.JOIN_ROUND)
        gcp=widget.window.new_gc(foreground=widget.my_colors.path,
                                 line_width=3, cap_style=gtk.gdk.CAP_ROUND, join_style=gtk.gdk.JOIN_ROUND)
        gcpa=widget.window.new_gc(foreground=widget.my_colors.pause)
        if model.is_done(): return
        if model.is_pause():
            layout=widget.create_pango_layout("Paused")
            layout.set_font_description(pango.FontDescription("sans bold 20"))
            layout_width, layout_height = layout.get_pixel_size()
            widget.window.draw_layout(gcpa,
                                      int(posx(model.width/2.0)) - int(layout_width/2),
                                      int(posy(model.height/2.0)) - int(layout_height/2), layout)
            return
        for y in range(model.height):
            for x in range(model.width):
                cell=model.cell_at(x,y)
                if cell!=None:
                    drawcell(gc,x,y,cell)
        if markedcell:
            x,y=markedcell
            widget.window.draw_rectangle(gcm, False,
                                         posx(x), posy(y),
                                         self.cellw, self.cellh)
        if path:
            pathline=path[0]
            pathcell=path[1]
            drawcell(gc,pathline[0][0],pathline[0][1],pathcell)
            drawcell(gc,pathline[-1][0],pathline[-1][1],pathcell)
            for i in range(len(pathline)-1):
                x1,y1=pathline[i]
                x2,y2=pathline[i+1]
                widget.window.draw_line(gcp,
                                        posx(x1)+self.cellw/2,
                                        posy(y1)+self.cellh/2,
                                        posx(x2)+self.cellw/2,
                                        posy(y2)+self.cellh/2)



class ShishenShoGame:
    def __init__(self):
        try:
            self.window = hildon.Window()
        except NameError:
            self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.window.set_title("Shisen Sho")
        self.window.connect("destroy", lambda w: gtk.main_quit())
        self.area = gtk.DrawingArea()
        self.area.set_size_request(670, 480)
        self.area.connect("expose-event", self.area_expose_cb)
        self.area.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        self.area.connect("button_press_event", self.area_press_event)
        self.renderer = ShishenShoRenderer()
        self.difficulty=0
        self.reinit_model()
        self.tilesetnr=0
        self.reload_renderer()
        self.area.my_colors=Colors(self.area.get_colormap())
        self.area.show()
        hbox=gtk.HBox()
        self.window.add(hbox)
        hbox.pack_start(self.area)
        buttonbox=gtk.VBox()
        self.label_info=gtk.Label("\n\n\n")
        self.button_new = gtk.Button(label="New")
        self.button_pause = gtk.Button(label="Pause")


        '''
        self.combo_tiles = gtk.combo_box_new_text()
        for tiles_name in TILES_NAMES:
            self.combo_tiles.append_text(tiles_name)
        self.combo_tiles.set_active(0)

        self.combo_dif = gtk.combo_box_new_text()
        for dif_name in DIF_NAMES:
            self.combo_dif.append_text(dif_name)
        self.combo_dif.set_active(0)
        '''

        self.button_score = gtk.Button(label="Scores")
        self.button_about = gtk.Button(label="About")
        #self.button_quit = gtk.Button(label="Quit")
        self.button_new.connect("button_press_event", self.action_new)
        self.button_pause.connect("button_press_event", self.action_pause)
        self.button_score.connect("button_press_event", self.action_score)
        self.button_about.connect("button_press_event", self.action_about)
        #self.button_quit.connect("button_press_event", self.action_quit)
        buttonbox.pack_start(self.label_info, False, False,1)
        buttonbox.pack_start(self.button_new, False, False,1)

        #buttonbox.pack_start(gtk.Label("  --  "),False, False,0)
        self.radio_tiles = gtk.RadioButton(group=None, label=TILES_NAMES[0])
        self.radio_tiles.set_active(0)

        buttonbox.pack_start(self.radio_tiles, False, False,1)
        self.radio_tiles = gtk.RadioButton(group=self.radio_tiles, label=TILES_NAMES[1])
        buttonbox.pack_start(self.radio_tiles, False, False,1)

        buttonbox.pack_start(gtk.Label("  ----  "),False, False,0)

        self.radio_dif = gtk.RadioButton(group=None, label=DIF_NAMES[0])
        self.radio_dif.set_active(0)

        buttonbox.pack_start(self.radio_dif, False, False,1)
        self.radio_dif = gtk.RadioButton(group=self.radio_dif, label=DIF_NAMES[1])
        buttonbox.pack_start(self.radio_dif, False, False,1)
        #buttonbox.pack_start(gtk.Label("  --  "),False, False,0)



        #buttonbox.pack_start(self.combo_dif, False, False,1)
        buttonbox.pack_start(self.button_pause, False, False,1)
        buttonbox.pack_start(self.button_score, False, False,1)
        buttonbox.pack_start(self.button_about, False, False,1)
        #buttonbox.pack_start(self.button_quit, False, False,5)
        buttonbox.show()
        buttonbox.set_size_request(130, 480)
        hbox.pack_end(buttonbox, False, False, 5)
        hbox.show_all()
        self.window.modify_bg(gtk.STATE_NORMAL,self.area.my_colors.bg2)
        self.area.modify_bg(gtk.STATE_NORMAL,self.area.my_colors.bg)
        self.label_info.modify_fg(gtk.STATE_NORMAL,self.area.my_colors.label)
        if FULLSCREEN: self.window.fullscreen()
        self.window.show()
        self.path=[]
        self.markedcell=None
        self.player_name='Player 1'
        self.scoretab=None
        self.redraw()
        self.clear_path_ID=None
        self.activate_clear_path()
        gobject.timeout_add(1000, self.update_timer)
    def update_timer(self):
        self.label_info.set_text("Time: %2d:%02d\nTiles left: %d"%
                                 (divmod(int(self.model.get_time()),60)+(self.model.moves_left(),)))
        return True
    def activate_clear_path(self):
        self.cancel_clear_path()
        self.clear_path_ID=gobject.timeout_add(400, self.clear_path_and_redraw)
    def cancel_clear_path(self):
        if self.clear_path_ID:
            gobject.source_remove(self.clear_path_ID)
    def clear_path_and_redraw(self):
        self.clear_path_ID=None
        self.path=[]
        self.redraw()
        return False
    def reload_renderer(self):
        self.renderer.load_tileset(*(TILES_DEF[self.tilesetnr]+DIF_DEF[self.difficulty][1]))
    def reinit_model(self):
        self.model = ShishenShoModel(*DIF_DEF[self.difficulty][0])
    def action_new(self, widget, event):
        self.markedcell=None
        self.path=[]
        rr=False
        dif=self.radio_dif.get_active()
        if dif!=self.difficulty:
            self.difficulty=dif
            self.reinit_model()
            rr=True
        self.model.newgame()
        nr=self.radio_tiles.get_active()
        if rr or nr!=self.tilesetnr:
            self.tilesetnr=nr
            self.reload_renderer()
        self.redraw()
    def action_pause(self, widget, event):
        if self.model.is_pause():
            self.model.resume()
            self.button_pause.set_label("Pause")
        else:
            self.model.pause()
            self.button_pause.set_label("Resume")
        self.redraw()
    def action_score(self, widget, event):
        self.show_score()
    def action_about(self,widget,event):
        dialog = gtk.Dialog(title='About Shisen Sho Game', parent=self.window, flags=gtk.DIALOG_MODAL)
        #                    buttons=(gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dialog.set_size_request(720,350)
        text1 = """
Shisen Sho - a Mah Jongg style game where you connect
pairs of tiles according to rules.

Copyright (C) 2007 Jakub Travnik, <jtra@seznam.cz>

This program is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 51 Franklin St, Fifth Floor,
Boston, MA 02110-1301 USA

Full text of the license may also be available at
/usr/share/shisensho/LICENSE (plain ASCII text file)

The game also uses tiles.png file from
osso-graphics-game-mahjong package. Which may come under
different terms.
"""
        text2 = """
Goal of the game is to remove all tiles in fastest time.

You can remove them by selecting a pair (two) of the
same-pattern tiles that can be connected. Tiles can be
connected by at most three joined vertical or horizontal
lines. Lines cannot cover present tiles except for the two
selected tiles. Lines can use border.

Game is done when all tiles are removed and time, player
name and board type description is noted in score table.

Game can be also stuck in state where no move is possible.
"""
        text3 = """
Version: 0.7
Aapo Rantalainen slightly modified game for Nokia N900/Maemo5.
--------------------
Version: 0.6
I (Jakub Travnik) have been inspired by commercial game set
called Kyodai Mahjongg somewhere in 2001 so that I wrote my
own version of the game in Ruby and Tk that allowed me to
experiment with input method. That was actually need since
game can be pretty fast once you learn it, especially if you
have a set of tiles which are easy to recognize.

Now that I have bought Nokia 770 - wanted to have this game
here too. But since support for Ruby is bad and Tk is not
available I decided to rewrite it in Python and GTK. This
the result.
"""
        text4 = "Version "+VERSION+" from "+VERSION_DATE
        notebook = gtk.Notebook()
        notebook.show()
        def add_scrolled_text_view(text, labeltext):
            scroll = gtk.ScrolledWindow()
            scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
            tv = gtk.TextView()
            tv.set_editable(False)
            tb = tv.get_buffer()
            scroll.add(tv)
            scroll.show()
            tv.show()
            tb.set_text(text)
            label = gtk.Label(labeltext)
            label.show()
            notebook.append_page(scroll, label)
        add_scrolled_text_view(text1, 'License')
        add_scrolled_text_view(text2, 'Instructions')
        add_scrolled_text_view(text3, 'History')
        add_scrolled_text_view(text4, 'Version')
        dialog.vbox.pack_start(notebook, True, True, 0)
        dialog.run()
        dialog.destroy()
    #def action_quit(self, widget, event):
    #    gtk.main_quit()
    def decodexy(self, ax, ay):
        return self.renderer.decodexy(self.area, self.model, ax, ay)
    def add_score(self):
        dialog = gtk.Dialog(title='Game done', parent=self.window, flags=gtk.DIALOG_MODAL,
                            buttons=(gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        label = gtk.Label("Enter your name for score table:")
        dialog.vbox.pack_start(label, True, True, 0)
        label.show()
        entry = gtk.Entry()
        entry.set_text(self.player_name)
        dialog.vbox.pack_start(entry, True, True, 0)
        entry.show()
        dialog.run()
        self.player_name=entry.get_text()
        scoreline=[int(self.model.get_time()*100),
                   DIF_NAMES_SCORE[self.difficulty], TILES_NAMES_SCORE[self.tilesetnr],
                   self.player_name]
        self.add_score_entry(scoreline)
        dialog.destroy()
        self.show_score(scoreline)
    def add_score_entry(self, scoreline):
        self.load_score()
        self.scoretab.append(scoreline)
        scores=group_by(self.scoretab, lambda x: x[1]+'/'+x[2])
        self.scoretab=[]
        for scoreclass in scores:
            scores[scoreclass].sort(key=lambda x: x[0])
            self.scoretab+=scores[scoreclass][0:SCORE_LIMIT]
        self.save_score()
    # score file entries are in format: time*100 SPC difficulty_name SPC tileset_name SPC player name
    def load_score(self):
        if self.scoretab==None:
            self.scoretab=[]
            if not os.path.exists(SCORE_FILE): return
            try:
                f=file(SCORE_FILE, 'rU')
                for line in f:
                    ent=line.split(' ', 3)
                    ent[0]=int(ent[0])
                    ent[-1]=ent[-1].strip()
                    self.scoretab.append(ent)
                f.close()
            except IOError, e:
                print e
    def save_score(self):
        if self.scoretab!=None:
            f=file(SCORE_FILE, 'w')
            for entry in self.scoretab:
                f.write("%d %s %s %s\n"%tuple(entry))
            f.close()
    def show_score(self, select_entry=None):
        self.load_score()
        dialog = gtk.Dialog(title='Score Table', parent=self.window, flags=gtk.DIALOG_MODAL,
                            buttons=(gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dialog.set_size_request(720,350)
        notebook = gtk.Notebook()
        notebook.show()
        notebook_ssv_sel=[None]
        def name_dif(symbol):
            try:
                difidx=DIF_NAMES_SCORE.index(symbol)
                return DIF_NAMES[difidx]
            except ValueError:
                return symbol
        def name_ts(symbol):
            try:
                tsidx=TILES_NAMES_SCORE.index(symbol)
                return TILES_NAMES[tsidx]
            except ValueError:
                return symbol
        def add_scrolled_score_view(labeltext, source, tabid):
            scroll = gtk.ScrolledWindow()
            scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
            liststore = gtk.ListStore(str, str, str, 'gboolean')
            treeview = gtk.TreeView(liststore)
            tvcolumn1 = gtk.TreeViewColumn('Rank')
            tvcolumn2 = gtk.TreeViewColumn('Time')
            tvcolumn3 = gtk.TreeViewColumn('Name')
            rank=0; lasttime=None
            ssv_sel=False
            for scoreline in source:
                if lasttime!=scoreline[0]:
                    rank+=1
                lasttime=scoreline[0]
                issel=(scoreline==select_entry)
                liststore.append([str(rank),'%d:%05.2f'%divmod((float(scoreline[0])/100),60),
                                  scoreline[3], issel])
                if issel:
                    notebook_ssv_sel[0]=tabid
            treeview.append_column(tvcolumn1)
            treeview.append_column(tvcolumn2)
            treeview.append_column(tvcolumn3)
            cell = gtk.CellRendererText()
            cell.set_property('cell-background', 'yellow')
            tvcolumn1.pack_start(cell, False)
            tvcolumn2.pack_start(cell, False)
            tvcolumn3.pack_start(cell, True)
            tvcolumn1.set_attributes(cell, text=0, cell_background_set=3)
            tvcolumn2.set_attributes(cell, text=1, cell_background_set=3)
            tvcolumn3.set_attributes(cell, text=2, cell_background_set=3)
            scroll.add(treeview)
            scroll.show()
            treeview.show()
            label = gtk.Label(labeltext)
            label.show()
            notebook.append_page(scroll, label)
        scores=group_by(self.scoretab, lambda x: x[1]+'/'+x[2])
        tabid=0
        for scoreclass in scores:
            firstentry=scores[scoreclass][0]
            dif=name_dif(firstentry[1])
            ts=name_ts(firstentry[2])
            add_scrolled_score_view(dif+' '+ts, scores[scoreclass], tabid)
            tabid+=1
        if notebook_ssv_sel[0]:
            notebook.set_current_page(notebook_ssv_sel[0])
        dialog.vbox.pack_start(notebook, True, True, 0)
        dialog.run()
        dialog.destroy()
    def area_press_event(self, widget, event):
        if self.model.is_pause(): return
        pos=self.decodexy(int(event.x), int(event.y))
        if pos:
            if not self.model.empty_at(*pos):
                if self.markedcell==None:
                    self.markedcell=pos
                else:
                    if self.markedcell==pos:
                        self.markedcell=None
                    else:
                        self.cancel_clear_path()
                        self.path=self.model.move(*(self.markedcell+pos))
                        if self.path:
                            self.activate_clear_path()
                            self.markedcell=None
                            if self.model.is_done():
                                dialog = gtk.MessageDialog(self.window,
                                                           gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                                           gtk.MESSAGE_INFO, gtk.BUTTONS_OK,
                                                           "Congratulations!\nYou have completed game in %d:%05.2f."%
                                                           divmod(float(int(self.model.get_time()*100))/100, 60))
                                dialog.run()
                                dialog.destroy()
                                self.add_score()
                        else:
                            self.markedcell=pos
        self.redraw()
    def area_expose_cb(self, area, event):
        self.renderer.draw(self.area, self.model, self.markedcell, self.path)
    def redraw(self):
        self.area.queue_draw()



def main():
    gtk.main()
    return 0

if __name__ == "__main__":
    ShishenShoGame()
    main()
