/**
    File: picasa_http.h
    
    Author: Tero Niemel

*/


#ifndef _PICASA_HTTP_H_
#define _PICASA_HTTP_H_

#include <curl/curl.h>


/**
 * http_type_t:
 *
 * HTTP reqeust type.
 */
typedef enum {
  POST = 0x01,
  GET = 0x02,
  PUT = 0x03
} http_type_t;

/**
 * http_result_t:
 *
 * HTTP function result codes.
 */
typedef enum {
  HTTP_SUCCESS,
  HTTP_ERROR
} http_result_t;

/**
 * http_request_body_t:
 * @data: Binary data
 * @sent: How much sent already
 *
 * Binary data for read callback.
 */
typedef struct {
  GString *data;
  size_t sent;
} picasa_http_body_t;

/**
 * http_request_state_t:
 *
 * HTTP request state.
 */
typedef enum {
  SENDING,
  DISCONNECTED,
  CANCELLED,
  ERROR
} http_request_state_t;

/**
 * http_request_t:
 * @state: Request state
 * @url: Url where to post
 * @hdr: Additional HTTP headers
 * @post: CURL post data
 * @last: CURL post data
 * @buf_size_max: Response buffer max size
 * @resp_buf: Pointer to response buffer
 * @progress_cb: Pointer to progress callback function
 * @progress_data: Pointer to progress data
 * @con: Conic connection
 * @read_data: Data for read callback
 * @body_size: Size of request body
 *
 * HTTP request data structure.
*/
typedef struct {
  http_request_state_t state;
  gchar *url;
  struct curl_slist *hdr;
  struct curl_httppost *post;
  struct curl_httppost *last;
  size_t  buf_size_max;
  GString *resp_buf;  
  curl_progress_callback progress_cb;
  gpointer progress_data; 
  ConIcConnection *con;
  picasa_http_body_t read_data;
  size_t body_size;
} http_request_t;


http_request_t 
*picasa_http_new (gchar *url);

http_result_t
picasa_http_append_read_data (http_request_t *request,
			      gchar *content);

http_result_t
picasa_http_add_header_data (http_request_t *request, 
			     gchar *name, gchar *value);

http_result_t
picasa_http_add_post_data (http_request_t *request, 
			   gchar *key, gchar *value);

http_result_t
picasa_http_add_post_data_with_type (http_request_t *request, 
				     gchar *value, gchar *type);

http_result_t
picasa_http_add_bin_file (http_request_t *request, 
			  const gchar *file, gchar *filename,
			  gchar *type);
http_result_t
picasa_http_send (http_request_t *request,
		  http_type_t type);

http_result_t
picasa_http_set_response_buffer (http_request_t *request, 
				 gint max_buf_size, GString *buf);

http_result_t
picasa_http_set_progress_cb (http_request_t *request, 
			     curl_progress_callback callback, gpointer data);

http_result_t
picasa_http_set_connection (http_request_t *request, 
			    ConIcConnection *con);

size_t 
picasa_http_print_http_post (http_request_t *request);

void 
picasa_http_free (http_request_t *request);


#endif /* _PICASA_HTTP_H_ */
