/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 * Copyright (C) 2009 Heikki Kallasjoki. All rights reserved.
 * Copyright (C) 2008-2009 Nokia Corporation. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sharing-plugin-interface.h>
#include <sharing-transfer.h>
#include <sharing-tag.h>
#include <conicconnection.h>
#include <osso-log.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <tablet-browser-interface.h> 

#include "blipfoto.h"
#include "blip_connection.h"
#include "blip_entry.h"
#include "blip_defines.h"

/* API Keys etc */

static const char* BlipAppKey= "2bf38220d8cb45f867c3fed390875bf1";
static const char* BlipAppSecret= "0b56a9ec119cfcee6760b671fde91cc2";
static const char* BlipAppPermissionsURL= "http://www.blipfoto.com/getpermission/711172";

/* Plugin interface definition: */

guint sharing_plugin_interface_init (gboolean* dead_mans_switch);
guint sharing_plugin_interface_uninit (gboolean* dead_mans_switch);
SharingPluginInterfaceSendResult sharing_plugin_interface_send (
		SharingTransfer* transfer, ConIcConnection* con,
		gboolean* dead_mans_switch);
SharingPluginInterfaceAccountSetupResult
        sharing_plugin_interface_account_setup (
                GtkWindow* parent,
                SharingService* service, SharingAccount** worked_on,
                osso_context_t* osso);
SharingPluginInterfaceEditAccountResult
        sharing_plugin_interface_edit_account (
                GtkWindow* parent,
                SharingAccount* account,
                ConIcConnection* con,
                gboolean* dead_mans_switch);
SharingPluginInterfaceAccountValidateResult
        sharing_plugin_interface_account_validate (
                SharingAccount* account, ConIcConnection* con,
                gboolean *cont, gboolean* dead_mans_switch);

/* Utilities */

BlipFoto CreateAuthorisedBlipFotoContext(SharingAccount* account,
                                         BlipError * oError);

static gboolean simple_label_dialog_with_ok (GtkWindow* parent,
                                             const char* title,
                                             const char* labelText);
static gboolean debug_dlg(GtkWindow* parent,
                          const char* message);

static gboolean display_notification(const gchar * type,
                                     const gchar * message,
                                     const gchar * extraArgs);
static gboolean display_banner(const gchar * message);
static gboolean display_note(const gchar * message);

/* Setup flow */

static gboolean account_setup_explain (GtkWindow* parent);
static gchar*   account_setup_acquire_token (GtkWindow* parent);
static void     account_setup_launch_blip_permissions (osso_context_t* osso);


/* UI flow helper definition: */

static SharingPluginInterfaceEditAccountResult edit_account (SharingAccount* account, GtkWindow* parent);

/**
 * sharing_plugin_interface_init:
 * @dead_mans_switch: while in this function, this switch should be set to FALSE at least every 30 seconds
 *
 * Initialize interface
 *
 * Returns: 0
 */
guint sharing_plugin_interface_init (gboolean* dead_mans_switch)
{
    ULOG_DEBUG_L("sharing_plugin_interface_init, blipfoto");
    *dead_mans_switch = FALSE;
    return 0;
}

/**
 * sharing_plugin_interface_uninit:
 * @dead_mans_switch: while in this function, this switch should be set to FALSE at least every 30 seconds
 *
 * Uninitialize interface
 *
 * Returns: 0
 */
guint sharing_plugin_interface_uninit (gboolean* dead_mans_switch)
{
    ULOG_DEBUG_L("sharing_plugin_interface_uninit, blipfoto");
    *dead_mans_switch = FALSE;
    return 0;
}

int qsort_cstring_compare(const void* a, const void* b)
{
    const char **ia = (const char **)a;
    const char **ib = (const char **)b;
    return strcmp(*ia, *ib);
}

gchar* create_tags_string(const GSList* tags)
{
    gchar* result= NULL;
    gsize tagCount= 0;
    gchar* filteredTags[64];
    SharingTag * sharingTag;
    SharingTagType tagType;
    const gchar* tagWord;
    for (const GSList* p= tags;
         p != NULL && tagCount<63;
         p= g_slist_next(p))
    {
        sharingTag= (SharingTag*) p->data;
        tagType= sharing_tag_get_type(sharingTag);
        if (tagType == SHARING_TAG_SHARE
            || tagType & (SHARING_TAG_GEO_CITY|SHARING_TAG_GEO_COUNTRY|SHARING_TAG_GEO_SUBURB))
        {
            tagWord= sharing_tag_get_word(sharingTag);
            if (tagWord[0] != '\0' && tagWord[0] != '@')
            {
                filteredTags[tagCount]= (gchar*) tagWord;
                ++tagCount;
            }
        }
    }
    if (tagCount > 0)
    {
        qsort(filteredTags, tagCount, sizeof(gchar*), qsort_cstring_compare);
        filteredTags[tagCount]= NULL;
        result= g_strjoinv(",", filteredTags);
    }
    return result;
}

static gboolean description_regex_cb (const GMatchInfo *info,
                                      GString          *res,
                                      gpointer          data)
{
    gchar *match;
    gchar *r;

    match = g_match_info_fetch (info, 0);
    r = g_hash_table_lookup ((GHashTable *)data, match);
    g_string_append (res, r);
    g_free (match);

    return FALSE;
}

static gchar* unescape_basic_html_entities(const gchar* description)
{
    gchar *result;
    GRegex *regex= g_regex_new("&(amp|lt|gt|quot|apos);", 0, 0, NULL);
    GHashTable *hashTable= g_hash_table_new (g_str_hash, g_str_equal);

    g_hash_table_insert (hashTable, "&amp;",  "&");
    g_hash_table_insert (hashTable, "&lt;",   "<");
    g_hash_table_insert (hashTable, "&gt;",   ">");
    g_hash_table_insert (hashTable, "&quot;", "\"");
    g_hash_table_insert (hashTable, "&apos;", "'");

    result= g_regex_replace_eval (regex, description, -1, 0, 0, description_regex_cb, hashTable, NULL);
    g_hash_table_destroy (hashTable);
    g_regex_unref(regex);

    return result;
}

struct Upload_s
{
    SharingTransfer* transfer;
    SharingEntryMedia* media;
    gboolean* dead_mans_switch;
    gdouble progress_start;
    gdouble progress_end;
};
typedef struct Upload_s Upload;

BlipConnectionStatus upload_progress(void* userdata,
                                     gdouble progress)
{
    Upload* upload= (Upload*) userdata;
    gdouble total= upload->progress_end - upload->progress_start;

    sharing_transfer_set_progress (upload->transfer, upload->progress_start+(progress*total));
    *upload->dead_mans_switch = FALSE;

    if (sharing_transfer_continue(upload->transfer))
        return BLIP_CONNECTION_CONTINUE;
    else
        return BLIP_CONNECTION_CANCEL;
}

gboolean send_media_as_blip(BlipFoto blip,
                            Upload* upload,
                            gchar** oEntryId,
                            gchar** oMessage,
                            BlipError * oError)
{
    gboolean result= FALSE;

    sharing_transfer_set_progress (upload->transfer, upload->progress_start);
    *upload->dead_mans_switch = FALSE;

    BlipEntry entry= blip_entry_create();

    gchar* media_title = sharing_entry_media_get_title (upload->media);

    if (media_title && strlen(media_title) > 0)
    {
        gchar* unescaped= unescape_basic_html_entities(media_title);
        g_free(media_title);
        blip_entry_set_title(entry, unescaped);
        g_free(unescaped);
    }

    const gchar* description= sharing_entry_media_get_desc(upload->media);
    if (description && strlen(description) > 0)
    {
        gchar* unescaped= unescape_basic_html_entities(description);
        blip_entry_set_description(entry, unescaped);
        g_free(unescaped);
    }

    const gchar* fullpath= sharing_entry_media_get_localpath(upload->media);
    if (fullpath)
    {
        blip_entry_set_image_full_path(entry, fullpath);
    }

    gchar* tags= create_tags_string(sharing_entry_media_get_tags(upload->media));
    if (tags)
    {
        gchar* unescaped= unescape_basic_html_entities(tags);
        g_free(tags);
        blip_entry_set_tags(entry, unescaped);
        g_free(unescaped);
    }

    BlipConnection conn= blip_connection_create();
    blip_connection_set_upload_callback(conn, upload_progress, upload);
    if (blip_post_entry(blip, entry, conn, oEntryId, oMessage, oError))
    {
        result= TRUE;
    }
    blip_connection_free(conn);

    blip_entry_free(entry);

    return result;
}

/**
 * sharing_plugin_interface_send:
 * @transfer: Transfer to be send
 * @con: Connection used
 * @dead_mans_switch: while in this function, this switch should be set to FALSE as often as possible. Do NOT use any GTK idle / timeout function to do this!
 *
 * Send interface.
 *
 * Returns: Result of send
 */
SharingPluginInterfaceSendResult sharing_plugin_interface_send (
        SharingTransfer* transfer,
        ConIcConnection* con,
        gboolean*        dead_mans_switch)
{
    ULOG_DEBUG_L ("sharing_plugin_interface_send, blipfoto");

    SharingPluginInterfaceSendResult result= SHARING_SEND_ERROR_UNKNOWN;

    SharingEntry *entry = sharing_transfer_get_entry (transfer);
    SharingAccount *account = sharing_entry_get_account (entry);
    BlipError error= NULL;
    BlipFoto blip= CreateAuthorisedBlipFotoContext(account, &error);
    gboolean validated;

    sharing_transfer_set_progress (transfer, 0.0);
    *dead_mans_switch = FALSE;

    if (!blip)
    {
        result= SHARING_SEND_ERROR_AUTH;
    }
    else if (blip_validate_authentication(blip, &validated, &error))
    {
        if (!validated)
        {
            result= SHARING_SEND_ERROR_AUTH;
        }
        else
        {
            result= SHARING_SEND_CANCELLED;

            if (!sharing_transfer_continue (transfer))
            {
                result= SHARING_SEND_CANCELLED;
            }
            else
            {
                guint64 total_bytes = 0;

                *dead_mans_switch = FALSE;

                for (GSList* p = sharing_entry_get_media (entry);
                p != NULL;
                p= g_slist_next (p))
                {
                    SharingEntryMedia* media = p->data;
                    if (!sharing_entry_media_get_sent (media))
                        total_bytes += sharing_entry_media_get_size (media);
                }

                if (total_bytes > 0)
                {
                    guint64 sent_bytes= 0;
                    result= SHARING_SEND_SUCCESS;

                    for (GSList* p = sharing_entry_get_media (entry);
                         p != NULL && result == SHARING_SEND_SUCCESS;
                         p= g_slist_next (p))
                    {
                        SharingEntryMedia* media = p->data;
                        guint64 media_size= sharing_entry_media_get_size (media);
                        gdouble progress_start=
                                1.0 - ((total_bytes - sent_bytes)
                                       / (gdouble)total_bytes);
                        gdouble progress_end=
                                1.0 - ((total_bytes - (sent_bytes + media_size))
                                       / (gdouble)total_bytes);

                        if (sharing_entry_media_get_sent (media))
                        {
                            sharing_transfer_set_progress (transfer, progress_end);
                        }
                        else if (!sharing_transfer_continue (transfer))
                        {
                            result= SHARING_SEND_CANCELLED;
                        }
                        else
                        {
                            gchar* entryId= NULL;
                            gchar* message= NULL;
                            BlipError error= NULL;
                            Upload upload;
                            upload.transfer= transfer;
                            upload.media= media;
                            upload.progress_start= progress_start;
                            upload.progress_end= progress_end;
                            upload.dead_mans_switch= dead_mans_switch;
                            if (send_media_as_blip(blip,
                                                   &upload,
                                                   &entryId,
                                                   &message,
                                                   &error))
                            {
                                gchar display[256];
                                g_snprintf(display, 256, "Blipfoto entry id %s\n%s", entryId, message);
                                g_free(entryId);
                                g_free(message);
                                display_note(display);
                            }
                            else if (blip_error_match(error,
                                                      BLIP_DOMAIN_BLIP_LIB,
                                                      BLIP_LIB_CONNECTION_CANCELLED_BY_USER,
                                                      BLIP_ERROR_MATCH_END))
                            {
                                result= SHARING_SEND_CANCELLED;
                            }
                            else
                            {
                                if (blip_error_message(error))
                                {
                                    gchar display[256];
                                    g_snprintf(display,
                                               256,
                                               "%s error (%d): %s",
                                               blip_error_domain_name(error),
                                               blip_error_code(error),
                                               blip_error_message(error));
                                    display_note(display);
                                }
                                /// @todo Map some of the blip api error codes to SHARING_SEND errors.
                                result= SHARING_SEND_ERROR_UNKNOWN;
                            }
                            blip_error_free(error);
                        }
                    }
                }
                sharing_transfer_set_progress (transfer, 1.0);
                *dead_mans_switch = FALSE;
            }
        }

        blip_free_context(blip);
    }

    if (error)
    {
        if (blip_error_domain(error) == BLIP_DOMAIN_TRANSPORT)
            result= SHARING_SEND_ERROR_CONNECTION;
        blip_error_free(error);
    }

    return result;
}

/**
 * sharing_plugin_interface_account_setup:
 * @parent: Parent window
 * @service: Service information
 * @worked_on: Who knows?
 * @osso: osso_context_t for the dialog.
 *
 * Custom account setup UI flow.
 *
 * Returns: Something?
 */
SharingPluginInterfaceAccountSetupResult
        sharing_plugin_interface_account_setup (
                GtkWindow* parent,
                SharingService* service, SharingAccount** worked_on,
                osso_context_t* osso)
{
	if (!worked_on || !*worked_on)
		return SHARING_ACCOUNT_SETUP_ERROR_UNKNOWN;

	SharingPluginInterfaceAccountSetupResult result;
	result= SHARING_ACCOUNT_SETUP_ERROR_UNKNOWN;

    BlipError error= NULL;
    BlipFoto blip= blip_create_context(BlipAppKey, BlipAppSecret, &error);

    if (blip && account_setup_explain(parent))
	{
		gchar* tempToken;
		gchar* authToken;
		gchar* username;

		account_setup_launch_blip_permissions(osso);

		tempToken= account_setup_acquire_token(parent);

		if (tempToken && strlen(tempToken)>0)
		{
			if (blip_get_token(blip,
                               tempToken,
                               &username,
                               &authToken,
                               &error))
			{
				sharing_account_set_param (*worked_on,
                                           "username",
                                           username);
				sharing_account_set_param (*worked_on,
                                           "authToken",
                                           authToken);
				g_free(authToken);
				g_free(username);

				result= SHARING_ACCOUNT_SETUP_SUCCESS;
			}
			g_free(tempToken);
		}
		blip_free_context(blip);
	}

    if (error)
    {
        if (blip_error_domain(error) == BLIP_DOMAIN_TRANSPORT)
            result= SHARING_ACCOUNT_SETUP_ERROR_CONNECTION;
        blip_error_free(error);
    }

    return result;
}

/**
 * sharing_plugin_interface_edit_account:
 * @parent: Parent window
 * @account: Account to edit
 * @con: Connection to use (for what?)
 * @dead_mans_switch: while in this function, this switch should be set to FALSE at least every 30 seconds
 */
SharingPluginInterfaceEditAccountResult
        sharing_plugin_interface_edit_account (
                GtkWindow* parent,
                SharingAccount* account,
                ConIcConnection* con,
                gboolean* dead_mans_switch)
{
	if (!account)
		return SHARING_EDIT_ACCOUNT_NOT_STARTED;

	return edit_account (account, parent);
}

BlipFoto CreateAuthorisedBlipFotoContext(SharingAccount* account,
                                         BlipError * oError)
{
    BlipFoto blip= blip_create_context(BlipAppKey, BlipAppSecret, oError);
    if (blip)
    {
        gchar* authToken= sharing_account_get_param (account, "authToken");
        if (authToken)
        {
            blip_set_auth_token(blip, authToken);
            g_free(authToken);
        }
        else
        {
            blip_free_context(blip);
            blip= NULL;
        }
    }
    return blip;
}

/**
 * sharing_plugin_interface_account_validate:
 * @account: Account tested
 * @con: Connection used to test account
 * @cont: ...
 * @dead_mans_switch: while in this function, this switch should be set to FALSE at least every 30 seconds
 *
 * Validates account information.
 *
 * Returns: Result of account validation
 */
SharingPluginInterfaceAccountValidateResult
        sharing_plugin_interface_account_validate (
                SharingAccount* account,
                ConIcConnection* con,
                gboolean *cont,
                gboolean* dead_mans_switch)
{
    ULOG_DEBUG_L ("sharing_plugin_interface_account_validate, blipfoto");

	SharingPluginInterfaceAccountValidateResult
            result= SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;

    *dead_mans_switch = FALSE;

    BlipError error= NULL;
    BlipFoto blip= CreateAuthorisedBlipFotoContext(account, &error);
	if (blip)
	{
	    gboolean validated;
        if (blip_validate_authentication(blip, &validated, &error))
	    {
            if (validated)
                result= SHARING_ACCOUNT_VALIDATE_SUCCESS;
            else
                result= SHARING_ACCOUNT_VALIDATE_FAILED;
	    }
	    blip_free_context(blip);
	}

    if (error)
    {
        if (blip_error_domain(error) == BLIP_DOMAIN_TRANSPORT)
            result= SHARING_ACCOUNT_VALIDATE_ERROR_CONNECTION;
        blip_error_free(error);
    }

	*dead_mans_switch = FALSE;

    return result;
}

/* UI flow helper implementations. */

static void gui_add_item (GtkWidget* table, guint row,
                          gchar* id, const gchar* label, const gchar* placeholder,
                          gboolean invis, SharingAccount* a, GHashTable* h);
static gboolean gui_read_item (GHashTable* h, const gchar* id, SharingAccount* a);
static gboolean gui_read (GHashTable* h, SharingAccount* a);

static void gui_add_item (GtkWidget* table, guint row,
                          gchar* id, const gchar* label, const gchar* placeholder,
                          gboolean invis, SharingAccount* a, GHashTable* h)
{
	GtkWidget* wlabel = gtk_label_new (label);
	gtk_table_attach (GTK_TABLE (table), wlabel, 0, 1, row, row+1, GTK_FILL, GTK_FILL|GTK_EXPAND, HILDON_MARGIN_DOUBLE, 0);

	GtkWidget* wentry = hildon_entry_new (HILDON_SIZE_AUTO);
	hildon_entry_set_placeholder (HILDON_ENTRY (wentry), placeholder);
	if (invis)
		hildon_gtk_entry_set_input_mode (GTK_ENTRY (wentry), HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_INVISIBLE);
	gtk_table_attach_defaults (GTK_TABLE (table), wentry, 1, 2, row, row+1);

	g_hash_table_insert (h, id, wentry);

	gchar* old = sharing_account_get_param (a, id);
	if (old)
	{
		gtk_entry_set_text (GTK_ENTRY (wentry), old);
		g_free (old);
	}
}

static gboolean gui_read_item (GHashTable* h, const gchar* id, SharingAccount* a)
{
	GtkWidget* wentry = g_hash_table_lookup (h, id);
	if (!wentry) return FALSE;

	gchar* old = sharing_account_get_param (a, id);
	const gchar* new = gtk_entry_get_text (GTK_ENTRY (wentry));

	gboolean changed = FALSE;

	if (!old || strcmp (old, new) != 0)
	{
		sharing_account_set_param (a, id, new);
		changed = TRUE;
	}

	g_free (old);
	return changed;
}

static gboolean gui_read (GHashTable* h, SharingAccount* a)
{
	gboolean changed = FALSE;
	if (gui_read_item (h, "username", a)) changed = TRUE;
	if (gui_read_item (h, "password", a)) changed = TRUE;
	if (gui_read_item (h, "urlbase", a)) changed = TRUE;
	if (gui_read_item (h, "albumname", a)) changed = TRUE;
	return changed;
}

static SharingPluginInterfaceEditAccountResult
        edit_account (SharingAccount* account, GtkWindow* parent)
{
	GtkWidget* dlg = 0;

	dlg = gtk_dialog_new_with_buttons (
            "Edit account - Blipfoto", parent,
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_OK, GTK_RESPONSE_YES,
            GTK_STOCK_DELETE, GTK_RESPONSE_NO,
            NULL);

	GtkWidget* dlg_content = gtk_dialog_get_content_area (GTK_DIALOG (dlg));

	GtkWidget* label = gtk_label_new("Nothing to edit yet, but hit \"delete\" to remove the account.");

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_container_add (GTK_CONTAINER (dlg_content), label);
	gtk_widget_show_all (GTK_WIDGET (dlg));

	gint result = gtk_dialog_run (GTK_DIALOG (dlg));

	gtk_widget_destroy (dlg);

	if (result == GTK_RESPONSE_NO)
		return SHARING_EDIT_ACCOUNT_DELETE;
	
	return SHARING_EDIT_ACCOUNT_CANCELLED;

#ifdef NOTDEF
	GHashTable* h = g_hash_table_new (g_str_hash, g_str_equal);

	GtkWidget* dlg = 0;

	dlg = gtk_dialog_new_with_buttons (
            "Edit account - Blipfoto", parent,
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_SAVE, GTK_RESPONSE_YES,
            GTK_STOCK_DELETE, GTK_RESPONSE_NO,
            NULL);

	GtkWidget* dlg_content = gtk_dialog_get_content_area (GTK_DIALOG (dlg));

	GtkWidget* table = gtk_table_new (4, 2, FALSE);
	gtk_container_add (GTK_CONTAINER (dlg_content), table);

	gui_add_item (table, 0, "username", "User name", "Blipfoto username", FALSE, account, h);
	gui_add_item (table, 1, "token", "Token", "Blipfoto account password", TRUE, account, h);
	gui_add_item (table, 2, "urlbase", "URL", "Blipfoto site address", FALSE, account, h);
	gui_add_item (table, 3, "albumname", "Album", "Album name to upload to", FALSE, account, h);

	gtk_widget_show_all (GTK_WIDGET (dlg));
	gint result = gtk_dialog_run (GTK_DIALOG (dlg));

	gboolean changed = FALSE;
	if (result == GTK_RESPONSE_YES)
		changed = gui_read (h, account);

	gtk_widget_destroy (dlg);
	g_hash_table_unref (h);

	if (result == GTK_RESPONSE_YES && (changed || setup))
		return SHARING_EDIT_ACCOUNT_SUCCESS;
	else if (result == GTK_RESPONSE_YES) /* !changed in edit */
		return SHARING_EDIT_ACCOUNT_NOT_STARTED;
	else if (result == GTK_RESPONSE_NO)
		return SHARING_EDIT_ACCOUNT_DELETE;
	else
		return SHARING_EDIT_ACCOUNT_CANCELLED;
#endif
}

static gboolean
        account_setup_explain (GtkWindow* parent)
{
	return
            simple_label_dialog_with_ok(
                    parent,
                    "Account setup - Blipfoto",
                    "In order to authenticate with Blipfoto.com, the "
                    "browser will now be launched. Please follow the "
                    "instructions there, and return here with the "
                    "6-digit access code.");
}

static gboolean
        simple_label_dialog_with_ok (GtkWindow* parent,
                                     const char* title,
                                     const char* labelText)
{
	GtkWidget* dlg = 0;

	dlg = gtk_dialog_new_with_buttons (
            title, parent,
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_OK, GTK_RESPONSE_YES,
            NULL);

	GtkWidget* dlg_content = gtk_dialog_get_content_area (GTK_DIALOG (dlg));

	GtkWidget* label = gtk_label_new(labelText);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_container_add (GTK_CONTAINER (dlg_content), label);
	gtk_widget_show_all (GTK_WIDGET (dlg));

	gint result = gtk_dialog_run (GTK_DIALOG (dlg));

	gtk_widget_destroy (dlg);

	if (result == GTK_RESPONSE_YES)
		return TRUE;

	return FALSE;
}

static gboolean
        debug_dlg(GtkWindow* parent,
                  const char* message)
{
	return simple_label_dialog_with_ok(parent,
                                       "debug",
                                       message ? message : "<null>");
}

static void
        account_setup_launch_blip_permissions (osso_context_t* osso)
{
	osso_rpc_run_with_defaults(osso, "osso_browser",
                               OSSO_BROWSER_OPEN_NEW_WINDOW_REQ, NULL,
                               DBUS_TYPE_STRING, BlipAppPermissionsURL,
                               DBUS_TYPE_INVALID);
}

static gchar*
        account_setup_acquire_token (GtkWindow* parent)
{
	GtkWidget* dlg = 0;

	dlg = gtk_dialog_new_with_buttons (
            "Account setup - Blipfoto", parent,
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_OK, GTK_RESPONSE_YES,
            NULL);

	GtkWidget* dlg_content = gtk_dialog_get_content_area (GTK_DIALOG (dlg));

	GtkWidget* table = gtk_table_new (1, 2, FALSE);
	gtk_container_add (GTK_CONTAINER (dlg_content), table);

	GtkWidget* label = gtk_label_new ("API Token");
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      GTK_FILL, GTK_FILL|GTK_EXPAND,
                      HILDON_MARGIN_DOUBLE,
                      0);

	GtkWidget* entry = hildon_entry_new (HILDON_SIZE_AUTO);
	hildon_entry_set_placeholder (HILDON_ENTRY (entry), "<6-digit code>");
	gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 0, 1);

	gtk_widget_show_all (GTK_WIDGET (dlg));
	gint result = gtk_dialog_run (GTK_DIALOG (dlg));

	gchar * token= NULL;

	if (result == GTK_RESPONSE_YES)
		token= g_strdup(gtk_entry_get_text (GTK_ENTRY(entry)));

	gtk_widget_destroy (dlg);

	return token;
}

static gboolean display_notification(const gchar * type,
                                     const gchar * message,
                                     const gchar * extraArgs)
{
    const gchar* displayString= message ? message : "<null>";
    const gchar * commandTemplate=
            "run-standalone.sh dbus-send "
            "--type=method_call "
            "--dest=org.freedesktop.Notifications "
            "/org/freedesktop/Notifications "
            "org.freedesktop.Notifications.%s "
            "string:'%s' %s";
    gchar * command= g_malloc(strlen(commandTemplate)
                              + strlen(type)
                              + strlen(displayString)
                              + strlen(extraArgs)
                              + 1);
    int status;
    g_sprintf(command, commandTemplate, type, displayString, extraArgs);
    status= system(command);
    g_free(command);
    return status >= 0;
}

static gboolean display_banner(const gchar * message)
{
    return display_notification("SystemNoteInfoprint", message, "");
}

static gboolean display_note(const gchar * message)
{
    return display_notification("SystemNoteDialog", message, "uint32:2 string:'Ok'");
}
