/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#ifndef __BLIPFOTO_H__
#define __BLIPFOTO_H__

#include <glib.h>

#include "blip_entry.h"
#include "blip_error.h"

G_BEGIN_DECLS

        struct BlipFoto_s;
typedef struct BlipFoto_s* BlipFoto;

/**
 * @param [in] apiKey
 *   The application key for your application.
 * @param [in] apiSecret
 *   The application secret for your application.
 * @param [out,optional] error
 *   If you pass in a non-null error pointer, any error will be
 *   recorded there. You must then free its resources with
 *   blip_error_free().
 * @return
 *   A newly created BlipFoto context on success. If anything went
 *   wrong, NULL is returned. You must use BlipFree() to release
 *   the resources created by this function.
 */
BlipFoto blip_create_context(const gchar* apiKey,
                             const gchar* apiSecret,
                             BlipError * oError);

/**
 * Frees up resources usd by the BlipFoto context.
 * @param blip
 *   A valid BlipFoto context, created by BlipCreate().
 */
void blip_free_context(BlipFoto blip);

/**
 * Sets the api authorisation token once it has been obtained.
 * @param [in] blip
 *   A valid BlipFoto context, created by BlipCreate().
 * @param [in] token
 *   The authorisation token received from the user's blipfoto
 *   permissions page. A copy will be taken.
 */
void blip_set_auth_token(BlipFoto blip, const gchar * token);

gboolean blip_post_image(BlipFoto blip,
                         const gchar* fullPathToImage,
                         BlipError* error);

/**
 * @brief Posts a new entry to blipfoto.com.
 * @param [in] blip
 *   A valid BlipFoto context, created by BlipCreate(), with a valid
 *   authorisation token set via blip_set_auth_token().
 * @param [in] entry
 *   A pre-prepared BlipEntry instance.
 * @param [in] connection
 *   A valid BlipConnection with relevant callbacks and options set.
 *   The entry's details wil automatically be applied to the connection,
 *   so don't call blip_entry_set_connection_options() yourself.
 * @param [out,optional] entryId
 *   If NULL is passed in, this argument is ignored. Otherwise the
 *   new unique identifier for a successfully posted entry is returned
 *   here. You must free it with g_free().
 * @param [out,optional] message
 *   If NULL is passed in, this argument is ignored. Otherwise the
 *   message in the response from blipfoto.com is returned here.
 *   You must free with g_free(). According to the api documentation,
 *   this is "a message describing the status of the entry. This must
 *   always be displayed to the user, as it may contain a moderation
 *   or other message."
 * @param [out,optional] error
 *   If you pass in a non-null error pointer, any error will be
 *   recorded there. You must then free its resources with
 *   blip_error_free().
 * @return
 *   If the server was contacted and no error was returned in the
 *   response from the server, then this function returns TRUE.
 *   If anything went wrong, FALSE is returned instead.
 */
gboolean blip_post_entry(BlipFoto   blip,
                         BlipEntry  entry,
                         BlipConnection connection,
                         gchar**    entryId,
                         gchar**    message,
                         BlipError* error);

/**
 * @brief Tests if entry posting is working correctly
 * @param [in] blip
 *   A valid BlipFoto context, created by BlipCreate(), with a valid
 *   authorisation token set via blip_set_auth_token().
 * @param [in] entry
 *   A pre-prepared BlipEntry instance.
 * @param [out,optional] message
 *   If NULL is passed in, this argument is ignored. Otherwise the
 *   message in the response from blipfoto.com is returned here.
 *   You must free with g_free(). According to the api documentation,
 *   this is "a message describing the status of the entry. This must
 *   always be displayed to the user, as it may contain a moderation
 *   or other message."
 * @param [out,optional] error
 *   If you pass in a non-null error pointer, any error will be
 *   recorded there. You must then free its resources with
 *   blip_error_free().
 * @return
 *   If the server was contacted, no error was returned in the
 *   response from the server and the expected entry_id of -1 was
 *   returned (signifying success, according to the api docs), then
 *   this function returns TRUE. If anything went wrong, FALSE is
 *   returned instead.
 */
gboolean blip_test_post_entry(BlipFoto   blip,
                              BlipEntry  entry,
                              gchar**    message,
                              BlipError* error);

/**
 * @param [in] blip
 *   A valid BlipFoto context.
 * @return
 *   The best server's time, suitable for constructing authenticated
 *   api calls.
 */
time_t blip_get_time(const BlipFoto blip);

/**
 * @param [in] blip
 *   A valid BlipFoto context.
 * @param [in] temporaryToken
 *   A six-character string provided by the user, after granting
 *   permissions to your application. See blipfoto API documentation,
 *   step 3 for details.
 * @param [out] username
 *   The username of the user
 * @param [out] idToken
 *   The user's Identity Token
 * @return
 *   If TRUE is returned, username and idToken have been filled with
 *   the results of the blip api call. They should subsequently be
 *   freed with g_free(). If FALSE is returned, the api call failed.
 */
gboolean blip_get_token(const BlipFoto blip,
                        const char* temporaryToken,
                        char** username,
                        char** idToken,
                        BlipError * oError);

/**
 * @param [in] blip
 *   A valid BlipFoto context.
 * @param [out] validated
 *   On successful call to the api, this will be set to whether or
 *   not the blip authentication is validating properly.
 * @param [out,optional] error
 *   If you pass in a non-null pointer to a BlipError, any error
 *   found along the way will be communicated here if FALSE is returned.
 *   You must then free its resources with blip_error_free().
 * @return
 *   If TRUE is returned, 'validated' field has been set. If FALSE is
 *   returned, some error occurred, so consult the error parameter.
 */
gboolean blip_validate_authentication(const BlipFoto blip,
                                      gboolean * validated,
                                      BlipError * error);

G_END_DECLS

#endif // ! __BLIPFOTO_H__
