/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#include <glib/gprintf.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <time.h>

#include <openssl/evp.h>

#include "blipfoto.h"
#include "blip_connection.h"
#include "blip_defines.h"
#include "blip_error.h"
#include "blip_p.h"
#include "blip_xml.h"

/*----------------------------------------------------------------------
 * Forward declarations
 *--------------------------------------------------------------------*/

static gchar*   CallGetTime(const gchar* apiKey, BlipError * oError);
static gchar*   CallGetTimeReal(const gchar* apiKey, BlipError * oError);
static gchar*   CallGetTimeDummy(const gchar* apiKey);
static gboolean DecodeGetTime(const gchar* callResponse, time_t* timestamp, BlipError * oError);
static gboolean GetTime(const char* apiKey, time_t* timestamp, BlipError * oError);

static gchar*   CallGetToken(const gchar* apiKey,
                             const gchar* temporaryToken,
                             BlipError * oError);
static gchar*   CallGetTokenReal(const gchar* apiKey,
                                 const gchar* temporaryToken,
                                 BlipError * oError);
static gchar*   CallGetTokenDummy(const gchar* apiKey,
                                  const gchar* temporaryToken);
static gboolean DecodeGetToken(const gchar* callResponse,
                               gchar** username,
                               gchar** idToken,
                               BlipError * oError);

gchar * blip_fetch_response(const gchar* url,
                            BlipError * oError);

/*----------------------------------------------------------------------
 * BlipFoto struct definition *
 *--------------------------------------------------------------------*/

struct BlipFoto_s
{
    gchar* apiKey;
    gchar* apiSecret;
    gchar* apiAuthToken;
    time_t timeDelta;
    EVP_MD_CTX mdctx;
};

/*----------------------------------------------------------------------
 * BlipFoto creation/deletion
 *--------------------------------------------------------------------*/

BlipFoto blip_create_context(const gchar* apiKey,
                             const gchar* apiSecret,
                             BlipError * oError)
{
    BlipFoto blip;

    if (apiKey == NULL)
    {
        if (oError)
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_LIB,
                                       BLIP_LIB_NULL_API_KEY,
                                       "Blipfoto API key was null");
        return NULL;
    }

    if (apiSecret == NULL)
    {
        if (oError)
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_LIB,
                                       BLIP_LIB_NULL_API_SECRET,
                                       "Blipfoto API secret was null");
        return NULL;
    }

    blip= (BlipFoto) g_try_malloc0(sizeof(struct BlipFoto_s));
    if (blip == NULL)
    {
        if (oError)
            // Is there any point in doing this? No doubt there'll be
            // no memory to create the BlipError either.
            *oError= blip_error_create(BLIP_DOMAIN_BLIP_LIB,
                                       BLIP_LIB_ERROR_OUT_OF_MEMORY,
                                       "No memory to create BlipFoto context");
        return NULL;
    }

    time_t serverTime;
    if (!GetTime(apiKey, &serverTime, oError))
    {
        // Any error already set in oError.
        g_free(blip);
        return NULL;
    }

    time_t dummy;
    blip->apiKey= g_strdup(apiKey);
    blip->apiSecret= g_strdup(apiSecret);
    blip->apiAuthToken= NULL;
    blip->timeDelta= time(&dummy) - serverTime;
    EVP_MD_CTX_init(&blip->mdctx);

    return blip;
}

void blip_free_context(BlipFoto blip)
{
    if (blip)
    {
        g_free(blip->apiKey);
        g_free(blip->apiSecret);
        g_free(blip->apiAuthToken);
        EVP_MD_CTX_cleanup(&blip->mdctx);
        g_free(blip);
    }
}

void blip_set_auth_token(BlipFoto blip, const gchar * token)
{
    if (blip)
    {
        g_free(blip->apiAuthToken);
        blip->apiAuthToken= g_strdup(token);
    }
}

/*----------------------------------------------------------------------
 * Utility functions
 *--------------------------------------------------------------------*/

gchar* blip_md5(BlipFoto blip, const void* data, size_t size)
{
    unsigned char md_value[EVP_MAX_MD_SIZE];
    unsigned int md_length;
    gchar* result= NULL;
    
    if (EVP_DigestInit_ex(&blip->mdctx, EVP_md5(), NULL)
        && EVP_DigestUpdate(&blip->mdctx, data, size)
        && EVP_DigestFinal_ex(&blip->mdctx, md_value, &md_length))
    {
        gchar* c= g_malloc((md_length*2)+1);
        result= c;
        for (int i= 0; i < md_length; ++i)
        {
            g_sprintf(c, "%02x", md_value[i]);
            c += 2;
        }
    }
    return result;
}

gchar* blip_build_api_string(const gchar* resource, ...)
{
	static const gsize ExtraCharsForSeparators= 2;
	static const gsize TotalSize= 1024;
	gchar *url= g_malloc(TotalSize);
	gchar *key;
	gchar *value;
	gchar *current;
	gboolean firstArgument= TRUE;
	va_list ap;

	g_sprintf(url, "%s/%s?", BLIP_API_BASE, resource);
	current= url+strlen(url);

	va_start(ap, resource);
	key= va_arg(ap, gchar*);
	while (key && url)
	{
		value= va_arg(ap, gchar*);
		if (value)
		{
			gsize used= current-url;
			gsize remaining= TotalSize-used;
			gsize required=
                    strlen(key)
                    + strlen(value)
                    + ExtraCharsForSeparators
                    + 1;

			if (required > remaining)
			{
				g_free(url);
				url= NULL;
			}
			else
			{
				current += g_sprintf(current,
                                     "%s%s=%s",
                                     firstArgument ? "" : "&",
                                     key,
                                     value);
				firstArgument= FALSE;
			}

			key= va_arg(ap, gchar*);
		}
		else
		{
			g_free(url);
			url= NULL;
		}
	}
	va_end(ap);

	return url;
}

gchar * blip_fetch_response(const gchar* url,
                            BlipError * oError)
{
	gchar * result= NULL;
	if (url)
	{
        BlipConnection conn= blip_connection_create();
        result= blip_connection_go(conn, url, oError);
        blip_connection_free(conn);
	}
	return result;
}

gchar* blip_generate_nonce(BlipFoto blip)
{
    if (blip)
    {
        time_t dummy;
        gchar nonce[128];
        g_snprintf(nonce, 128, "%d%ld", rand(), time(&dummy));
        return blip_md5(blip, nonce, strlen(nonce));
    }
    return NULL;
}

gchar* blip_generate_signature(BlipFoto blip)
{
    if (!blip || !blip->apiAuthToken)
        return NULL;

    gchar fullSignature[512];
    gchar timestamp[32];
    gchar* signMe;
    gchar* nonce;
    gchar* signature;

    g_snprintf(timestamp, 32, "%ld", blip_get_time(blip));

    nonce= blip_generate_nonce(blip);
    signMe= g_strjoin(NULL,
                      timestamp,
                      nonce,
                      blip->apiAuthToken,
                      blip->apiSecret,
                      NULL);
    signature= blip_md5(blip, signMe, strlen(signMe));
    g_free(signMe);

    g_snprintf(fullSignature, 512,
               "timestamp=%s&nonce=%s&token=%s&signature=%s",
               timestamp,
               nonce,
               blip->apiAuthToken,
               signature);

    g_free(nonce);
    g_free(signature);

    return g_strdup(fullSignature);
}

gchar* blip_sign_url(BlipFoto blip, const gchar* url)
{
    gchar* result= NULL;
    gchar* signature= blip_generate_signature(blip);
    if (signature)
    {
        result= g_strjoin("&",
                          url,
                          signature,
                          NULL);
        g_free(signature);
    }
    return result;
}

gchar* blip_build_signed_api_string(BlipFoto blip,
                             const gchar* resourceName)
{
    gchar* url= blip_build_api_string(resourceName,
                                      BLIP_ARG_API_KEY, blip->apiKey,
                                      BLIP_ARGS_DONE);
    gchar* signedurl= blip_sign_url(blip, url);
    g_free(url);
    return signedurl;
}

static gchar* post_entry(BlipFoto blip,
                         BlipEntry entry,
                         BlipConnection conn,
                         const gchar* resourceName,
                         BlipError * oError)
{
    gchar * result= NULL;
    if (blip && entry && conn)
    {
        blip_entry_set_connection_options(entry, conn);

        gchar* url= blip_build_signed_api_string(blip, resourceName);
        result= blip_connection_go(conn, url, oError);

        g_free(url);
    }
    return result;
}

static gboolean DecodePostEntryResponse(const gchar* response,
                                        gchar** entryId,
                                        gchar** message,
                                        BlipError* oError)
{
    gboolean success= FALSE;
    BlipXml blipXml= blip_xml_create(response, oError);
    if (blipXml)
    {
        gchar* result= blip_xml_get_text_node_contents(blipXml,
                                                       "/blipapi/data/result/text()",
                                                       NULL);

        if (entryId)
            *entryId= blip_xml_get_text_node_contents(blipXml,
                                                      "/blipapi/data/entry_id/text()",
                                                      NULL);

        if (result)
        {
            success= TRUE;
            if (message)
                *message= result;
            else
                g_free(result);
        }
        else if (oError)
        {
            *oError= blip_xml_extract_error(blipXml);
        }

        blip_xml_free(blipXml);
    }
    return success;
}

gboolean blip_post_entry(BlipFoto   blip,
                         BlipEntry  entry,
                         BlipConnection conn,
                         gchar**    entryId,
                         gchar**    message,
                         BlipError* oError)
{
    gboolean success= FALSE;
    gchar* response= post_entry(blip, entry, conn, BLIP_RESOURCE_POST_ENTRY, oError);
    if (response)
    {
        success= DecodePostEntryResponse(response, entryId, message, oError);
        g_free(response);
    }
    return success;
}

gboolean blip_test_post_entry(BlipFoto   blip,
                              BlipEntry  entry,
                              gchar**    message,
                              BlipError* oError)
{
    gboolean success= FALSE;
    BlipConnection conn= blip_connection_create();
    gchar* response= post_entry(blip, entry, conn, BLIP_RESOURCE_TESTPOST_ENTRY, oError);
    blip_connection_free(conn);
    if (response)
    {
        gchar* entryId;
        if (DecodePostEntryResponse(response, &entryId, message, oError))
        {
            if (atoi(entryId) == -1)
            {
                success= TRUE;
            }
            g_free(entryId);
        }
        g_free(response);
    }
    return success;
}

gboolean blip_post_image(BlipFoto blip,
                         const gchar* fullPathToImage,
                         BlipError* oError)
{
    gboolean success= FALSE;
    if (blip && fullPathToImage)
    {
        BlipEntry entry= blip_entry_create();
        if (entry)
        {
            blip_entry_set_image_full_path(entry, fullPathToImage);
            BlipConnection conn= blip_connection_create();
            gchar* response= post_entry(blip, entry, conn, BLIP_RESOURCE_POST_IMAGE, oError);
            blip_connection_free(conn);
            if (response)
            {
                gchar* message;
                if (DecodePostEntryResponse(response, NULL, &message, oError))
                {
                    success= TRUE;
                    // I don't care about the message. The fact we got one out
                    // of the xml response means it all worked.
                    g_free(message);
                }
                g_free(response);
            }
            blip_entry_free(entry);
        }
    }
    return success;
}


/*----------------------------------------------------------------------
 * Resource: BLIP_RESOURCE_GET_TIME
 *--------------------------------------------------------------------*/

/**
 * Wrapper around CallGetTimeXyz() to potentially call the real resource
 * or return a faked response.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetTime(const gchar* apiKey,
                          BlipError * oError)
{
    return CallGetTimeReal(apiKey, oError);
    //return CallGetTimeDummy(apiKey);
}

/**
 * Calls the "get/time/" resource and returns the raw response from
 * the api.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetTimeReal(const gchar* apiKey,
                              BlipError * oError)
{
    gchar* url= blip_build_api_string(BLIP_RESOURCE_GET_TIME,
                                      BLIP_ARG_API_KEY,    apiKey,
                                      BLIP_ARGS_DONE);
    gchar* response= blip_fetch_response(url, oError);
    g_free(url);
    return response;
}

/**
 * Dummy to avoid actually calling blip api.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetTimeDummy(const gchar* apiKey)
{
    return g_strdup("<blipapi>"
                    "  <request_id></request_id>"
                    "  <error></error>"
                    "  <data>"
                    "    <timestamp>1245328188</timestamp>"
                    "  </data>"
                    "</blipapi>");
}

/**
 * Extracts the timestamp from a get/time response.
 * @param [in] callResponse
 *   The raw data that was returned by the call to the api.
 * @param [out] timestamp
 *   If the timestamp is found in callResponse, this is set.
 * @return
 *   If everything worked and timestamp was successfully retrieved,
 *   TRUE is returned. Otherwise FALSE is returned.
 */
static gboolean DecodeGetTime(const gchar* callResponse,
                              time_t* timestamp,
                              BlipError * oError)
{
    BlipXml blipXml;
    gboolean result= FALSE;

    blipXml= blip_xml_create(callResponse, oError);
    if (blipXml)
    {
        gchar* timeStr;
        timeStr= blip_xml_get_text_node_contents(blipXml,
                                                 "/blipapi/data/timestamp/text()",
                                                 oError);
        if (timeStr)
        {
            *timestamp= atoi(timeStr);
            g_free(timeStr);
            result= TRUE;
        }

        blip_xml_free(blipXml);
    }

    return result;
}

static gboolean GetTime(const char* apiKey,
                        time_t* timestamp,
                        BlipError * oError)
{
    gboolean result= FALSE;

    gchar* callResponse= CallGetTime(apiKey, oError);
    if (callResponse != NULL)
    {
        if (DecodeGetTime(callResponse, timestamp, oError))
        {
            result= TRUE;
        }
        g_free(callResponse);
    }
    return result;
}

time_t blip_get_time(const BlipFoto blip)
{
    if (blip)
    {
        time_t dummy;
        return time(&dummy) - blip->timeDelta;
    }
    return 0;
}

/*----------------------------------------------------------------------
 * Resource: BLIP_RESOURCE_GET_TOKEN
 *--------------------------------------------------------------------*/

/**
 * Wrapper around CallGetTokenXyz() to potentially call the real resource
 * or return a faked response.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetToken(const gchar* apiKey,
                           const gchar* temporaryToken,
                           BlipError * oError)
{
    return CallGetTokenReal(apiKey, temporaryToken, oError);
    //return CallGetTokenDummy(apiKey, temporaryToken);
}

/**
 * Calls the "get/token/" resource and returns the raw response from
 * the api.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetTokenReal(const gchar* apiKey,
                               const gchar* temporaryToken,
                               BlipError * oError)
{
    gchar* url= blip_build_api_string(BLIP_RESOURCE_GET_TOKEN,
                                      BLIP_ARG_API_KEY,    apiKey,
                                      BLIP_ARG_TEMP_TOKEN, temporaryToken,
                                      BLIP_ARGS_DONE);
    gchar* response= blip_fetch_response(url, oError);
    g_free(url);
    return response;
}

/**
 * Dummy to avoid actually calling blip api.
 * @return use g_free() to destroy returned string.
 */
static gchar* CallGetTokenDummy(const gchar* apiKey,
                                const gchar* temporaryToken)
{
    return g_strdup("<blipapi>"
                    "  <request_id/>"
                    "  <error/>"
                    "  <data>"
                    "    <username>somebody</username>"
                    "    <token>12341234123412341234123412341234</token>"
                    "  </data>"
                    "</blipapi>");
}

/**
 * Extracts the username and token from a get/token response.
 * @param [in] callResponse
 *   The raw data that was returned by the call to the api.
 * @param [out] username
 *   If the username is found in callResponse, this is set. Use
 *   g_free() to destroy the returned string if function was
 *   successful.
 * @param [out] idToken
 *   If the token is found in callResponse, this is set. Use
 *   g_free() to destroy the returned string if function was
 *   successful.
 * @return
 *   If both values were successfully retrieved, TRUE is returned.
 *   Otherwise FALSE is returned.
 */
static gboolean DecodeGetToken(const gchar* callResponse,
                               gchar** username,
                               gchar** idToken,
                               BlipError * oError)
{
    gboolean result= FALSE;
    BlipXml blipXml;

    blipXml= blip_xml_create(callResponse, oError);
    if (blipXml)
    {
        if (*username= blip_xml_get_text_node_contents(blipXml,
                                                       "/blipapi/data/username/text()",
                                                       oError))
        {
            if (*idToken= blip_xml_get_text_node_contents(blipXml,
                                                          "/blipapi/data/token/text()",
                                                          oError))
            {
                result= TRUE;
            }
            else
            {
                g_free(*username);
                *username= NULL;
            }
        }
        blip_xml_free(blipXml);
    }

    return result;
}

gboolean blip_get_token(const BlipFoto blip,
                        const gchar* temporaryToken,
                        gchar** username,
                        gchar** idToken,
                        BlipError * oError)
{
    gboolean result= FALSE;

    if (blip && temporaryToken)
    {
        gchar* callResponse= CallGetToken(blip->apiKey,
                                          temporaryToken,
                                          oError);
        if (callResponse != NULL)
        {
            if (DecodeGetToken(callResponse, username, idToken, oError))
            {
                result= TRUE;
            }
            g_free(callResponse);
        }
    }
    return result;
}

static gchar* CallGetDateValidation(const BlipFoto blip,
                                    guint year,
                                    guint month,
                                    guint day,
                                    BlipError * oError)
{
    gchar entrydate[11];
    g_sprintf(entrydate,
              "%04d-%02d-%02d",
              year,
              month,
              day);

    gchar* url= blip_build_api_string(BLIP_RESOURCE_GET_DATEVALIDATION,
                                      BLIP_ARG_API_KEY,    blip->apiKey,
                                      BLIP_ARG_ENTRY_DATE, entrydate,
                                      BLIP_ARGS_DONE);

    gchar* signedurl= blip_sign_url(blip, url);
    g_free(url);

    gchar* response= blip_fetch_response(signedurl, oError);
    g_free(signedurl);
    return response;
}

BlipError DecodeGetDateValidation(BlipFoto blip,
                                  const gchar* callResponse,
                                  gboolean * success)
{
    BlipXml blipXml;
    BlipError error= NULL;

    blipXml= blip_xml_create(callResponse, NULL);
    if (blipXml)
    {
        error= blip_xml_extract_error(blipXml);
        if (!error)
        {
            *success= TRUE;
        }
        else if (blip_error_domain(error) == BLIP_DOMAIN_BLIP_API
                 && blip_error_code(error) == BLIP_API_ERROR_DATE_USED_ALREADY)
        {
            *success= FALSE;
            blip_error_free(error);
            error= NULL;
        }
        blip_xml_free(blipXml);
    }
    return error;
}

gboolean blip_validate_authentication(const BlipFoto blip,
                                      gboolean * validated,
                                      BlipError * oError)
{
    BlipError error= NULL;
    if (blip)
    {
        time_t seconds;
        struct tm * timeinfo;
        time(&seconds);
        timeinfo= localtime(&seconds);
        gchar* callResponse= CallGetDateValidation(blip,
                                                   timeinfo->tm_year+1900,
                                                   timeinfo->tm_mon+1,
                                                   timeinfo->tm_mday,
                                                   &error);
        if (callResponse != NULL)
        {
            gboolean dateValid;
            error= DecodeGetDateValidation(blip,
                                           callResponse,
                                           &dateValid);
            if (!error)
            {
                // Don't care about dateValidation, just that the
                // call worked without error, implying authentication
                // is working.
                *validated= TRUE;
                blip_error_free(error);
                error= NULL;
            }
            else if (blip_error_match(error,
                                      BLIP_DOMAIN_BLIP_API,
                                      BLIP_API_ERROR_AUTHENTICATION_FAILED,
                                      BLIP_ERROR_MATCH_END))
            {
                *validated= FALSE;
                blip_error_free(error);
                error= NULL;
            }
            g_free(callResponse);
        }
    }

    if (oError)
        *oError= error;
    else
        blip_error_free(error);

    return error == NULL;
}
