/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#include <curl/curl.h>

#include "blip_connection.h"
#include "blip_response.h"

static size_t ResponseAppendCB(void *buffer,
                               size_t size,
                               size_t nmemb,
                               void *userp);

static int CurlProgressCB(void *clientp,
                          double dltotal,
                          double dlnow,
                          double ultotal,
                          double ulnow);

struct BlipConnection_s
{
    struct curl_httppost* post_first;
    struct curl_httppost* post_last;
    char curl_error[CURL_ERROR_SIZE];
    CURL * curl;
    BlipResponse response;
    BlipConnectionUploadCallback uploadCallback;
    void * uploadClientData;
};

BlipConnection blip_connection_create()
{
    BlipConnection conn= (BlipConnection) g_try_malloc0(sizeof(struct BlipConnection_s));
    if (conn)
    {
        conn->curl= curl_easy_init();
        conn->response= blip_response_create();
        curl_easy_setopt(conn->curl, CURLOPT_ERRORBUFFER, conn->curl_error);
        curl_easy_setopt(conn->curl, CURLOPT_WRITEFUNCTION, ResponseAppendCB);
        curl_easy_setopt(conn->curl, CURLOPT_WRITEDATA, conn->response);
        curl_easy_setopt(conn->curl, CURLOPT_PROGRESSFUNCTION, CurlProgressCB);
        curl_easy_setopt(conn->curl, CURLOPT_PROGRESSDATA, conn);
        curl_easy_setopt(conn->curl, CURLOPT_NOPROGRESS, 0);
    }
    return conn;
}

void blip_connection_free(BlipConnection conn)
{
    if (conn)
    {
        if (conn->post_first)
            curl_formfree(conn->post_first);
        blip_response_free(conn->response);
        curl_easy_cleanup(conn->curl);
        g_free(conn);
    }
}

gchar* blip_connection_go(BlipConnection conn,
                          const gchar* url,
                          BlipError * oError)
{
    gchar * result= NULL;
    if (conn && url)
    {
        curl_easy_setopt(conn->curl, CURLOPT_URL, url);

        if (conn->post_first)
            curl_easy_setopt(conn->curl, CURLOPT_HTTPPOST, conn->post_first);

        CURLcode status= curl_easy_perform(conn->curl);

        if (status == CURLE_OK)
        {
            result= blip_response_get_text(conn->response);
        }
        else if (oError)
        {
            if (status == CURLE_ABORTED_BY_CALLBACK)
            {
                *oError= blip_error_create(BLIP_DOMAIN_BLIP_LIB,
                                           BLIP_LIB_CONNECTION_CANCELLED_BY_USER,
                                           "Connection cancelled by user");
            }
            else
            {
                *oError= blip_error_create(BLIP_DOMAIN_TRANSPORT,
                                           status,
                                           conn->curl_error);
            }
        }
    }
    return result;
}

void blip_connection_post_text(BlipConnection conn, const gchar *fieldName, const gchar *fieldText)
{
    if (conn && fieldName && fieldText)
    {
        curl_formadd(&conn->post_first,
                     &conn->post_last,
                     CURLFORM_PTRNAME, fieldName,
                     CURLFORM_PTRCONTENTS, fieldText,
                     CURLFORM_END);
    }
}

void blip_connection_post_file(BlipConnection conn, const gchar *fieldName, const gchar *fileFullPath)
{
    if (conn && fieldName && fileFullPath)
    {
        curl_formadd(&conn->post_first,
                     &conn->post_last,
                     CURLFORM_PTRNAME, fieldName,
                     CURLFORM_FILE, fileFullPath,
                     CURLFORM_END);
    }
}

void blip_connection_set_upload_callback(BlipConnection conn,
                                         BlipConnectionUploadCallback callback,
                                         void *client_data)
{
    if (conn)
    {
        conn->uploadCallback= callback;
        conn->uploadClientData= client_data;
    }
}

// --------------------------------------------------------------------------------
// Static functions
// --------------------------------------------------------------------------------

static size_t ResponseAppendCB(void *buffer,
                               size_t size,
                               size_t nmemb,
                               void *userp)
{
    return blip_response_append((BlipResponse)userp, (gchar*)buffer, size*nmemb);
}

static int CurlProgressCB(void *clientp,
                          double dltotal,
                          double dlnow,
                          double ultotal,
                          double ulnow)
{
    int result= 0;
    BlipConnection conn= (BlipConnection) clientp;
    if (conn && conn->uploadCallback)
    {
        BlipConnectionStatus status= conn->uploadCallback(conn->uploadClientData, (gdouble)ulnow/(gdouble)ultotal);
        if (status != BLIP_CONNECTION_CONTINUE)
        {
            result= 1;
        }
    }
    return result;
}
