/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "calendar-recurrence-editor.h"
#include <libintl.h>
#include <locale.h>
#include <hildon/hildon.h>
#include <CEvent.h>
#include <CRecurrence.h>
#include <Common.h>
#include "calendar-event-preview.h"

static void
_show_hide_end_selection_button(GtkToggleButton* button, gpointer user_data)
{
  if(gtk_toggle_button_get_active(button))
  {
    gtk_widget_show(GTK_WIDGET(user_data));
  }
  else
  {
    gtk_widget_hide(GTK_WIDGET(user_data));
  }
}

static int
_get_recurrence_until(CEvent* event, int& until)
{
  until = 0;
  CRecurrence* recurrence = event->getRecurrence();
  if(recurrence)
  {
    vector<CRecurrenceRule*> rules = recurrence->getRecurrenceRule();
    if(rules.size()==1)
    {
      until = rules[0]->getUntil();
    }
  }
}

static int
_get_interval(CEvent* event)
{
  int interval = 0;
  CRecurrence* recurrence = event->getRecurrence();
  if(recurrence)
  {
    vector<CRecurrenceRule*> rules = recurrence->getRecurrenceRule();
    if(rules.size()==1)
    {
      CRecurrenceRule cr;
      cr.rruleParser(rules[0]->toString());
      interval = cr.getInterval();
    }
  }
  return interval;
}

static void
_enable_disable_recurrence_details(HildonTouchSelector* selector, gint column, gpointer user_data)
{
  if(hildon_touch_selector_get_active(selector, 0) == 0)
  {
    gtk_widget_set_sensitive(GTK_WIDGET(user_data), FALSE);
  }
  else
  {
    gtk_widget_set_sensitive(GTK_WIDGET(user_data), TRUE);
  }
}

static
GtkWidget*
_create_recurrence_touch_selector_for_event(CEvent* event)
{
  GtkWidget* recurrence_picker_selector = hildon_touch_selector_new_text();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_never"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_every_day"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_every_workday"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_every_week"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_every_month"));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				    dgettext("calendar", "cal_va_every_year"));

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
				   0, 0);
  CRecurrence* recurrence = event->getRecurrence();

  if(recurrence != NULL)
  {
    int f = recurrence->getRtype();
    vector<CRecurrenceRule*> recrules = recurrence->getRecurrenceRule();
    if(recurrence->getRrule().size()>1)
    {
	hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
					  dgettext("calendar", "cal_fi_repeat_complex"));
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 6);
    }
    else
    {
      switch(f)
      {
      case E_DAILY:
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 1);
	break;
      case E_WEEKDAY:
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 2);
	break;
      case E_WEEKLY:
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 3);
	break;
      case E_MONTHLY:
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 4);
	break;
      case E_YEARLY:
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 5);
	break;
      case E_COMPLEX:
	hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(recurrence_picker_selector),
					  dgettext("calendar", "cal_fi_repeat_complex"));
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 
					 0, 6);
	break;
      }
    }
  }
  return recurrence_picker_selector;
}

static GtkWidget*
_create_interval_picker_touch_selector(CEvent* event)
{
  GtkWidget* interval_picker_selector = hildon_touch_selector_new_text();
  gchar buf[3];
  int i;
  for(i=1;i<31;++i)
  {
    sprintf(buf, "%2d",i);
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(interval_picker_selector),
				      buf);
  }
  int interval = _get_interval(event);
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(interval_picker_selector), 0, interval-1);
  return interval_picker_selector;
}

static void
_set_recurrence_type(CEvent* event, 
		     HildonTouchSelector* recurrence_picker_selector)
{
  int recurrence_type = hildon_touch_selector_get_active(recurrence_picker_selector, 0);  
  if(recurrence_type == 0)
  {
    event->removeRecurrence();
  }
  else
  {
    if(!event->getRecurrence())
    {
      CRecurrence* newRecurrence = new CRecurrence();
      event->setRecurrence(newRecurrence);
      delete newRecurrence;
    }
    switch(recurrence_type)
    {
    case 1:
      event->setRtype(E_DAILY);
      break;
    case 2:
      event->setRtype(E_WEEKDAY);
      break;
    case 3:
      event->setRtype(E_WEEKLY);
      break;
    case 4:
      event->setRtype(E_MONTHLY);
      break;
    case 5:
      event->setRtype(E_YEARLY);
      break;
    }
  }
}

static gchar*
_create_text_for_recurrence(CEvent* event)
{
  CRecurrence* recurrence = event->getRecurrence();
  if(recurrence)
  {
    const gchar* recurrence_text = NULL;
    switch(event->getRtype())
    {
    case E_DAILY:
      recurrence_text = dgettext("calendar", "cal_va_every_day");
      break;
    case E_WEEKDAY:
      recurrence_text = dgettext("calendar", "cal_va_every_workday");
      break;
    case E_WEEKLY:
      recurrence_text = dgettext("calendar", "cal_va_every_week");
      break;
    case E_MONTHLY:
      recurrence_text = dgettext("calendar", "cal_va_every_month");
      break;
    case E_YEARLY:
      recurrence_text = dgettext("calendar", "cal_va_every_year");
      break;
      }
    vector<CRecurrenceRule*> rules = recurrence->getRecurrenceRule();
    if(rules.size()==1)
    {
      if(rules[0]->checkRuleIsComplex())
      {
	return g_strdup(dgettext("calendar", "cal_fi_repeat_complex"));	
      }
      CRecurrenceRule cr;
      cr.rruleParser(rules[0]->toString());
      int interval = cr.getInterval();
      return g_strdup_printf("%s (%s %d)", recurrence_text, gettext("Interval"), interval);
    }
    return g_strdup_printf("%s", recurrence_text);
  }
  return g_strdup(dgettext("calendar", "cal_va_never"));  
}

static string
_create_rrule(int until, uint recurrence_type, uint interval_type)
{
  static string rrules[] = {"DAILY", "WEEKLY;BYDAY=MO,TU,WE,TH,FR", "WEEKLY", "MONTHLY", "YEARLY"};

  struct icaltimetype uical;
  icaltimezone* ictz = CMulticalendar::MCInstance()->getSystemTimeZoneAsIcalTimeZone();
  uical = icaltime_from_timet_with_zone(until, 0, ictz);
  string rrule_str = "FREQ=" + rrules[recurrence_type-1];
  if(interval_type>1)
  {
    stringstream ss;
    ss << ";INTERVAL=" << interval_type;
    rrule_str += ss.str();
  }
  if(until>0)
  {
    string until_string = icaltime_as_ical_string(uical);
    rrule_str +=";UNTIL=" +until_string;
  }
  return rrule_str;
}

void
open_recurrence_selection_dialog(GtkButton* button, gpointer user_data)
{
  CEvent* event = (CEvent*)user_data;
  GtkWidget* dialog;
  GtkWidget* recurrence_button;
  GtkWidget* recurrence_detail_box;
  GtkWidget* interval_button;
  GtkWidget* ends_never_radio_button;
  GtkWidget* ends_at_time_radio_button;
  GtkWidget* select_time_button;
  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       GTK_DIALOG_NO_SEPARATOR,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);

  recurrence_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
					       HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

  hildon_button_set_title(HILDON_BUTTON(recurrence_button), 
			  dgettext("calendar", "cal_fi_repeat"));

  ends_never_radio_button = hildon_gtk_radio_button_new(HILDON_SIZE_FINGER_HEIGHT, NULL);
  ends_at_time_radio_button = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT,
								      GTK_RADIO_BUTTON(ends_never_radio_button));

  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(ends_never_radio_button), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(ends_at_time_radio_button), FALSE);


  gtk_button_set_label(GTK_BUTTON(ends_never_radio_button), dgettext("calendar", "cal_va_never"));
  gtk_button_set_label(GTK_BUTTON(ends_at_time_radio_button), dgettext("calendar", "cal_fi_enddate"));

  select_time_button = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);


  g_signal_connect(ends_at_time_radio_button, "toggled", G_CALLBACK(_show_hide_end_selection_button), select_time_button);

  recurrence_detail_box = gtk_vbox_new(FALSE, 0);
  GtkWidget* ends_box = gtk_hbox_new(TRUE, 0);
  GtkWidget* end_selection_box = gtk_hbox_new(TRUE, 0);

  interval_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
					     HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
  hildon_button_set_title(HILDON_BUTTON(interval_button), 
			  gettext("Interval"));
  GtkWidget* interval_picker_selector = _create_interval_picker_touch_selector(event);

  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(interval_button),
				    HILDON_TOUCH_SELECTOR(interval_picker_selector));
  
  GtkWidget* recurrence_picker_selector = _create_recurrence_touch_selector_for_event(event);
  
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(recurrence_button),
				    HILDON_TOUCH_SELECTOR(recurrence_picker_selector));
  g_signal_connect(recurrence_picker_selector, "changed", G_CALLBACK(_enable_disable_recurrence_details), recurrence_detail_box);
  GtkWidget* ends_frame = gtk_frame_new(gettext("Ends:"));
  
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), recurrence_button, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(recurrence_detail_box), interval_button, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(ends_box), ends_never_radio_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(ends_box), ends_at_time_radio_button, TRUE, TRUE, 0);

  gtk_container_add(GTK_CONTAINER(ends_frame), ends_box);
  gtk_box_pack_start(GTK_BOX(recurrence_detail_box), ends_frame, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(recurrence_detail_box), end_selection_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(end_selection_box), select_time_button, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), recurrence_detail_box, TRUE, FALSE, 0);

  gtk_widget_show_all(dialog);
  gtk_widget_hide(select_time_button);

  if(event->getRecurrence())
  {
    time_t until = 0;
    until = event->getUntil();
    if(until>0)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ends_at_time_radio_button), TRUE);
      gtk_widget_show(select_time_button);
      struct tm* time_stamp;
      time_stamp = localtime(&until);
      hildon_date_button_set_date(HILDON_DATE_BUTTON(select_time_button),
				  time_stamp->tm_year+1900,
				  time_stamp->tm_mon,
				  time_stamp->tm_mday);
    }
  }
  else
  {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ends_never_radio_button), TRUE);
  }
  int ret = gtk_dialog_run(GTK_DIALOG(dialog));

  if(ret == GTK_RESPONSE_ACCEPT)
  {
    _set_recurrence_type(event, HILDON_TOUCH_SELECTOR(recurrence_picker_selector));

    if(event->getRecurrence())
    {
      int until = -1;

      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ends_never_radio_button)))
      {
	until = -1;
      }
      else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ends_at_time_radio_button)))
      {
	struct tm time_stamp;
	time_t end_date = event->getDateEnd();
	struct tm* end_time = localtime(&end_date);
	time_stamp.tm_hour = end_time->tm_hour;
	time_stamp.tm_min = end_time->tm_min;
	time_stamp.tm_sec = end_time->tm_sec;
	hildon_date_button_get_date(HILDON_DATE_BUTTON(select_time_button),
				    (guint*)&time_stamp.tm_year,
				    (guint*)&time_stamp.tm_mon,
				    (guint*)&time_stamp.tm_mday);
	time_stamp.tm_mon;
	time_stamp.tm_year-=1900;
	until = mktime(&time_stamp);
      }

      int interval_type = 1 + hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(interval_picker_selector), 0);
      int recurrence_type = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(recurrence_picker_selector), 0);
      if(recurrence_type>0 && recurrence_type < 6)
      {
	vector<CRecurrenceRule*> recrules = event->getRecurrence()->getRecurrenceRule();
	if(recrules.size()==1)
	{
	  string rrule = _create_rrule(until, recurrence_type, interval_type);
	  recrules[0]->setRrule(rrule);
	}
	event->setUntil(until);
      }
    }
    gchar* value_text = _create_text_for_recurrence(event);
    hildon_button_set_value(HILDON_BUTTON(button),value_text);
    g_free(value_text);
  }
  gtk_widget_destroy(dialog);
}
