/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        hiscore.cpp ( Letter Hunt Library, C++ )
 *
 * COMMENTS:
 */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "hiscore.h"
#include "msg.h"
#include "grammar.h"
#include "rand.h"
#include "score.h"
#include "text.h"

#include "avatar.h"
#include "glbdef.h"

#include <fstream>
#include <iostream>
using namespace std;

#define MAXLEN 20
#define MAXWORDLEN 200
#define NUMSCORE 10

class HISCORE_ENTRY
{
public:
    char		name[MAXLEN+1];
    double		prob;
    int			exp;

    char		stats[4];

    void		calcChecksum();
    char		checksum[4];
};

void
HISCORE_ENTRY::calcChecksum()
{
    unsigned		hash;
    const char		*s;

    hash = 0;
    for (s = name; *s; s++)
    {
	hash += *s;
	hash *= 37;
    }
    hash = rand_wanginthash(hash);
    hash ^= (int) (prob * 100+0.5);
    hash = rand_wanginthash(hash);
    hash ^= exp;
    hash = rand_wanginthash(hash);

    hash ^= stats[0] + stats[1] * 256 + stats[2] * 256;
    hash = rand_wanginthash(hash);

    // Convert to checksum...
    checksum[0] = (hash % 26) + 'A';
    hash /= 26;
    checksum[1] = (hash % 26) + 'A';
    hash /= 26;
    checksum[2] = (hash % 26) + 'A';
    checksum[3] = 0;
}

HISCORE_ENTRY	glbScores[NUMSCORE];

int
hiscore_getplace(bool victory, double prob, int exp)
{
    int		place;
    
    for (place = 0; place < NUMSCORE; place++)
    {
	if (victory && glbScores[place].stats[0] != 'V')
	    break;
	if (!victory && glbScores[place].stats[0] == 'V')
	    continue;
	if (exp < glbScores[place].exp)
	    break;
	if (exp == glbScores[place].exp)
	{
	    if (prob < glbScores[place].prob)
		break;
	}
    }

    return place;
}

void
hiscore_addscoreinternal(double prob, int exp, const char *name,
			 const char *stats)
{
    // Determine where it goes...
    int		place, i;

    place = hiscore_getplace(stats[0] == 'V', prob, exp);

    if (place < NUMSCORE)
    {
	// Made it on the top scores...
	for (i = NUMSCORE-2; i >= place; i--)
	{
	    glbScores[i+1] = glbScores[i];
	}

	// Write in our new one.
	glbScores[place].prob = prob;
	glbScores[place].exp = exp;
	strncpy(glbScores[place].name, name, MAXLEN);
	glbScores[place].name[MAXLEN] = '\0';

	strncpy(glbScores[place].stats, stats, 4);
	glbScores[place].stats[3] = 0;

	// Save new scores
	hiscore_save();
    }
}

#define HISCORE_FILE "../save/hiscore.txt"
void
hiscore_load()
{
    int			i, pos;
    ifstream		is(HISCORE_FILE);
    char		line[500];
    char		name[MAXLEN+1];
    double		prob;
    int			exp;
    char		checksum[4];
    char		stats[4];

    checksum[3] = 0;

    // Initialize our hiscores to empty...
    for (i = 0; i < NUMSCORE; i++)
    {
	glbScores[i].prob = 1.0;
	glbScores[i].exp = 0;
	sprintf(glbScores[i].name, "%20s", "No one");
	sprintf(glbScores[i].stats, "d??");
	glbScores[i].calcChecksum();
    }
    
    // Read in the hiscores.
    while (is.getline(line, 500))
    {
	text_striplf(line);
	// Parse the entry...
	for (i = 0; i < MAXLEN; i++)
	{
	    name[i] = line[i];
	    // Abort on a bad result.
	    if (!line[i])
		return;
	}
	name[MAXLEN] = 0;
	// Back search for non-white space.
	for (i = MAXLEN-1; i > 0; i--)
	{
	    if (name[i] == ' ')
		name[i] = '\0';
	    else
		break;
	}
	
	// Extract the score component.
	pos = MAXLEN+1;

	//Extract stats package.
	if (line[pos])
	    stats[0] = line[pos++];
	if (line[pos])
	    stats[1] = line[pos++];
	if (line[pos])
	    stats[2] = line[pos++];
	if (line[pos])
	    pos++;
	stats[3] = '\0';
	
	prob = 0.0;
	exp = 0;
	while (line[pos] && isdigit(line[pos]))
	{
	    prob *= 10.0;
	    prob += line[pos] - '0';
	    pos++;
	}
	if (line[pos] == '.')
	{
	    double		frac = 0.1;
	    pos++;

	    while (line[pos] && isdigit(line[pos]))
	    {
		prob += (line[pos] - '0') * frac;
		frac *= 0.1;
		pos++;
	    }
	}
	// Finally, exponent.
	if (line[pos] == 'E')
	{
	    pos++;
	    exp = atoi(&line[pos]);
	}
	
	// Skip over exponent and any other probability junk
	while (line[pos] && !isspace(line[pos]))
	    pos++;
	// Go to checksume...
	if (!line[pos])
	    continue;
	pos++;

	// Read checksum
	i = 0;
	while (line[pos] && i < 3)
	{
	    checksum[i] = line[pos];
	    i++;
	    pos++;
	}

	// Verify the score.
	{
	    HISCORE_ENTRY	entry;
	    
	    strcpy(entry.name, name);
	    entry.prob = prob;
	    entry.exp = exp;
	    strcpy(entry.stats, stats);

	    entry.calcChecksum();
	    if (strcmp(entry.checksum, checksum))
	    {
		cerr << "Checksum failed: " << entry.name << " rejected!" << endl;
		// Reject!
		continue;	
	    }
	}

	hiscore_addscoreinternal(prob, exp, name, stats);
    }
}

void
hiscore_save()
{
    ofstream	os(HISCORE_FILE);

    char	line[500];
    int		i;

    for (i = 0; i < NUMSCORE; i++)
    {
	glbScores[i].calcChecksum();
	sprintf(line, "%20s %3s %.3fE%d %3s\n", glbScores[i].name, 
			glbScores[i].stats,
			glbScores[i].prob, glbScores[i].exp,
			glbScores[i].checksum);
	os << line;
    }
}

void
hiscore_display()
{
    int		i;
    char	line[500];

    msg_report("High Scores:\n");
    
    for (i = 0; i < NUMSCORE; i++)
    {
	sprintf(line, "%20s - %3s %.3fE%d\n", glbScores[i].name, 
		glbScores[i].stats,
		glbScores[i].prob, glbScores[i].exp);
	msg_report(line);
    }
}

void
hiscore_addscore(double prob, int exp)
{
    int		place;

    place = hiscore_getplace(avatar_victory(), prob, exp);

    if (place >= NUMSCORE)
    {
	msg_report("You did not make the high score list.\n");
    }
    else
    {
	char		buf[MAXLEN+1];
	char		name[MAXLEN+1];
	msg_report("You came in ");
	msg_report(gram_createplace(place+1));
	msg_report(" place!\n");
	msg_getString("Your name? ", buf, MAXLEN);

	// Convert buf into %20s form.
	sprintf(name, "%20s", buf);

	// Create our stat package
	char		stats[4];

	stats[0] = avatar_victory() ? 'V' : 'd';
	stats[1] = glb_racedefs[avatar_race()].symbol;
	stats[2] = glb_roledefs[avatar_role()].symbol;
	stats[3] = '\0';

	hiscore_addscoreinternal(prob, exp, name, stats);
    }
}
