/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        main.cpp ( Save Scummer Library, C++ )
 *
 * COMMENTS:
 */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <time.h>

#include "gfxengine.h"
#include "glbdef.h"
#include "map.h"
#include "mob.h"
#include "panel.h"
#include "msg.h"
#include "text.h"
#include "speed.h"
#include "item.h"
#include "score.h"
#include "hiscore.h"
#include "worldstate.h"
#include "dpdf.h"
#include "avatar.h"

#include <fstream>
using namespace std;

// Might as well make these global and save some pain.
PANEL	glb_output(79 - 37, 28, true);
PANEL	glb_status(35, 2);

// DO we stop at death message?
bool	glb_playthroughdeath = false;

#define SAVEFILE_NAME "../save/savescummer.sav"
void
save_stuff(MAP *mainmap)
{
    // Did I mention my hatred of streams?
#ifdef WIN32
    ofstream	os(SAVEFILE_NAME, ios::out | ios::binary);
#else
    ofstream	os(SAVEFILE_NAME);
#endif

    score_save(os);
    avatar_save(os);

    MAP		*map;
    u8		 c;

    for (map = mainmap; map; map = map->peekDownMap())
    {
	c = 1;
	os.write((const char *) &c, 1);
	map->save(os);
    }
    c = 0;
    os.write((const char *) &c, 1);
}

MAP *
load_stuff()
{
    // Destroy any existing maps...
    MOB::setAvatar(0);

    // Open file for reading.
#ifdef WIN32
    ifstream	is(SAVEFILE_NAME, ios::in | ios::binary);
#else
    ifstream	is(SAVEFILE_NAME);
#endif

    if (!is)
	return 0;

    u8			c;
    
    score_load(is);
    avatar_load(is);

    // Load the maps...
    MAP			*map, *lastmap, *mainmap;

    lastmap = 0;
    while (1)
    {
	is.read((char *) &c, 1);
	if (!c)
	    break;

	map = new MAP;
	map->load(is);

	if (lastmap)
	{
	    lastmap->setDownMap(map);
	    map->setUpMap(lastmap);
	}
	else
	    mainmap = map;

	lastmap = map;
    }

    // Find the avatar...
    mainmap->locateAvatar(MOB_AVATAR);

    msg_report(text_lookup("welcomeback", "You"));

    msg_newturn();
    
    return mainmap;
}

void
writestatusline()
{
    MOB		*avatar;

    avatar = MOB::getAvatar();

    glb_status.clear();
    if (!avatar)
    {
	glb_status.appendText("You are Dead.");
    }
    else
    {
	char	buf[100];

	sprintf(buf, "Health: %5d   Depth: %3d", avatar->getHP(), avatar->getMap()->getDepth());
	glb_status.appendText(buf);
	glb_status.newLine();
	sprintf(buf, "Mana:   %5d   Prob:  %.3fE%d", avatar_mp(), avatar_prob(), avatar_probexp());
	glb_status.appendText(buf);

	// Compute the proportion to fill.
	double		prop;
	int		col;

	prop = ((float)avatar->getHP() / avatar_maxhp());
	col = (int) (35 * prop + 0.5);

	glb_status.fillRect(0, 0, 35, 2, ATTR_NORMAL);
	// If we are invulnerable, signal it.
	if (avatar->hasItem(ITEM_INVULNERABLE))
	    glb_status.fillRect(0, 0, col, 1, ATTR_INVULNERABLE);
	else
	    glb_status.fillRect(0, 0, col, 1, ATTR_HEALTH);

	// Fill up the mana chart
	prop = ((float)avatar_mp() / avatar_maxmp());
	col = (int) (35 * prop + 0.5);
	glb_status.fillRect(0, 1, col, 1, ATTR_MANA);
    }
}

// Contrary to its name, it redraws everything EXCEPT the map!
void
fullRedraw()
{
    writestatusline();

    glb_output.redraw();
    glb_status.redraw();

    gfx_update();
}

// Rebuilds the avatar's FOV and its map.
// Does not do gfx update!
void
redrawAvatarMap()
{
    MOB		*avatar;

    avatar = MOB::getAvatar();
    if (!avatar)
	return;
    avatar->getMap()->buildFOV(avatar->getX(), avatar->getY());
    avatar->getMap()->redraw(1, 1, 35, 25, 17, 12);
}
    

bool
getdirection(int key, int &dx, int &dy, bool allowfast = false)
{
    dx = dy = 0;
    bool isfast = false;

    switch (key)
    {
	case 'Y':
	    isfast = true;
	case 'y':
	case '7':
	    dx = -1;
	    dy = -1;
	    break;
	case 'U':
	    isfast = true;
	case 'u':
	case '9':
	    dx = 1;
	    dy = -1;
	    break;
	case 'H':
	    isfast = true;
	case '4':
	case 'h':
	case GFX_KEYLEFT:
	    dx = -1;
	    dy = 0;
	    break;
	case 'J':
	    isfast = true;
	case '2':
	case 'j':
	case GFX_KEYDOWN:
	    dx = 0;
	    dy = 1;
	    break;
	case 'K':
	    isfast = true;
	case '8':
	case 'k':
	case GFX_KEYUP:
	    dx = 0;
	    dy = -1;
	    break;
	case 'L':
	    isfast = true;
	case '6':
	case 'l':
	case GFX_KEYRIGHT:
	    dx = 1;
	    dy = 0;
	    break;
	case 'B':
	    isfast = true;
	case 'b':
	case '1':
	    dx = -1;
	    dy = 1;
	    break;
	case 'N':
	    isfast = true;
	case 'n':
	case '3':
	    dx = 1;
	    dy = 1;
	    break;
	case '5':
	case '.':
	    dx = 0;
	    dy = 0;
	    break;
	    
	default:
	    // No valid direction!
	    return false;
    }

    if (isfast && !allowfast)
	return false;
    if (isfast)
    {
	dx *= 5;
	dy *= 5;
    }

    return true;
}

bool
selectsquare(int &x, int &y, PANEL &output, PANEL &status)
{
    bool		selected;
    bool		isblind;
    int			key, dx, dy;
    MOB			*avatar;

    avatar = MOB::getAvatar();
    if (!avatar)
	return false;

    isblind = avatar->hasItem(ITEM_BLIND);
    
    selected = false;
    // Examine...
    while (1)
    {
	// Highlight current position
	avatar->getMap()->redraw(1, 1, 35, 25, 17, 12);

	u8		sym;
	ATTR_NAMES  attr;

	gfx_getchar(x+1, y+1, sym, attr);
	attr = ATTR_HILITE;
	gfx_printchar(x+1, y+1, sym, attr);

	fullRedraw();
	
	// Get a direction & apply it.
	key = gfx_getKey();
	
	// Exit if bad...
	if (!getdirection(key, dx, dy, true))
	{
	    if (key == ' ' || key == '\n' || key == '\r')
	    {
		selected = true;
		break;
	    }
	    else if (key == '>')
	    {
		int		tx, ty;
		if (avatar->getMap()->findTile(TILE_DOWNSTAIRS, tx, ty))
		{
		    dx = tx - x;
		    dy = ty - y;
		}
	    }
	    else if (key == '<')
	    {
		int		tx, ty;
		if (avatar->getMap()->findTile(TILE_UPSTAIRS, tx, ty))
		{
		    dx = tx - x;
		    dy = ty - y;
		}
	    }
	    else if (key == '/')
	    {
		int		tx, ty;
		
		// Search for map symbol...
		msg_report("Jump to what symbol?  ");
		fullRedraw();
		key = gfx_getKey();
		tx = x; 
		ty = y;
		if (avatar->getMap()->findNextSymbol(key, tx, ty))
		{
		    msg_report("Done.  ");
		    dx = tx - x;
		    dy = ty - y;
		}
		else
		{
		    msg_report("Not found.  ");
		}
	    }
	    else
	    {
		break;
	    }
	}

	x += dx;
	y += dy;

	if (!avatar->getMap()->isValidCoord(x, y))
	{
	    if (x < 0)
		x = 0;
	    if (x >= avatar->getMap()->getWidth())
		x = avatar->getMap()->getWidth()-1;
	    if (y < 0)
		y = 0;
	    if (y >= avatar->getMap()->getHeight())
		y = avatar->getMap()->getHeight()-1;
	}

	avatar->getMap()->describeSquare(x, y, isblind);
	
	msg_newturn();
    }

    return selected;
}

MOB *
createAvatar(MAP *mainmap)
{
    MOB		*avatar;
    int		x, y;
    
    avatar = MOB::create(MOB_AVATAR);
    mainmap->findTile(TILE_UPSTAIRS, x, y);
    avatar->move(x, y);
    mainmap->addMob(avatar);
    avatar->setAvatar(avatar);
    MOB::getAvatarHP_DPDF() = DPDF(avatar->getHP());
    // avatar gets a melee weapon to start
    avatar->addItem(ITEM::create(ITEM_WEAPON, 0));

    // Determine a role and set our hp/mp
    avatar_setrole((ROLE_NAMES) rand_choice(NUM_ROLES));
    avatar_setrace((RACE_NAMES) rand_choice(NUM_RACES));

    avatar->gainHP(avatar_maxhp() - avatar->getHP());
    avatar_setmp(avatar_maxmp());

    avatar_setvictory(false);
    avatar_setprob(1.0, 0);

    return avatar;
}

MAP *
newGame(MAP *mainmap, PTRLIST<WORLDSTATE *> &statelist)
{
    MOB		*avatar;
    bool	 firstpass = true;

    score_init();

    while (1)
    {
	char	buf[200];
	int	key;

	// Destroy all old map lists.
	{
	    WORLDSTATE	*state;
	    while (state = statelist.pop())
		state->decRef();
	}
	statelist.clear();
	msg_clearhistory();

	// Delete the old map.
	delete mainmap;

	mainmap = new MAP;
	// Go straight to the dungeons!
	mainmap->build(0);

	// Create a new avatar.
	avatar = createAvatar(mainmap);

	statelist.append(WORLDSTATE::pushState(mainmap));

	if (firstpass)
	{
	    msg_report(text_lookup("welcome", avatar->getRawName()));
	    firstpass = false;
	}

	sprintf(buf, "\nYou are %s%s %s.\n(Left: reroll.  Right: accept.)\n",
		gram_getarticle(glb_racedefs[avatar_race()].name),
		glb_racedefs[avatar_race()].name,
		glb_roledefs[avatar_role()].name);
	msg_report(buf);

	redrawAvatarMap();
	fullRedraw();

	if (glb_playthroughdeath)
	    key = GFX_KEYRIGHT;
	else
	{
	    // Eat keys until acceptable!
	    while (1)
	    {
		key = gfx_getKey();
		if (key == GFX_KEYLEFT)
		{
		    // Re roll.
		    break;
		}
		else if (key == GFX_KEYRIGHT)
		{
		    // Accept.
		    break;
		}
	    }
	}

	if (key == GFX_KEYRIGHT)
	    break;
    }
    msg_newturn();

    return mainmap;
}

void
displaySavePoints(PTRLIST<WORLDSTATE *> &savepoint)
{
    char		buf[300];
    int			i;

    for (i = 0; i < 10; i++)
    {
	if (savepoint(i))
	{
	    sprintf(buf, "%d - T: %d, H: %d P: %.3fE%d\n",
			i,
			savepoint(i)->myTime,
			savepoint(i)->myAvatarHP,
			savepoint(i)->myAvatarProb,
			savepoint(i)->myAvatarProbExp);
	}
	else
	{
	    sprintf(buf, "%d - Empty\n", i);
	}
	msg_report(buf);
    }
}

int
main()
{
    gfx_init();
    text_init();
    spd_init();
    score_init();

    rand_setseed((long) time(0));

#ifdef USE_CURSES
    avatar_setplaybackdelay(2);
#else
    // SDL is already painfully slow :>
    avatar_setplaybackdelay(0);
#endif

    MAP		*mainmap;
    int		key;
    MOB		*avatar;
    PTRLIST<WORLDSTATE *>	statelist;
    PTRLIST<WORLDSTATE *>	savepoint;

    hiscore_load();

    gfx_clearbox(0, 0, 80, 30, ' ', ATTR_BORDER);

    glb_output.move(37, 1);
    glb_status.move(1, 27);

    // Scroll to bottom of output...
    {
	int		i;
	for (i = 0; i < 30; i++)
	    glb_output.newLine();
    }

    msg_registerpanel(&glb_output);

    int		dx, dy;
    bool	didmove;
    bool	forcemove;
    bool	hasquit = false, newturn = false;
    // Do we run the avatar AI?
    bool	autoplay = false;
    // Do we let the player do old-fashioned actions
    bool	exploremode = false;
		    
    if (mainmap = load_stuff())
    {
	// On successful load, delete the save game.
	unlink(SAVEFILE_NAME);
    }
    else
    {
	// start the game!
	mainmap = newGame(0, statelist);
    }

    savepoint.resize(10);

    // Create a new avatar.
    avatar = MOB::getAvatar();

    // Store initial state.
    statelist.append(WORLDSTATE::pushState(mainmap));
    
    while (!hasquit)
    {
	avatar = MOB::getAvatar();

	if (!avatar || avatar_victory())
	{
	    // Stop active states
	    if (avatar_victory() || !glb_playthroughdeath)
		autoplay = false;

	    // Quit the game!
	    msg_report("\n");

	    if (avatar_victory())
		msg_report(text_lookup("game", "victory"));
	    else
		msg_report(text_lookup("game", "lose"));

	    // Give the chance to go back...
	    msg_report("Hit Q to Start Again, Left Arrow to Undo.");

	    fullRedraw();

	    if (!avatar_victory() && glb_playthroughdeath)
	    {
		key = 'Q';
	    }
	    else
	    {
		while (1)
		{
		    key = gfx_getKey();

		    if (key == 'Q' || key == GFX_KEYLEFT)
			break;
		}
	    }
	    
	    switch (key)
	    {
		case GFX_KEYLEFT:
		    // Move back in time.
		    if (statelist.entries() > 1)
		    {
			delete mainmap;
			statelist.pop()->decRef();
			mainmap = statelist.top()->popState();
			avatar = MOB::getAvatar();
			break;
		    }
		    else
		    {
			// This is patently impossible, but we will just fall
			// through to quitting if you manage to die
			// before any checkpoints.

			// FALLTHROUGH
		    }

		case 'Q':
		{
		    int	    i;

		    if (!glb_playthroughdeath || avatar_victory())
		    {
			// We don't want the current probability of death,
			// which may very well be zero since the avatar
			// is dead, but the probability the turn before death.
			if (statelist.entries() >= 2)
			{
			    hiscore_addscore(
				    statelist.top(1)->myAvatarProb,
				    statelist.top(1)->myAvatarProbExp);
			}
		    }
		    
		    // start the game!
		    mainmap = newGame(mainmap, statelist);

		    // Reset our save points
		    for (i = 0; i < 10; i++)
		    {
			if (savepoint(i))
			{
			    savepoint(i)->decRef();
			    savepoint.set(i, 0);
			}
		    }

		    avatar = MOB::getAvatar();
		    break;
		}
	    }
	}

	// avatar now guaranteed valid.

	didmove = false;
	forcemove = false;

	redrawAvatarMap();

	fullRedraw();

	// Check for force move.
	didmove = false;
	if (newturn)
	    forcemove = avatar->aiForcedAction();
	if (forcemove)
	    didmove = true;

	// Status line may trigger an update, due to losing shields.
	fullRedraw();

	// Apply the playback delay so curses doesn't necessarily
	// whizz by.
	if (avatar_playbackdelay() > 0)
	    gfx_delay(avatar_playbackdelay());

	// Avatar could die in forced action
	avatar = MOB::getAvatar();
	if (!avatar)
	    continue;

	// Cancel autoplay if key pressed.
	if (autoplay && gfx_isKeyWaiting())
	{
	    // Eat the key
	    gfx_getKey();
	    msg_report("Paused.  ");
	    autoplay = false;
	}

	if (!didmove && autoplay)
	{
	    // Use the avatar AI to play.
	    didmove = avatar->aiDoAI();

	    // Avatar could die in ai action
	    avatar = MOB::getAvatar();
	    if (!avatar)
		continue;

	    // We always set didmove to true even if the ai did something
	    // stupid.
	    didmove = true;
	}
	
	if (!didmove)
	    key = gfx_getKey();
	else
	    key = 0;

	if (exploremode)
	{
	    // Pre-empt usual keypresses with explore mode presses.
	    if (getdirection(key, dx, dy))
	    {
		// Move in the requested direction, reset the key
		key = 0;
		didmove = avatar->actionBump(dx, dy);
	    }

	    if (key == '<' || key == '>')
	    {
		didmove = avatar->actionClimb();
		key = 0;
	    }

	    // Check for firing...
	    if (key == 'f')
	    {
		ITEM		*b;

		b = avatar->lookupItem(ITEM_SPELLBOOK);
		if (!b)
		{
		    msg_report("You have no spellbook.  ");
		}
		else
		{
		    if (b->getRangeMana() > avatar_mp())
		    {
			msg_report("You lack the needed mana.  ");
		    }
		    else
		    {
			char		buf[300];

			sprintf(buf, "Cast from %s in what direction?  ",
				b->getName());

			msg_report(buf);
			fullRedraw();
			key = gfx_getKey();
			if (getdirection(key, dx, dy))
			{
			    didmove = avatar->actionFire(dx, dy);
			}
			else
			{
			    msg_report("You decide not to cast a spell.  ");
			}
		    }
		}
		key = 0;
	    }
	}

	dx = dy = 0;
	switch (key)
	{
	    case GFX_KEYLEFT:
		// Move back in time.
		if (statelist.entries() > 1)
		{
		    delete mainmap;
		    statelist.pop()->decRef();
		    mainmap = statelist.top()->popState();
		}
		break;

	    case GFX_KEYRIGHT:
		// Move forward in time.
		avatar->aiDoAI();
		didmove = true;
		break;

	    case ' ':
		// Start the auto play.
		didmove = false;
		autoplay = true;
		msg_newturn();
		break;

	    case 'z':
	    {
		msg_report(text_lookup("welcome", avatar->getRawName()));
		msg_newturn();
		break;
	    }

	    case 's':
	    {
		int		key;
		
		// Quick save.
		msg_report("Save to a backup file...\n");
		displaySavePoints(savepoint);
		msg_report("Which slot? [0-9]\n");
		fullRedraw();

		key = gfx_getKey();
		if (key < '0' || key > '9')
		{
		    // Cancel.
		    msg_report("Cancelled.\n");
		}
		else
		{
		    int		i;

		    i = key - '0';
		    if (savepoint(i))
		    {
			savepoint(i)->decRef();
		    }
		    savepoint.set(i, statelist.top());
		    savepoint(i)->incRef();

		    msg_report("Slot written.\n");
		}
		
		msg_newturn();
		break;
	    }

	    case 'r':
	    {
		int		key;
		
		// Quick load.
		msg_report("Restore which backup file...\n");
		displaySavePoints(savepoint);
		msg_report("Which slot? [0-9]\n");
		fullRedraw();

		key = gfx_getKey();
		if (key < '0' || key > '9')
		{
		    // Cancel.
		    msg_report("Cancelled.\n");
		}
		else
		{
		    int		i;

		    i = key - '0';
		    if (!savepoint(i))
		    {
			// Can't load non-existant save point.
			msg_report("Cannot load empty save point.\n");
		    }
		    else
		    {
			// Push our current state.
			statelist.append(WORLDSTATE::pushState(mainmap));

			// Load our world state from the state.
			delete mainmap;
			mainmap = savepoint(i)->popState();
			msg_report("Restored save game.\n");
		    }
		}
		
		msg_newturn();
		break;
	    }

	    case 'c':
	    case 'i':
	    {
		char		buf[100];
		msg_report("Your character sheet:\n");

		sprintf(buf, "\nYou are %s%s %s.\n",
			gram_getarticle(glb_racedefs[avatar_race()].name),
			glb_racedefs[avatar_race()].name,
			glb_roledefs[avatar_role()].name);
		msg_report(buf);

		// Compute the probability of being alive
		{
		    int	min, q1, q2, q3, max;

		    MOB::getAvatarHP_DPDF().getQuartile(min, q1, q2, q3, max);

		    sprintf(buf, "HP Quartiles: %d..%d..%d..%d..%d\n",
				min, q1, q2, q3, max);
		    msg_report(buf);
		}

		ITEM		*inv;

		inv = avatar->getInventory();
		if (inv)
		{
		    msg_report("You are carrying:\n");
		    while (inv)
		    {
			msg_format("  %S\n", inv);
			// Write detailed information about weapons
			// and spellbooks.
			{
			    const char *buf;

			    buf = inv->getDetailedDescription();

			    if (buf)
			    {
				msg_report(buf);
			    }
			}
			inv = inv->getNext();
		    }
		}
		msg_newturn();

		break;
	    }

	    case 'A':
	    {
		if (glb_playthroughdeath)
		{
		    glb_playthroughdeath = false;
		    msg_report("Play through death disabled.");
		}
		else
		{
		    glb_playthroughdeath = true;
		    msg_report("Play through death enabled.");
		}
		msg_newturn();
		break;
	    }

	    case 'X':
	    {
		if (exploremode)
		{
		    exploremode = false;
		    msg_report("Explore mode disabled.  ");
		}
		else
		{
		    exploremode = true;
		    msg_report("Explore mode enabled.  ");
		}
		msg_newturn();
		break;
	    }

	    case '+':
	    {
		char		buf[200];
		// Speed up playback.
		if (avatar_playbackdelay() >= 0)
		{
		    avatar_setplaybackdelay(avatar_playbackdelay()-1);
		    sprintf(buf, "Playback delay now %d.\n", avatar_playbackdelay());
		    msg_report(buf);
		}
		else
		{
		    msg_report("This is as fast as it gets.\n");
		}
		msg_newturn();
		break;
	    }

	    case '-':
	    {
		char		buf[200];
		// Slow down playback.
		avatar_setplaybackdelay(avatar_playbackdelay()+1);
		sprintf(buf, "Playback delay now %d.\n", avatar_playbackdelay());
		msg_report(buf);
		msg_newturn();
		break;
	    }

	    case 'Q':
	    {
		msg_report("Saving...  ");

		fullRedraw();

		save_stuff(mainmap);

		msg_report("Done.  Hit a key to quit.  ");
		
		fullRedraw();

		gfx_breakKeyRepeat();
		gfx_clearKeyBuffer();

		key = gfx_getKey();
		key = 0;
		hasquit = true;
		break;
	    }

	    case 'x':
	    {
		int		x, y;
		bool		selected;
		
		x = avatar->getX();
		y = avatar->getY();

		selected = selectsquare(x, y, glb_output, glb_status);
		
		if (selected)
		{
		    // If there is a mob there print its info.
		    MOB		*m;

		    m = avatar->getMap()->getMob(x, y);

		    if (m && !avatar->hasItem(ITEM_BLIND))
		    {
			m->viewDescription();
		    }
		}
		msg_newturn();
		
		// Screen restored for us...
		break;
	    }

	    case '?':
		msg_report("Help:\n");
		msg_report(text_lookup("game", "help"));
		msg_newturn();
		break;

	    case 'a':
		msg_report("About:\n");
		msg_report(text_lookup("game", "about"));
		msg_newturn();
		break;

	    case 't':
		hiscore_display();
		msg_newturn();
		break;
	}

	// IFF didmove is set do we update all the NPCs.
	if (didmove)
	{
	    avatar->getMap()->doMoveNPC();
	    spd_inctime();
	    newturn = true;

	    if (!forcemove)
	    {
		// We want to new turn before statelist appending
		// so we will go back in time properly.
		msg_newturn();

		// Update the probability of having died.  We reset
		// our HP DPDF to account for the fact it will now
		// represent the DPDF given that we didn't die on this
		// turn.
		// Roll in the current chance...
		avatar_multprob(MOB::getAvatarHP_DPDF().probabilityGreaterThan(0));
		// Scale back our DPDF as we know we are alive now!
		MOB::getAvatarHP_DPDF().applyGivenGreaterThan(0);

		// Store the state after everything is done prior
		// to next move.  This means we don't have to worry about
		// ensuing non-relevant commands.
		statelist.append(WORLDSTATE::pushState(mainmap));
	    }
	}
	else
	    newturn = false;
    }

    text_shutdown();
    gfx_shutdown();

    return 0;
}
