/*cc rs.c  -o rs $(pkg-config sqlite3 hildon-1 gnome-vfs-2.0 glib-2.0 libhildonmime dbus-1 libosso --cflags --libs) -lrs */

/*
    This file belongs to rs (Remote security) package for Nokia N900
    Copyright (C) 2009  Pavan Patale <pavan.patale@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libosso.h>
#include <osso-log.h>
#include <sqlite3.h>
#include <gconf/gconf-client.h>
#include <hildon-mime.h>
#include <hildon/hildon.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "rs-utils.h"
#include "rs-ui.h"
#include "rs-registry.h"


#define MEDIAFILE "/home/user/MyDocs/.sounds/test.mp3"
#define MONITORFILE "/home/user/.rtcom-eventlogger/el.db"
#define DBCOMMAND "select free_text from Events order by rowid desc limit 1"



static osso_context_t *osso_context = NULL;
static DBusConnection *con = NULL;
static sqlite3 *db = NULL;

/* Create needed variables */
static HildonProgram *program = NULL;
static HildonWindow *window = NULL;
static gboolean registered = FALSE;
static GnomeVFSMonitorHandle *handler = NULL;


static int access_db();
static void register_for_ext_cmds();
static int db_callback(void *NotUsed, gint argc, gchar **argv, gchar **field);
static void execute_command(int command);
static void watch_offline_mode();
static DBusHandlerResult handle_mode_cb (DBusConnection *conn,
		                         DBusMessage    *message,
				         void           *data);
static gboolean sim_changed();
static void change_detected();
static void start_acting();



static void   monitor_cb(GnomeVFSMonitorHandle *handle,
                  const gchar *monitor_uri,
                  const gchar *info_uri,
                  GnomeVFSMonitorEventType event_type,
                  gpointer user_data)
{
    if(event_type == GNOME_VFS_MONITOR_EVENT_CHANGED) {
	//TODO: get return value and take action
	if(!access_db()) {
	    /*-----*/
	}
    }
}


static void execute_command(int command)
{
    gint result = 0;
    switch(command)
    {
        case PLAY	:    result = hildon_mime_open_file (con, MEDIAFILE);
			     if (result != 1) {
                                 g_print ("launch failed.\n");
                             }
			     break;
			     //TODO: Search for a cleaner way 
        case LOCK       :    change_detected(); 
                             break;
        case UNLOCK     :    rs_registry_unlock_device();
                             break;
        default		:    ;//g_print("Loooks like you got wrong command\n");
			     
    }
    	
}


static int db_callback(void *NotUsed, gint argc, gchar **argv, gchar **field)
{
    int cmd = 0;
    gchar *msg = NULL;
    if(argv[0]) 
    {
        msg = g_strdup(argv[0]);	
	if(!strcmp("Play1234", msg))  //Only for testing purpose. Remove later.
		cmd = PLAY;
	else
	    cmd = rs_registry_compare_cmd(msg);
	execute_command(cmd);
    }
    if(msg)
        g_free(msg);
    return 0;
}

static int access_db()
{
  gchar *error = NULL;
  int ret = 0;

  ret = sqlite3_open(MONITORFILE, &db);
  if( ret ){
    //g_print("Can't open database: %s\n", sqlite3_errmsg(db));
    sqlite3_close(db);
    return -1;
  }
  ret = sqlite3_exec(db, DBCOMMAND , db_callback, 0, &error);
  if( ret!=SQLITE_OK ){
    //g_print("SQL error: %s\n", error);
    sqlite3_free(error);
    sqlite3_close(db);
    return -1;
  }
  sqlite3_close(db);
  return 1;
}


static DBusHandlerResult
handle_mode_cb (DBusConnection *conn,
                DBusMessage    *message,
                void           *data)
{
    gchar *mode = NULL;
    DBusMessageIter args;


    /* Check received message */
    if (!dbus_message_is_signal (message, "com.nokia.mce.signal", "sig_device_mode_ind")) {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    if (dbus_message_iter_init (message, &args) == FALSE) {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (dbus_message_iter_get_arg_type (&args) != DBUS_TYPE_STRING) {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    /* Get message arguments */
    dbus_message_iter_get_basic (&args, &mode);

    /* Check whether offline mode is enabled or disabled and take appropriate action. */
    if (g_ascii_strcasecmp (mode, "flight") == 0) {
        //g_print ("Offline mode. Pls dont goto offline until the user authenticates\n");
	change_detected();
    } else {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    return DBUS_HANDLER_RESULT_HANDLED;
}




static void watch_offline_mode()
{
    DBusGConnection *gcon = NULL;	
    DBusConnection  *conn = NULL;
    DBusError error;	

    gcon = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (gcon == NULL) {
        //g_print ("Error getting bus");
        return;
     }

    conn = dbus_g_connection_get_connection (gcon);

    dbus_error_init (&error);

    dbus_bus_add_match (conn,
		        "type='signal',"
			"interface='com.nokia.mce.signal',"
			"member='sig_device_mode_ind'",
			&error);
    if (dbus_error_is_set (&error)) {
        //g_print("Unable to match offline mode signal %s\n", error.message);	    
        dbus_error_free (&error);
        return;
    }

    if (!dbus_connection_add_filter (conn, handle_mode_cb, NULL, NULL)) {
        //g_print("Unable to watch offline mode signal\n");	    
    }

}	

static void change_detected()
{
    gchar *pw = NULL;
    const gchar *current = NULL;
    if(!rs_registry_remote_enabled())
	    return;
    pw = rs_ui_create_get_pw_dialog(GTK_WIDGET(window), "Enter remote unlock code");
    if(pw && rs_registry_compare_cmd(pw) == UNLOCK)
    {
        current = rs_utils_phone_get_current_imsi();
        if(current)
	{
            rs_registry_set_current_imsi(current);
        }
    }
    else
    {
	rs_registry_set_rs_choice(ACTIVE);
	rs_registry_lock_device();
        change_detected();
    }
}

static gboolean sim_changed()
{
    gchar *last = NULL;
    const gchar *current = NULL;
    gboolean change = FALSE;

    last = rs_registry_get_last_imsi();
    current = rs_utils_phone_get_current_imsi();
    if(last && current) {
        if(!strcmp(last, current)) {
          //  g_print("Sims are same\n");
        } else {
	  //  g_print("Sims are not same take action\n");
	    change = TRUE;
        }
    } else {
        // g_print("Take action\n");
	change = TRUE;
    }
    if(last) {
        g_free(last);	    
    }
    return change;
}

static void register_for_ext_cmds()
{
    
    if(!registered)
    {
        gnome_vfs_monitor_add (&handler,
                            MONITORFILE,
                            GNOME_VFS_MONITOR_FILE,
                            monitor_cb,
                            NULL);
	registered = TRUE;
    }

}

static void deregister_for_ext_cmds()
{
    if(registered) 
    {
        gnome_vfs_monitor_cancel(handler);
        registered = FALSE;
	handler = NULL;
    }
}

static void remote_watch_cb(GConfClient *client,
		       guint cnxn_id,
		       GConfEntry *entry,
		       gpointer user_data)
{
    start_acting();	
}	

static void start_acting()
{
    if(!rs_registry_remote_enabled() || !(rs_registry_get_rs_choice()==ACTIVE))
    {
        deregister_for_ext_cmds();
	return;
    }
    register_for_ext_cmds();
    watch_offline_mode(); 
    if(sim_changed()){
        change_detected();
    }
}

int main(int argc, char *argv[])
{

    gnome_vfs_init();

    /* Initialize the GTK+ and hildon libraries. */
    hildon_gtk_init(&argc, &argv);

    /* Create the hildon program and setup the title */
    if(!program)
    {
        program = HILDON_PROGRAM(hildon_program_get_instance());
        g_set_application_name("Extra Secure");
    }

    /* Create HildonWindow and set it to HildonProgram */
    window = HILDON_WINDOW(hildon_window_new());
    hildon_program_add_window(program, window);

    osso_context = osso_initialize("com.nokia.rs",
                                    "0.1",
                                    FALSE, NULL);
    if(!osso_context) {
        osso_log(LOG_CRIT, "Unable to get the osso_context");
	return -1;
    }

    con = osso_get_dbus_connection(osso_context);
    if(!con) {
        osso_log(LOG_CRIT, "Unable to get dbus connection\n");
	return -1;
    }

    if(!rs_registry_init()){
        osso_log(LOG_CRIT, "Unable to get registry details\n");
	return -1;
    }

    rs_registry_watch_remote_lock(remote_watch_cb);
    if(rs_registry_remote_enabled())
	start_acting();
    gtk_main();

    return 0;
}


//TODO: Is this really required ?
#if 0
static void clean_up()
{
    g_object_unref(gclient);
    dbus_connection_unref (con);
//    gnome_vfs_monitor_cancel(handler);
//  g_main_loop_quit(loop);
}
#endif

