/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "dispatcher.h"
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libringtoned/ringtoned.h>
#include <libringtoned/utils.h>
#include <string.h>
#include "anerley-tp-observer.h"
#include "call.h"
#include "plugin.h"


/* Method calls to monitor to know when the call is muted */
#define HILDON_SV_NOTIFICATION_DAEMON_IFACE \
    "com.nokia.HildonSVNotificationDaemon"
#define HILDON_SV_NOTIFICATION_DAEMON_PATH \
    "/com/nokia/HildonSVNotificationDaemon"
#define STOP_EVENT_METHOD "StopEvent"
#define STOP_EVENT_MATCH_RULE \
    "type='method_call'," \
    "interface='" HILDON_SV_NOTIFICATION_DAEMON_IFACE "'," \
    "path='" HILDON_SV_NOTIFICATION_DAEMON_PATH "'," \
    "member='" STOP_EVENT_METHOD "'"

typedef void (*RingtonedPluginInitFunc) (RingtonedDispatcher *);

#define RINGTONED_DISPATCHER_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RINGTONED_TYPE_DISPATCHER, \
                                  RingtonedDispatcherPrivate))

enum
{
    PROP_0,
};

typedef struct
{
    gint priority;
    RingtonedFilterCb cb;
    gpointer user_data;
    GDestroyNotify destroy_notify_cb;
    gchar *debug_info;
} Filter;

struct _RingtonedDispatcherPrivate
{
    AnerleyTpObserver *observer;
    DBusConnection *connection;

    gboolean is_loading_plugins;
    GList *plugins; /* GList of GModule pointers */
    GList *filters; /* GList of Filter pointers */

    GList *calls;
};

G_DEFINE_TYPE (RingtonedDispatcher, ringtoned_dispatcher, G_TYPE_OBJECT)

RingtonedDispatcher *
ringtoned_dispatcher_new (void)
{
    return g_object_new (RINGTONED_TYPE_DISPATCHER, NULL);
}

static void
mute_all (RingtonedDispatcher *self)
{
    GList *l;
    RingtonedCall *call;
    RingtonedCallStatus status;

    /* There is no way for us to know which event is being stopped, so we
     * just mute all the calls. Hopefully you should not have multiple
     * ringing calls, alarms, etc. all at the same time */

    for (l = self->priv->calls; l; l = l->next)
    {
        call = l->data;
        g_object_get (call, "status", &status, NULL);
        if (status == RINGTONED_CALL_STATUS_RINGING)
            g_object_set (call, "mute", TRUE, NULL);
    }
}

static DBusHandlerResult
message_filter (DBusConnection *connection,
                DBusMessage    *message,
                gpointer        user_data)
{
    RingtonedDispatcher *self = user_data;
    gint type;
    const gchar *interface;
    const gchar *member;

    type = dbus_message_get_type (message);
    interface = dbus_message_get_interface (message);
    member = dbus_message_get_member (message);

    if (type == DBUS_MESSAGE_TYPE_METHOD_CALL &&
        g_strcmp0 (interface, HILDON_SV_NOTIFICATION_DAEMON_IFACE) == 0 &&
        g_strcmp0 (member, STOP_EVENT_METHOD) == 0)
    {
        mute_all (self);
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
ringtoned_dispatcher_dispatch (RingtonedDispatcher *self,
                               RingtonedCall       *call)
{
    GList *l;
    Filter *filter; 

    DEBUG ("Dispatching call %p", call);

    g_object_set (call,
            "ringtone-path", ringtoned_profile_get_default_ringtone (),
            NULL);

    for (l = self->priv->filters; l; l = l->next)
    {
        filter = l->data;

        DEBUG ("Running filter %s", filter->debug_info);

        if (filter->cb (self, call, filter->user_data))
        {
            DEBUG ("Filter %s stopped processing of call",
                    filter->debug_info);
            break;
        }
    }

    DEBUG ("Dispatching of call %p finished", call);

    ringtoned_call_play_ringtone (call);
}

static void
call_destroyed_cb (gpointer  user_data,
                   GObject  *objp)
{
    RingtonedDispatcher *self = user_data;

    DEBUG ("Removing call %p from the known calls", objp);
    self->priv->calls = g_list_remove (self->priv->calls, objp);
}

static void
observer_new_channel_cb (AnerleyTpObserver *observer,
                         const gchar       *account_path,
                         TpChannel         *channel,
                         gpointer           user_data)
{
    RingtonedDispatcher *self = user_data;
    const TpIntSet *local_pending;
    const gchar *account_name;
    const gchar *identifier;
    McAccount *account;
    const gchar *profile_name;
    McProfile *profile;
    const gchar *vcard_field;
    RingtonedCall *call;

    /* The observer calls this function when the channel is ready. */
    local_pending = tp_channel_group_get_local_pending (channel);
    if (!tp_intset_is_member (local_pending,
                tp_channel_group_get_self_handle (channel)))
    {
        DEBUG ("Skipping outgoing call (channel: %p)", channel);
        return;
    }

    if (strncmp (account_path, MC_ACCOUNT_DBUS_OBJECT_BASE,
                MC_ACCOUNT_DBUS_OBJECT_BASE_LEN) == 0)
    {
        account_name = &account_path[MC_ACCOUNT_DBUS_OBJECT_BASE_LEN];
    }
    else
    {
        /* FIXME: be sure to play the default ringtone even if
         * something fails */
        return;
    }

    /* FIXME: handle any of the values being NULL */
    identifier = tp_channel_get_identifier (channel);
    account = osso_abook_account_manager_lookup_by_name (NULL, account_name);
    profile_name =  mc_account_compat_get_profile (account);
    profile = mc_profile_lookup (profile_name);
    vcard_field =  mc_profile_get_vcard_field (profile);

    call = ringtoned_call_new (channel, vcard_field, identifier, account);

    self->priv->calls = g_list_prepend (self->priv->calls, call);
    g_object_weak_ref (G_OBJECT (call), call_destroyed_cb, self);

    if (ringtoned_debug_is_enabled ())
    {
        GList *contacts;

        DEBUG ("New channel (call: %p, channel: %p)\n"
                "    Account name: %s\n"
                "    Profile name: %s\n"
                "    VCard field: %s\n"
                "    Identifier: %s",
                call, channel, account_name, profile_name,
                vcard_field, identifier);

        g_object_get (call, "contacts", &contacts, NULL);

        if (contacts)
        {
            GList *l;

            DEBUG ("    Found %d matching contact(s) for %s (field: %s)",
                    g_list_length (contacts), identifier, vcard_field);

            for (l = contacts; l; l = l->next)
            {
                DEBUG ("        %s (id: %s)",
                        osso_abook_contact_get_display_name (l->data),
                        osso_abook_contact_get_uid (l->data));
            }
        }
        else
            DEBUG ("No contacts found for %s (field: %s)",
                    !IS_EMPTY (identifier) ? identifier : "<unknown contact>",
                    vcard_field);
    }

    ringtoned_dispatcher_dispatch (self, call);

    /* The call keeps itself alive until it's ended. Plugins could make the
     * life span longer by keeping a reference */
    g_object_unref (call);

    g_object_unref (profile);
}

void
ringtoned_dispatcher_add_filter_real (RingtonedDispatcher *self,
                                      gint                 priority,
                                      const gchar         *plugin_name,
                                      const gchar         *func_name,
                                      RingtonedFilterCb    cb,
                                      gpointer             user_data,
                                      GDestroyNotify       destroy_notify_cb)
{
    Filter *filter;

    g_return_if_fail (RINGTONED_IS_DISPATCHER (self));
    g_return_if_fail (cb);
    g_return_if_fail (self->priv->is_loading_plugins);

    DEBUG ("Adding filter %s:%s (priority %d, dispatcher %p)",
            plugin_name, func_name, priority, self);

    filter = g_new0 (Filter, 1);
    filter->priority = priority;
    filter->cb = cb;
    filter->user_data = user_data;
    filter->destroy_notify_cb = destroy_notify_cb;
    filter->debug_info = g_strdup_printf ("%s:%s", plugin_name, func_name);

    self->priv->filters = g_list_prepend (self->priv->filters, filter);
}

static gint
filters_cmp (gconstpointer p1,
             gconstpointer p2)
{
    const Filter *f1 = p1;
    const Filter *f2 = p2;

    return f1->priority - f2->priority;
}

static void
load_plugins (RingtonedDispatcher *self)
{
    GDir *dir;
    const gchar *filename;
    gchar *path;
    GModule *module;
    RingtonedPluginInitFunc plugin_init;

    dir = g_dir_open (PLUGINDIR, 0, NULL);
    if (!dir)
        return;

    self->priv->is_loading_plugins = TRUE;

    while ((filename = g_dir_read_name (dir)))
    {
        if (!g_str_has_suffix (filename, "." G_MODULE_SUFFIX))
            continue;

        path = g_build_filename (PLUGINDIR, filename, NULL);
        module = g_module_open (path, G_MODULE_BIND_LAZY);
        g_free (path);

        if (!module)
        {
            g_warning ("Failed to load plugin: %s", filename);
            continue;
        }

        if (!g_module_symbol (module, "ringtoned_plugin_init",
                    (gpointer) &plugin_init) ||
            plugin_init == NULL)
        {
            g_warning ("Plugin %s is not a valid plugin", filename);
            g_module_close (module);
            continue;
        }

        plugin_init (self);

        self->priv->plugins = g_list_prepend (self->priv->plugins, module);
    }

    self->priv->is_loading_plugins = FALSE;
    self->priv->filters = g_list_sort (self->priv->filters, filters_cmp);

    g_dir_close (dir);
}

static void
ringtoned_dispatcher_get_property (GObject    *object,
                                   guint       param_id,
                                   GValue     *value,
                                   GParamSpec *pspec)
{
    switch (param_id)
    {
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_dispatcher_set_property (GObject      *object,
                                   guint         param_id,
                                   const GValue *value,
                                   GParamSpec   *pspec)
{
    switch (param_id)
    {
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_dispatcher_init (RingtonedDispatcher *self)
{
    DEBUG ("Dispatcher running");

    self->priv = RINGTONED_DISPATCHER_GET_PRIVATE (self);

    self->priv->observer = anerley_tp_observer_new ("Ringtoned");
    g_signal_connect (self->priv->observer, "new-channel",
            G_CALLBACK (observer_new_channel_cb), self);

}

static void
ringtoned_dispatcher_constructed (GObject *object)
{
    RingtonedDispatcher *self = RINGTONED_DISPATCHER (object);
    DBusGConnection *gconnection;

    load_plugins (self);

    gconnection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
    self->priv->connection = dbus_g_connection_get_connection (gconnection);

    dbus_connection_add_filter (self->priv->connection, message_filter,
            self, NULL);
    dbus_bus_add_match (self->priv->connection, STOP_EVENT_MATCH_RULE, NULL);

    if (G_OBJECT_CLASS (ringtoned_dispatcher_parent_class)->constructed)
        G_OBJECT_CLASS (ringtoned_dispatcher_parent_class)->constructed (object);
}

static void
ringtoned_dispatcher_finalize (GObject *object)
{
    RingtonedDispatcher *self = RINGTONED_DISPATCHER (object);

    g_object_unref (self->priv->observer);

    while (self->priv->calls)
    {
        g_object_weak_unref (self->priv->calls->data, call_destroyed_cb, self);
        self->priv->calls = g_list_delete_link (self->priv->calls,
                self->priv->calls);
    }

    g_list_foreach (self->priv->plugins, (GFunc) g_module_close, NULL);
    g_list_free (self->priv->calls);

    while (self->priv->filters)
    {
        Filter *f = self->priv->filters->data;

        if (f->destroy_notify_cb)
            f->destroy_notify_cb (f->user_data);
        g_free (f->debug_info);
        g_free (f);

        self->priv->filters = g_list_delete_link (self->priv->filters,
                self->priv->filters);
    }

    dbus_connection_remove_filter (self->priv->connection,
            message_filter, self);

    G_OBJECT_CLASS (ringtoned_dispatcher_parent_class)->finalize (object);
}

static void
ringtoned_dispatcher_class_init (RingtonedDispatcherClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    object_class->constructed = ringtoned_dispatcher_constructed;
    object_class->finalize = ringtoned_dispatcher_finalize;
    object_class->get_property = ringtoned_dispatcher_get_property;
    object_class->set_property = ringtoned_dispatcher_set_property;

    g_type_class_add_private (object_class, sizeof (RingtonedDispatcherPrivate));
}
