/*************************************************************************}
{ qkineticwidget.cpp - QWidget with kinetic scrolling                     }
{                                                                         }
{ (c) Alexey Parfenov, 2011                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This library is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This library is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 31 Mar 2012                                              }
{*************************************************************************/

#include "qkineticwidget.h"

QKineticWidget::QKineticWidget(QWidget *parent) :
    QWidget(parent)
{
    setMouseTracking(true);

    ix = iy = 0;
    x = y = 0.0;
    mx = my = 0;
    spdX = spdY = 0.0;
    frictionX = frictionY = 1.05;
    backFrictionX = backFrictionY = 1.75;
    backSpeedX = backSpeedY = 20.0;
    thresholdX = thresholdY = 5.0;
    innerWidth = innerHeight = 0;
    offX = offY = 0;
    minX = minY = 0;
    moveByX = moveByY = true;
    isClick = false;
    isDown = false;
    timer = 0;
    stopSteps = 10;
    stopStepsLeft = 0;
    activeWidget = NULL;
    cssNormal = "";
    cssActive = "";
    setTimerInterval(20);
}

void QKineticWidget::setTimerInterval(int msecs)
{
    if(timer)
        killTimer(timer);
    timer = startTimer(msecs);
    timerInterval = msecs;
}

void QKineticWidget::setInnerWidth(int value)
{
    innerWidth = value;
    updateConstraintsX();
}

void QKineticWidget::setInnerHeight(int value)
{
    innerHeight = value;
    updateConstraintsY();
}

void QKineticWidget::mousePressEvent(QMouseEvent *event)
{
    isDown = true;
    isClick = true;
    mx = event->x();
    my = event->y();
    spdX = 0;
    spdY = 0;
    offX = 0;
    offY = 0;

    QWidget* item = getChildAt(mx, my);
    if(item)
        emit onItemDown(item);
    if(activeWidget)
        activeWidget->setStyleSheet(cssNormal);
    activeWidget = item;
    if(activeWidget)
        activeWidget->setStyleSheet(cssActive);
}

void QKineticWidget::mouseMoveEvent(QMouseEvent *event)
{
    if(isDown)
    {
        int omx = mx;
        int omy = my;
        mx = event->x();
        my = event->y();
        spdX = mx - omx;
        spdY = my - omy;
        scrollX(spdX);
        scrollY(spdY);
        offX += spdX;
        offY += spdY;
        if(
                (moveByX && (abs(offX) > thresholdX))
                    ||
                (moveByY && (abs(offY) > thresholdY))
        )
        {
            isClick = false;
        }
        stopStepsLeft = stopSteps;
    }
}

void QKineticWidget::mouseReleaseEvent(QMouseEvent *event)
{
    Q_UNUSED(event);
    isDown = false;
    if(activeWidget)
    {
        emit onItemUp(activeWidget, isClick);
        if(activeWidget)
            activeWidget->setStyleSheet(cssNormal);
        if(isClick)
            emit onItemClick(activeWidget);
        activeWidget = NULL;
    }
}

QWidget* QKineticWidget::getChildAt(int xPos, int yPos)
{
    QWidget* w;
    foreach(QObject* child, children())
    {
        w = qobject_cast<QWidget*>(child);
        if(w)
        {
            if(w->geometry().contains(xPos, yPos))
                return w;
        }
    }
    return NULL;
}

void QKineticWidget::updateConstraintsX()
{
    minX = qMin(size().width() - innerWidth, 0);
}

void QKineticWidget::updateConstraintsY()
{
    minY = qMin(size().height() - innerHeight, 0);
}

void QKineticWidget::resizeEvent(QResizeEvent *event)
{
    Q_UNUSED(event);
    updateConstraintsX();
    updateConstraintsY();
}

void QKineticWidget::contextMenuEvent(QContextMenuEvent *event)
{
    if(activeWidget)
    {
        emit onContextMenu(activeWidget, event);
        activeWidget->setStyleSheet(cssNormal);
        activeWidget = NULL;
    }
    isDown = false;
}

void QKineticWidget::childEvent(QChildEvent *event)
{
    if(event->type() == QEvent::ChildAdded)
    {
        if(event->child()->isWidgetType())
            ((QWidget*)event->child())->setStyleSheet(cssNormal);
    }
}

void QKineticWidget::timerEvent(QTimerEvent *event)
{
    Q_UNUSED(event);
    if(!isDown)
    {
        if(moveByX)
        {
            if((x < minX) || (x > 0))
            {
                spdX = spdX / backFrictionX;
                if(abs(spdX) < 1)
                {
                    spdX = 0;
                    if(x > 0)
                    {
                        scrollX(-backSpeedX);
                        if(x < 0)
                            setX(0);
                    }
                    else
                    {
                        scrollX(backSpeedX);
                        if(x > minX)
                            setX(minX);
                    }
                }
                else
                {
                    scrollX(spdX);
                }
            }
            else
            {
                spdX = spdX / frictionX;
                scrollX(spdX);
            }
        }

        if(moveByY)
        {
            if((y < minY) || (y > 0))
            {
                spdY = spdY / backFrictionY;
                if(abs(spdY) < 1)
                {
                    spdY = 0;
                    if(y > 0)
                    {
                        scrollY(-backSpeedY);
                        if(y < 0)
                            setY(0);
                    }
                    else
                    {
                        scrollY(backSpeedY);
                        if(y > minY)
                            setY(minY);
                    }
                }
                else
                {
                    scrollY(spdY);
                }
            }
            else
            {
                spdY = spdY / frictionY;
                scrollY(spdY);
            }
        }
    }
    else
    {
        if(stopStepsLeft)
        {
            stopStepsLeft--;
            if(!stopStepsLeft)
            {
                spdY = 0;
                spdX = 0;
            }
        }
    }
}

void QKineticWidget::setX(float value)
{
    if(moveByX)
    {
        int dx = value - ix;
        scroll(dx, 0);
        ix += dx;
        x = value;
    }
}

void QKineticWidget::setY(float value)
{
    if(moveByY)
    {
        int dy = value - iy;
        scroll(0, dy);
        iy += dy;
        y = value;
    }
}

void QKineticWidget::scrollX(float offset)
{
    setX(x + offset);
}

void QKineticWidget::scrollY(float offset)
{
    setY(y + offset);
}

QKineticWidget::~QKineticWidget()
{
    if(timer)
        killTimer(timer);
}

void QKineticWidget::deleteAllChildren()
{
    foreach(QObject* child, children())
        delete child;
}

void QKineticWidget::repositionChildrenV(int itemsHeight, int rows)
{
    int ww = width()/rows;
    int c = 0;
    int rowHeight = 0;
    int y = 0;
    int x = 0;
    int ih;

    QWidget* wgt;
    foreach(QObject* child, children())
    {
        wgt = qobject_cast<QWidget*>(child);
        if(!wgt)
            continue;
        //if(!wgt->isVisible())
        //    continue;

        if(!c)
        {
            y += rowHeight;
            x = 0;
            c = rows;
            rowHeight = 0;
        }
        else
        {
            x += ww;
        }

        c--;
        if(itemsHeight >= 0)
            ih = itemsHeight;
        else
            ih = wgt->height();
        wgt->resize(ww, ih);
        wgt->move(x, y);
        rowHeight = qMax(rowHeight, ih);
    }

    setInnerHeight(y + rowHeight);
}

void QKineticWidget::centerWidgetV(const QWidget *widget)
{
    int y = getY()-widget->y()+(height()-widget->height())/2;
    if(y > 0)
    {
        y = 0;
    }
    else
    {
        if(y < minY)
            y = minY;
    }
    setY(y);
}

void QKineticWidget::setCssNormal(const QString &css)
{
    cssNormal = css;
    if(isDown && activeWidget)
    {
        foreach(QObject* child, children())
            if(child != activeWidget)
                ((QWidget*)child)->setStyleSheet(cssNormal);
    }
    else
    {
        foreach(QObject* child, children())
            ((QWidget*)child)->setStyleSheet(cssNormal);
    }
}

void QKineticWidget::setCssActive(const QString &css)
{
    cssActive = css;
    if(isDown && activeWidget)
    {
        foreach(QObject* child, children())
            if(child == activeWidget)
            {
                ((QWidget*)child)->setStyleSheet(cssActive);
                break;
            }
    }
}
