/*************************************************************************}
{ phononplayer.cpp - Phonon-based audio player                            }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 21 Mar 2012                                              }
{*************************************************************************/

#include "phononplayer.h"
#include "mainwindow.h"

PhononPlayer::PhononPlayer(QObject *parent) :
    QObject(parent)
{
    knownPlaylistExtensions << "m3u" << "m3u8" << "pls";
    updateKnownMime();
    updateSupportedMime();
    findUnzip();
    dev = NULL;
    state = pp_state_Idle;
    sourceType = pp_stype_None;

    equalizer = new Equalizer(this);
    equalizerEnabled = false;
    lastPos = -1;

    createMediaObject();

    tmpFilenameBase = QDesktopServices::storageLocation(QDesktopServices::TempLocation)+"/___"+APP_NAME+"_temp.";

#ifdef MAEMO5
    QDBusConnection::systemBus().connect(
                    "org.freedesktop.Hal",
                    "/org/freedesktop/Hal/devices/platform_headphone",
                    "org.freedesktop.Hal.Device",
                    "Condition",
                    this,
                    SLOT(onHeadsetCondition())
                    );

    QDBusConnection::systemBus().connect(
                    "org.freedesktop.Hal",
                    "/org/freedesktop/Hal/Manager",
                    "org.freedesktop.Hal.Manager",
                    "DeviceAdded",
                    this,
                    SLOT(onDeviceAdd())
                    );
    QDBusConnection::systemBus().connect(
                    MCE_SERVICE,
                    MCE_SIGNAL_PATH,
                    MCE_SIGNAL_IF,
                    MCE_CALL_STATE_SIG,
                    this,
                    SLOT(onCallStateChange(QString))
                    );

    QDBusMessage message = QDBusMessage::createMethodCall(
                MCE_SERVICE,
                MCE_REQUEST_PATH,
                MCE_REQUEST_IF,
                MCE_CALL_STATE_GET);
    message = QDBusConnection::systemBus().call(message);
    QString callState = message.arguments().at(0).toString();
    calling = (callState != MCE_CALL_STATE_NONE);
#else
    calling = false;
#endif
}

PhononPlayer::~PhononPlayer()
{
    close();
    delete equalizer;
    delete mediaObject;
    delete audioOutput;
}

bool PhononPlayer::play(const QString &filename)
{
    if(filename.indexOf("://") >= 0)
    {
        if(!filename.startsWith("http://"))
            return false;
        // TODO: network stream
        //sourceType = pp_stype_None;
        return false;
    }
    else
    {
        close();
        mediaObject->setCurrentSource(Phonon::MediaSource(filename));

        currentFilename = filename;

        sourceType = pp_stype_File;
        mediaObject->play();
    }
    return true;
}

void PhononPlayer::pause()
{
    mediaObject->pause();
}

void PhononPlayer::unpause()
{
    mediaObject->play();
}

void PhononPlayer::stop()
{
    close();
}

bool PhononPlayer::isHeadsetConnected()
{
#ifdef MAEMO5
    QFile statusFile("/sys/devices/platform/gpio-switch/headphone/state");
    statusFile.open(QIODevice::ReadOnly);
    QString statusText = QString::fromAscii(statusFile.readAll().constData()).trimmed().toLower();
    statusFile.close();
    return statusText == "connected";
#else
    return false; // stub
#endif
}

bool PhononPlayer::setEqualizerEnabled(bool doEnable)
{
    if(!equalizer->getEffect())
        return false;
    if(doEnable == equalizerEnabled)
        return true;
    // TODO:
    // when applying/deleting effect, the mediaSource is reset,
    // so there is need to restore its previous position
    /*
    if(state != pp_state_Idle)
        lastPos = mediaObject->currentTime();
    else
        lastPos = -1;
    */
    mediaPath.removeEffect(equalizer->getEffect()); // always remove the effect, just in case
    if(doEnable)
        mediaPath.insertEffect(equalizer->getEffect());
    equalizerEnabled = doEnable;
    return true;
}

void PhononPlayer::close(bool emitStop)
{
    state = pp_state_Idle;
    mediaObject->stop();
    mediaObject->clearQueue();
    if(dev)
    {
        dev->close();
        dev = NULL;
    }
    sourceType = pp_stype_None;

    // the whole thing below is
    // a workaround for a one strange gstreamer bug, which
    // is not allowing to play *.ogg after playing *.flac
    if(emitStop)
        stateChanged(Phonon::StoppedState, mediaObject->state()); // in case the stop event had not been fired yet
    // because we're calling the current function from a mediaObject's slot;
    // delete audioOutput as well, because, it will not be possible to make a path otherwise;
    killLater(mediaObject); // also, sometimes, there are errors popping out of nowhere, when using deleteLater,
    killLater(audioOutput); // so we delete objects after 10 seconds (in killLater function)
    createMediaObject(); // create mediaObject anew
}

void PhononPlayer::updateSupportedMime()
{
    supportedMime = Phonon::BackendCapabilities::availableMimeTypes();
    supportedExtensions.clear();

    int n = supportedMime.size();
    for(int a=0; a<n ;a++)
        supportedMime[a] = supportedMime[a].trimmed().toLower();

    QHashIterator<QString, QStringList> i(mimeExtensions);
    while(i.hasNext())
    {
         i.next();
         const QString& value = i.key();
         foreach(QString mimeType, supportedMime)
         {
             if(mimeType == value)
             {
                 supportedExtensions.append(i.value());
                 break;
             }
         }
    }

    supportedExtensions.removeDuplicates();
    supportedExtensions.sort(); // alphabetically
}

void PhononPlayer::updateKnownMime()
{
    QFileEx mimes;
    QString filename = APP_DIR + "data/mimes";

    mimes.setFileName(filename);
    if(!mimes.open(QFile::ReadOnly))
        return;
    QStringList lines = mimes.readLinesUTF8();
    mimes.close();

    mimeExtensions.clear();
    knownExtensions.clear();
    QString mimeType;
    QString ext;
    QStringList list;
    int a, n;
    foreach(QString line, lines)
    {
        line = line.trimmed();
        if(line.isEmpty()) // empty line or a line with whitespace only
            continue;
        if(line.startsWith("#")) // comment line
            continue;
        line = line.toLower();
        list = line.split(",", QString::SkipEmptyParts);
        if(list.size() < 2) // no extensions specified or a string with commas only
            continue;
        mimeType = list.takeFirst().trimmed();
        mimeType.replace("/x-", "/"); // convert eXperimental MIMEs to normal ones
        n = list.size();
        for(a=0; a<n; a++)
        {
            ext = list[a].trimmed();
            knownExtensions.append(ext);
            list[a] = ext;
        }
        mimeExtensions[mimeType] = list;
        mimeType.replace("/", "/x-"); // add eXperimental version of a current MIME
        mimeExtensions[mimeType] = list;
    }

    knownExtensions.removeDuplicates();
    knownExtensions.sort(); // alphabetically
}

void PhononPlayer::findUnzip()
{
    unzipPath = FSMan::which("unzip");
}

void PhononPlayer::stateChanged(Phonon::State newstate, Phonon::State oldstate)
{
    switch(newstate)
    {
        case Phonon::ErrorState:
            qDebug() << currentFilename << mediaObject->errorString();
            if(mediaObject->errorType() == Phonon::FatalError)
            {
                if(sourceType == pp_stype_File)
                {
                    if(unzip(currentFilename))
                    {
                        close(false);
                        mediaObject->setCurrentSource(Phonon::MediaSource(tmpFilename));
                        //mediaObject->setCurrentSource(Phonon::MediaSource(dev));
                        //sourceType = pp_stype_Memory;
                        mediaObject->play();
                        return;
                    }
                }
                close();
                onError();
            }
            break;

        case Phonon::StoppedState:
            if(
                (oldstate != Phonon::PlayingState) &&
                (oldstate != Phonon::PausedState)
            )
            {
                break;
            }
            state = pp_state_Idle;
            emit onStateChanged();
            break;

        case Phonon::PlayingState:
            state = pp_state_Playing;
            emit onStateChanged();
            break;

        case Phonon::PausedState:
            if(
                (oldstate != Phonon::PlayingState) ||
                (state != pp_state_Playing)
            )
            {
                break;
            }
            state = pp_state_Paused;
            emit onStateChanged();
            break;

        default:
            break;
    }
}

void PhononPlayer::onCustomPlayEnd()
{
    state = pp_state_Idle;
    emit onStateChanged();
    emit onPlayEnd();
}

void PhononPlayer::onHeadsetCondition()
{
    if(!isHeadsetConnected())
        emit onHeadsetStatusChange(false);
}

void PhononPlayer::onDeviceAdd()
{
    QTimer::singleShot(1000, this, SLOT(onCheckHeadset())); // safe interval
}

void PhononPlayer::onCheckHeadset()
{
    if(isHeadsetConnected())
        emit onHeadsetStatusChange(true);
}

void PhononPlayer::onCallStateChange(QString callState)
{
#ifdef MAEMO5
    bool newCalling = (callState != MCE_CALL_STATE_NONE);
    if(newCalling == calling)
        return;
    calling = newCalling;
    if(calling)
        emit onCallStart();
    else
        emit onCallEnd();
#else
    Q_UNUSED(callState);
#endif
}

void PhononPlayer::onKillObsoleteObject()
{
    if(!obsoleteObjects.isEmpty())
        obsoleteObjects.takeFirst()->deleteLater();
}

void PhononPlayer::onMetaDataChanged()
{
    QString title;
    QStringList titles = mediaObject->metaData(Phonon::TitleMetaData);
    if(!titles.isEmpty())
        title = titles.at(0);
    if(title.isEmpty())
    {
        QFileInfo info(currentFilename);
        title = info.completeBaseName();
    }
    QString artist;
    QStringList artists = mediaObject->metaData(Phonon::ArtistMetaData);
    if(!artists.isEmpty())
        artist = artists.at(0);
    QString fullTitle;
    if(artist.isEmpty())
        fullTitle = title;
    else
        fullTitle = artist+" - "+title;
    if(lastSentFullTitle == fullTitle)
        return; // to prevent multiple events with a same title
    emit onTitleChange(fullTitle);
    lastSentFullTitle = fullTitle;
}

void PhononPlayer::createMediaObject()
{
    mediaPath.removeEffect(equalizer->getEffect()); // always remove the effect, just in case
    mediaObject = new Phonon::MediaObject(this);
    audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    mediaPath = Phonon::createPath(mediaObject, audioOutput);
    if(equalizerEnabled)
        mediaPath.insertEffect(equalizer->getEffect());
    connect(mediaObject, SIGNAL(finished()), SLOT(onCustomPlayEnd()));
    connect(mediaObject, SIGNAL(stateChanged(Phonon::State,Phonon::State)), SLOT(stateChanged(Phonon::State,Phonon::State)));
    connect(mediaObject, SIGNAL(metaDataChanged()), SLOT(onMetaDataChanged()));
    lastSentFullTitle = "";
}

bool PhononPlayer::unzip(const QString &filename)
{
    if(unzipPath.isEmpty())
        return false;

    QFile f(filename);
    if(!f.open(QIODevice::ReadOnly))
        return false;

    if((f.size() > 1024*1024*10) || (f.size() < 2))
    {
        f.close();
        return false;
    }

    char magic[2];
    magic[0] = magic[1] = 0;
    f.seek(0);
    f.peek(&magic[0], 2);
    f.close();
    if((magic[0]!='P')||(magic[1]!='K')) // zip file header
        return false;

    QByteArray mem;

    /*
    QProcess proc;
    proc.setStandardInputFile(filename);
    proc.start(unzipPath);
    if(!proc.waitForFinished())
        return false;

    mem = proc.readAllStandardOutput();
    */
    FSMan::getRawProcessOutput(mem, unzipPath, (QStringList() << "-p" << filename));

    int n = mem.size();
    if(!n)
        return false; // no output was returned

    // we must create an actual file to circumvent most
    // gstreamer problems regarding "magick" file identification
    QFileInfo info(filename);
    tmpFilename = tmpFilenameBase + info.suffix();
    QFile tmp(tmpFilename);
    if(!tmp.open(QIODevice::WriteOnly))
        return false;
    if(tmp.write(mem) != n)
    {
        tmp.close();
        return false;
    }
    tmp.close();

/*
    close();

    if(!devMem.open(QIODevice::ReadWrite))
        return false;
    if(!devMem.seek(0))
    {
        devMem.close();
        return false;
    }
    if(devMem.write(mem) != n)
    {
        devMem.close();
        return false;
    }
    if(!devMem.seek(0))
    {
        devMem.close();
        return false;
    }

    dev = &devMem;
*/
    return true;
}

void PhononPlayer::killLater(QObject *obsoleteObject)
{
    obsoleteObject->disconnect();
    obsoleteObjects.append(obsoleteObject);
    QTimer::singleShot(10000, this, SLOT(onKillObsoleteObject())); // 10 seconds more to live
}
